﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ut/ut_Inlines.h>
#include <nw/demo/pad/demo_PadProcessor.h>

namespace nw  {
namespace demo {

//---------------------------------------------------------------------------
void
PadProcessor::Update(u32 key)
{
    m_PressPrev = m_Press;
    m_Press     = key;
    m_Trigger   = (m_Press ^ m_PressPrev) & m_Press;
    m_Release   = (m_Press ^ m_PressPrev) & m_PressPrev;

    // 排他されたキーリピートを更新
    m_LastPressedExclusive = this->UpdateLastRepeatKey_( m_LastPressedExclusive );

    if ( m_LastPressedExclusive )
    {
        m_PressExclusive = ( m_Press & m_LastPressedExclusive & m_Setting.exclusiveMask );
    }
    else
    {
        m_PressExclusive = 0;
    }

    this->UpdateRepeate_();
    this->UpdateRepeateAccel_();
    this->UpdateClick_();
}


//---------------------------------------------------------------------------
//! @brief        キーのリピート情報を更新します。
//---------------------------------------------------------------------------
void
PadProcessor::UpdateRepeate_()
{
    const u32 REPEAT_START = m_Setting.repeatStart;
    const u32 REPEAT_SPAN  = m_Setting.repeatSpan;

    m_Repeat = m_Trigger;

    // キーリピートの計算
    for ( uint i = 0; i < MAX_KEY; i++ )
    {
        if ( m_Press & ( 1 << i ) )
        {
            ++m_Counter[ i ];

            if ( (--m_RepeateCount[ i ]) == 0 )
            {
                m_Repeat |= ( 1 << i );
                m_RepeateCount[ i ] = REPEAT_SPAN;
            }
        }
        else
        {
            m_Counter[ i ] = 0;
            m_RepeateCount[ i ] = REPEAT_START;
        }
    }

    // キーの排他リピート情報更新
    m_LastRepeatedExclusive = this->UpdateLastRepeatKey_( m_LastRepeatedExclusive );

    if ( m_LastRepeatedExclusive )
    {
        // 他の排他キーをマスクします
        m_Repeat &= ~(m_Setting.exclusiveMask & ~m_LastRepeatedExclusive);
    }
}


//---------------------------------------------------------------------------
//! @brief        キーの加速付きリピート情報を更新します。
//---------------------------------------------------------------------------
void
PadProcessor::UpdateRepeateAccel_()
{
    const s32 REPEAT_START = m_Setting.repeatStart;
    const s32 REPEAT_SPAN  = m_Setting.repeatSpan;

    m_RepeatAccel  = m_Trigger;

    // キーリピートの計算
    for ( uint i = 0; i < MAX_KEY; i++ )
    {
        if ( m_Press & ( 1 << i ) )
        {
            if ( (--m_RepeateAccelCount[ i ]) == 0 )
            {
                m_RepeatAccel |= ( 1 << i );
                m_RepeateAccelCount[ i ] = nw::ut::Max( REPEAT_SPAN - (m_RepeateTriggerCount[ i ] >> 1), 1 );
            }
        }
        else
        {
            m_RepeateAccelCount[ i ] = REPEAT_START;
        }

        // キーリピートのカウント
        if ( m_RepeatAccel & ( 1 << i ) )
        {
            ++m_RepeateTriggerCount[ i ];
        }

        if ( ! (m_Press & ( 1 << i ))  )
        {
            m_RepeateTriggerCount[ i ] = 0;
        }
    }

    // キーの排他リピート情報更新
    m_LastRepeatedAccelExclusive = this->UpdateLastRepeatKey_( m_LastRepeatedAccelExclusive );

    if ( m_LastRepeatedAccelExclusive )
    {
        // 他の排他キーをマスクします
        m_RepeatAccel &= ~(m_Setting.exclusiveMask & ~m_LastRepeatedAccelExclusive);
    }
}


//---------------------------------------------------------------------------
//! @brief        前回押されていたキーを優先して、排他キーの更新をおこないます。
//!
//! @param[in]    lastPressedExclusive  前回おされていたキーです。
//!
//! @return       更新したキー状態を返します。
//---------------------------------------------------------------------------
u32
PadProcessor::UpdateLastRepeatKey_( u32 lastPressedExclusive )
{
    u32 mask = m_Setting.exclusiveMask;

    // 十字キーの斜め押し対策
    // 最後にREPEATフラグが立ったボタンが離されるまでは他の十字キーの REPEAT フラグは立たせない
    if ( lastPressedExclusive )
    {
        // 最後に押された十字キーが離されていた場合には状態を解除します。
        if ( ! (m_Press & lastPressedExclusive) )
        {
            lastPressedExclusive = 0;
        }
    }

    if ( ! lastPressedExclusive )
    {
        for ( uint i = 0; i < MAX_KEY; ++i )
        {
            u32 i_mask = u32( 1 << i );
            if ( mask & i_mask & m_Press )
            {
                lastPressedExclusive = i_mask;
                break;
            }
        }
    }

    return lastPressedExclusive;
}


//---------------------------------------------------------------------------
//! @brief        キーのクリック情報、長押し情報を更新します。
//---------------------------------------------------------------------------
void
PadProcessor::UpdateClick_()
{
    const  u32  HOLD_COUNT   = m_Setting.holdCount;
    const  u32  CLICK_COUNT  = m_Setting.clickCount;

    m_Click = 0;
    m_LongPress  = 0;

    for ( s32 i = 0; i < MAX_KEY; i++ )
    {
        u32 i_mask = u32( 1 << i );

        if ( m_Press & i_mask )
        {
            // 長押し判定のカウントを過ぎていたら長押しフラグを立てます。
            ++m_ClickHold[ i ];

            if ( m_ClickHold[ i ] == HOLD_COUNT )
            {
                m_LongPress |= i_mask;
            }
        }
        else
        {
            // キーを離した時点でクリック判定時間より短ければクリックフラグを立てます。
            u32 count = m_ClickHold[ i ];
            m_ClickHold[ i ] = 0;

            if ( count > 0           &&
                 count < CLICK_COUNT &&
                 (! (m_ClickIgnore & i_mask) ) )
            {
                m_Click |= i_mask;
            }

            m_ClickIgnore &= ~i_mask;
        }
    }
}

//---------------------------------------------------------------------------
u32
PadProcessor::IsRepeated( u32 key, u32 start, u32 span ) const
{
    u32 repeat;

    repeat = m_Trigger;

    for (int i = 0; i < MAX_KEY; ++i)
    {
        u32 i_mask = ( 1 << i );
        if ( ! (m_Press & i_mask) ) { continue; }

        s32 count = m_Counter[ i ] - start;
        if ( count >= 0 && count % span == 0 )
        {
            repeat |= i_mask;
        }
    }

    return repeat & key;
}



} // namespace demo
} // namespace nw

