﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_PERFORMANCE_MONITOR_CAFE_H_
#define NW_DEMO_PERFORMANCE_MONITOR_CAFE_H_

#include <nw/dev/dev_DevTextWriter.h>
#include <nw/ut/ut_MoveArray.h>
#include <nw/ut/ut_SafeString.h>
#include <cafe/pmcpu.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief        デモ用 Performance Monitor 計測・結果描画クラスです。
//---------------------------------------------------------------------------
class PerformanceMonitorCafe
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    textWriter 設定する DevTextWriter です。
    //---------------------------------------------------------------------------
    /* ctor */ explicit PerformanceMonitorCafe( nw::dev::DevTextWriter* textWriter );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    /* dtor */ ~PerformanceMonitorCafe() {}


    //---------------------------------------------------------------------------
    //! @brief        パフォーマンスカウンタレジスタで監視するイベントを設定します。
    //!
    //!               計測を開始した後は次のフレームの開始まで SetGroup を行うことはできません。
    //!
    //! @param[in]    e1        パフォーマンスカウンタレジスタ 1 で監視するイベントです。
    //! @param[in]    label1    イベント 1 の結果表示時の名前です。
    //! @param[in]    e2        パフォーマンスカウンタレジスタ 2 で監視するイベントです。
    //! @param[in]    label2    イベント 2 の結果表示時の名前です。
    //! @param[in]    e3        パフォーマンスカウンタレジスタ 3 で監視するイベントです。
    //! @param[in]    label3    イベント 3 の結果表示時の名前です。
    //! @param[in]    e4        パフォーマンスカウンタレジスタ 4 で監視するイベントです。
    //! @param[in]    label4    イベント 4 の結果表示時の名前です。
    //---------------------------------------------------------------------------
    void SetGroup( u32 e1, const char* label1, u32 e2, const char* label2, u32 e3, const char* label3, u32 e4, const char* label4 );

    //---------------------------------------------------------------------------
    //! @brief        パフォーマンスモニタによる計測を開始します。
    //!
    //!               BeginMeasure と EndMeasure で挟まれた区間を計測します。
    //!               複数区間を計測することも可能です。最大 32 区間計測できます。
    //!
    //! @param[in]    区間名です。結果表示時にラベルとして表示されます。
    //---------------------------------------------------------------------------
    void BeginMeasure( const char* name );

    //---------------------------------------------------------------------------
    //! @brief        パフォーマンスモニタによる計測を終了します。
    //---------------------------------------------------------------------------
    void EndMeasure();

    //---------------------------------------------------------------------------
    //! @brief        計測したパフォーマンスモニタの結果を描画します。
    //---------------------------------------------------------------------------
    void DrawResult();

    //---------------------------------------------------------------------------
    //! @brief        フレーム終了時処理を行います。
    //---------------------------------------------------------------------------
    void EndFrame();


    //---------------------------------------------------------------------------
    //! @brief        文字列描画用の行列を設定します。
    //!
    //! @param[in]    projMtx   プロジェクション行列です。
    //! @param[in]    viewMtx    ビュー行列です。
    //---------------------------------------------------------------------------
    void SetMatrix( const nw::math::MTX44& projMtx, const nw::math::MTX34& viewMtx )
    {
        m_ProjMtx = &projMtx;
        m_ViewMtx = &viewMtx;
    }

    //---------------------------------------------------------------------------
    //! @brief        文字描画に用いる DevTextWriter を設定します。
    //!
    //! @param[in]    textWriter 設定する DevTextWriter です。
    //---------------------------------------------------------------------------
    void SetTextWriter( nw::dev::DevTextWriter* textWriter ) { m_TextWriter = textWriter; }

    //---------------------------------------------------------------------------
    //! @brief        設定されている DevTextWriter を取得します。
    //!
    //! @return       設定されている DevTextWriter を返します。
    //---------------------------------------------------------------------------
    nw::dev::DevTextWriter* GetTextWriter() const { return m_TextWriter; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の位置を設定します。
    //!
    //! @param[in]    x         X 座標です。
    //! @param[in]    y         Y 座標です。
    //---------------------------------------------------------------------------
    void SetPosition( f32 x, f32 y ) { m_PosX = x; m_PosY = y; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の X 座標位置を設定します。
    //!
    //! @param[in]    x         X 座標です。
    //---------------------------------------------------------------------------
    void SetPositionX( f32 x ) { m_PosX = x; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の Y 座標位置を設定します。
    //!
    //! @param[in]    y         Y 座標です。
    //---------------------------------------------------------------------------
    void SetPositionY( f32 y ) { m_PosY = y; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の X 座標位置を取得します。
    //!
    //! @return       X 座標を返します。
    //---------------------------------------------------------------------------
    f32 GetPositionX() const { return m_PosX; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の Y 座標位置を取得します。
    //!
    //! @return       Y 座標を返します。
    //---------------------------------------------------------------------------
    f32 GetPositionY() const { return m_PosY; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示のセルの最小幅を設定します。
    //!
    //! @param[in]    width     最小幅です。
    //---------------------------------------------------------------------------
    void SetCellMinWidth( f32 width ) { m_CellMinWidth = width; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示のセルの最小幅を取得します。
    //!
    //! @return       最小幅を返します。
    //---------------------------------------------------------------------------
    f32 GetCellMinWidth() const { return m_CellMinWidth; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示のセルのマージンを設定します。
    //!
    //! @param[in]    margin    マージンです。
    //---------------------------------------------------------------------------
    void SetCellMargin( f32 margin ) { m_CellMargin = margin; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示のセルのマージンを取得します。
    //!
    //! @return       マージンを返します。
    //---------------------------------------------------------------------------
    f32 GetCellMargin() const { return m_CellMargin; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の文字色を設定します。
    //!
    //! @param[in]    color     文字色です。
    //---------------------------------------------------------------------------
    void SetTextColor( const nw::ut::Color4u8& color ) { m_TextColor = color; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の文字色を取得します。
    //!
    //! @return       文字色を返します。
    //---------------------------------------------------------------------------
    const nw::ut::Color4u8 GetTextColor() const { return m_TextColor; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の罫線の色を設定します。
    //!
    //! @param[in]    color     罫線の色です。
    //---------------------------------------------------------------------------
    void SetLineColor( const nw::ut::Color4u8& color ) { m_LineColor = color; }

    //---------------------------------------------------------------------------
    //! @brief        結果表示の罫線の色を取得します。
    //!
    //! @return       罫線の色を返します。
    //---------------------------------------------------------------------------
    const nw::ut::Color4u8 GetLineColor() const { return m_LineColor; }

    //---------------------------------------------------------------------------
    //! @brief        計測表示の有効/無効を設定します。
    //!
    //! @param[in]    isEnabled 有効にする場合、 true を設定します。
    //---------------------------------------------------------------------------
    void SetIsEnabled( bool isEnabled ) { m_IsEnabled = isEnabled; }

    //---------------------------------------------------------------------------
    //! @brief        計測表示の有効/無効を取得します。
    //!
    //! @return       有効の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool GetIsEnabled() const { return m_IsEnabled; }

private:
    //! @brief 区間計測用構造体です。
    struct Section
    {
        PMCPUGroup                  counters;       //!< PM グループ構造体です。
        nw::ut::FixedSafeString<32> labelSection;   //!< 区間名です。
    };

    static const u32            MAX_SECTION = 32;       //!< 計測できる区間数の最大値です。
    nw::ut::MoveArray<Section>  m_SectionArray[2];      //!< 計測区間の配列です。
    Section                     m_SectionBuffer[MAX_SECTION * 2];  //!< 計測区間の配列用バッファです。
    s32                         m_CountersNum[2];       //!< 計測区間数を保持します。
    s32                         m_CurrentBuffer;        //!< 現在の計測用バッファです。
    Section                     m_CurrentSection;       //!< 現在計測に用いている Section です。
    s32                         m_Count;                //!< 計測区間数をカウントします。
    s32                         m_OverNum;              //!< バッファを超えた計測区間数です。
    nw::ut::FixedSafeString<32> m_Label[4];             //!< 結果表示用ラベルです。
    f32                         m_PosX;                 //!< 結果表示位置の X 座標です。
    f32                         m_PosY;                 //!< 結果表示位置の Y 座標です。
    f32                         m_CellMinWidth;         //!< 結果表示のセルの最小幅です。
    f32                         m_CellMargin;           //!< 結果表示のセルのマージンです。
    nw::ut::Color4u8            m_TextColor;            //!< 結果表示の文字色です。
    nw::ut::Color4u8            m_LineColor;            //!< 結果表示の罫線の色です。
    nw::dev::DevTextWriter*     m_TextWriter;           //!< 文字列描画に用いる DevTextWriter です。
    const nw::math::MTX44*      m_ProjMtx;              //!< 描画用プロジェクション行列です。
    const nw::math::MTX34*      m_ViewMtx;              //!< 描画用ビュー行列です。
    bool                        m_IsEnabled;            //!< 計測・表示が有効かを表します。
};

} // namespace nw::demo
} // namespace nw

#endif // NW_DEMO_PERFORMANCE_MONITOR_CAFE_H_
