﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{StatTracker.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a helper class that
 *  sets up command buffers to report GPU
 *  statistic counters and reset those counters.
 *  It also provides functionality to print
 *  those counters to the screen using a
 *  given DebugTextRenderer.
 */

#pragma once

#include <nvn/nvn.h>
#include "DebugTextRenderer.h"
#include "MemoryPool.h"
#include "FrameBufferManager.h"
#include "UniformBufferManager.h"

/*
 * GPU Statistics
 * --------------
 * NVN has a set of counters that can be queried to view statistics about various GPU events.
 * The types of these counters are defined in NVNcounterType with the name NVN_COUNTER_TYPE_**
 * and are listed below. When a counter is reported, it contains both a timestamp and the queried data.
 *
 * TIMESTAMP                             - No counter data, just the timestamp.
 *
 * SAMPLES_PASSED                        - Number of samples that passed the depth/stencil test.
 *
 * INPUT_VERTICES                        - Total number of vertices specified by the draw command.
 *
 * INPUT_PRIMITIVES                      - Number of primitives specified by the draw command.
 *
 * VERTEX_SHADER_INVOCATIONS             - Number of verex shader invocations. May not match INPUT_VERTICES
 *                                         if some indices are duplicated.
 *
 * TESS_CONTROL_SHADER_INVOCATIONS       - Number of tesselation control shader invocations.
 *
 * TESS_EVALUATION_SHADER_INVOCATIONS    - Number of tesselation evaluation shader invocations.
 *
 * GEOMETRY_SHADER_INVOCATIONS           - Number of geometry shader invocations.
 *
 * FRAGMENT_SHADER_INVOCATIONS           - Number of fragment shader invocations. Fragments may be discarded
 *                                         before invoking the fragment because of the depth/stencil test.
 *
 * TESS_EVALUATION_SHADER_PRIMITIVES     - Number of primitives generated by the tesselator.
 *
 * GEOMETRY_SHADER_PRIMITIVES            - Number of primitives generated by the geometry shader.
 *
 * CLIPPER_INPUT_PRIMITIVES              - Number of primitives sent to the clipper.
 *
 * CLIPPER_OUTPUT_PRIMITIVES             - Number of primitives left after clipping.
 *
 * PRIMITIVES_GENERATED                  - Number of primitives produced for clipping and rasterization.
 *
 * TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN - Number of primitives captured during transform feedback.
 *
 */

enum CounterFlags
{
    CounterFlags_Timestamp                               = 0x00000001,
    CounterFlags_Samples_Passed                          = 0x00000002,
    CounterFlags_Input_Vertices                          = 0x00000004,
    CounterFlags_Input_Primitives                        = 0x00000008,
    CounterFlags_Vertex_Shader_Invocations               = 0x00000010,
    CounterFlags_Tesselation_Control_Shader_Invocations  = 0x00000020,
    CounterFlags_Tesselation_EvaluationShaderInvocations = 0x00000040,
    CounterFlags_Geometry_Shader_Invocations             = 0x00000080,
    CounterFlags_Fragment_Shader_Invocations             = 0x00000100,
    CounterFlags_Tesselation_EvaluationShaderPrimitives  = 0x00000200,
    CounterFlags_Geometry_Shader_Primitives              = 0x00000400,
    CounterFlags_Clipper_Input_Primitives                = 0x00000800,
    CounterFlags_Clipper_Output_Primitives               = 0x00001000,
    CounterFlags_Primitives_Generated                    = 0x00002000,
    CounterFlags_Transform_Feedback_Primitives_Written   = 0x00004000,
    CounterFlags_Vertex_Fragment                         = 0x0000391F,
    CounterFlags_All_Counters                            = 0x00007FFF,
};

struct StatCounters
{
    uint64_t m_Timestamp;
    uint64_t m_SamplesPassed;
    uint64_t m_InputVertices;
    uint64_t m_InputPrimitives;
    uint64_t m_VertexShaderInvocations;
    uint64_t m_TessControlShaderInvocations;
    uint64_t m_TessEvalShaderInvocations;
    uint64_t m_GeometryShaderInvocations;
    uint64_t m_FragmentShaderInvocations;
    uint64_t m_TessEvaluationShaderPrimitives;
    uint64_t m_GeometryShaderPrimitives;
    uint64_t m_ClipperInputPrimitives;
    uint64_t m_ClipperOutputPrimitives;
    uint64_t m_PrimitivesGenerated;
    uint64_t m_TransformFeedbackPrimitivesWritten;
};

class StatCounterManager : public FrameBufferedMemoryManager
{
    public:
        NN_IMPLICIT StatCounterManager(NVNdevice* pDevice, int numChunks = 2);
        virtual ~StatCounterManager();

        virtual void SwapPools();
        virtual int GetNumChunks();

        char* GetCounterBufferPointer();
        NVNbufferAddress GetCounterBufferAddress();

        StatCounters* GetCurrentStatCounter();
        StatCounters* GetCounterForPrint();

    private:
        NVNdevice*              m_pDevice;
        StatCounters*           m_pStatCounters;

        int                     m_NumChunks;
        unsigned                m_CurrentChunk;
        unsigned                m_PreviousChunk;

        MemoryPool              m_BufferMemoryPool;
        NVNbuffer               m_Buffer;
        std::vector<ptrdiff_t>  m_Offsets;
};

class StatTracker
{
    public:
        StatTracker();
        ~StatTracker();

        void Init(NVNdevice* pDevice, int flags, FrameBufferedSyncManager* pSyncManager, unsigned numCounters = 2);

        void ReportCounters(NVNqueue* pQueue);
        void ResetCounters(NVNqueue* pQueue);
        void GetCounters(NVNqueue* pQueue);
        void PrintCounters(DebugTextRenderer* pTextRenderer, float column, float startingLine);

        void Shutdown();

    private:
        NVNdevice*              m_pDevice;
        MemoryPool              m_MemoryPool;

        int                     m_CounterAlignment;

        int                     m_CounterFlags;

        ManagedCommandBuffer*   m_pManagedCommandBuffer;
        NVNcommandBuffer        m_ResetCommandBuffer;

        void*                   m_pResetControlMemoryPool;

        NVNsync                 m_CommandBufferSync;
        NVNcommandHandle        m_ReportHandle;
        NVNcommandHandle        m_ResetHandle;
        StatCounterManager*     m_pStatCounterManager;
};
