﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
/**
 * @file mm_MovieVideoOutputHandler.h
 * @brief Handle video decoded data from movie::decoder
 */

#include "mm_MoviePlayerUtils.h"
#include "mm_NvnVideoRenderer.h"

/**
 * @brief
 * MovieVideoOutputHandler class for processing video data from movie::decoder
 *
 */
void MovieVideoOutputHandlerThreadFunction(void *arg);
class MovieVideoOutputHandler
{
public:
   /**
    * @brief
    * MovieAudioOutputHandler constructor.
    *
    * @param[in] nativeWindow                   VI NativeWindowHandle to use.
    * @param[in] videoDecoderMode               Video decoder mode(CPU or Native texture).
    * @param[in] videoDecoderOutputFormat       Video decoder output format.
    *
    * @details
    * An instance of MovieVideoOutputHandler is created. Also, an instance of NvnVideoRenderer is created and
    * initialized.
    *
    */
    NN_IMPLICIT MovieVideoOutputHandler(nn::vi::NativeWindowHandle nativeWindow,
        movie::DecoderMode videoDecoderMode = movie::DecoderMode_Cpu,
        movie::DecoderOutputFormat videoDecoderOutputFormat = movie::DecoderOutputFormat_VideoColorNv12);

   /**
    * @brief
    * MovieVideoOutputHandler destructor.
    *
    * @details
    * NvnVideoRenderer is finalized and instance is deleted. MovieVideoOutputHandler instance is deleted.
    *
    */
    ~MovieVideoOutputHandler();

   /**
    * @brief
    * Initialize MovieVideoOutputHandler.
    *
    * @param[in] mediaClock                     Clock component to do audio-video sync.
    * @param[in] videoRenderComplete            Client event to be signaled when rendering is complete.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Failed to create thread.
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    *
    * @details
    * MovieVideoOutputHandler is initialized. Main processing thread is created, resources are initialized.
    *
    */
    movie::Status Initialize(MediaClock *mediaClock, nn::os::EventType *videoRenderComplete);

   /**
    * @brief
    * Finalize MovieVideoOutputHandler.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * Once Finalize() is called MovieVideoOutputHandler can be initialized again using Initialize().
    * Main processing thread is stopped and destroyed.
    *
    */
    movie::Status Finalize();

   /**
    * @brief
    * Open MovieVideoOutputHandler.
    *
    * @param[in] videoDecoder                   movie::Decoder (video) to use.
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    *
    * @details
    * This API is used for passing video decoder handle to MovieVideoOutputHandler. Video decoder handle is
    * used for acquiring decoded video frame from decoder.
    *
    */
    movie::Status Open(movie::Decoder* videoDecoder, int32_t width, int32_t height);

   /**
    * @brief
    * Start processing data from movie::Decoder.
    *
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Failed to start playback.
    *
    * @details
    * When Start() API is called, MovieVideoOutputHandler main thread is started. The thread loop will call Render() API.
    * This will result in getting video frame from decoder, preparing it for rendering, and finally submitting to
    * NvnVideoRenderer for drawing on display.
    *
    */
    movie::Status Start(int32_t width, int32_t height);

   /**
    * @brief
    * Render video
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             No video frame to render.
    *
    * @details
    * Get video frame from decoder, sync video frame with audio using MediaCLock. If video frame is late it will be
    * dropped, otherwise  get free buffer or textures from NvnVideoRenderer, copy video data (in CPU mode) from
    * decoder buffer. Send video buffer to renderer for drawing on display. When last video frame is seen, an event may
    * be signaled, if it was registered during initialization.
    *
    */
    movie::Status Render();

   /**
    * @brief
    * Stop processing video data from movie::Decoder
    *
    * @return none
    *
    */
    void Stop();

   /**
    * @brief
    * Pause processing video data from movie::Decoder
    *
    * @return none
    *
    * @details
    * Application can use this API to pause playback
    *
    */
    void Pause();

   /**
    * @brief
    * Resume processing video data from movie::Decoder
    *
    * @return none
    *
    * @details
    * Application can use this API to resume playback
    *
    */
    void Resume();

   /**
    * @brief
    * Flush movie::Decoder (video)
    *
    * @return none
    *
    * @details
    * Application can use this API to flush video decoder and output buffers
    * with MovieVideoOutputHandler.
    *
    */
    void Flush();

   /**
    * @brief
    * Close MovieVideoOutputHandler and release resources.
    *
    * @return none
    *
    */
    void Close();

   /**
    * @brief
    * Thread function for MovieVideoOutputHandler thread.
    *
    * @param[in] arg                            Arguments for thread.
    *
    * @return None
    *
    * @details
    * The thread loop function.
    *
    */
    friend void MovieVideoOutputHandlerThreadFunction(void *arg);

   /**
    * @brief
    * Video output buffer available call back.
    *
    * @param[in] index                          Output buffer index from decoder
    * @param[in] presentationTimeUs             Presentation time for this video buffer
    * @param[in] flags                          movie:: buffer flags
    *
    * @return None
    *
    * @details
    * When output buffers are available, callbacks are received by MovieVideoOutputHandler.
    * Decoded video data is pulled from decoder in Render() method.
    *
    */
    void OnOutputAvailable(int32_t index, int64_t presentationTimeUs, uint32_t flags);

   /**
    * @brief
    * DoneProcessing
    *
    * @return boolean
    * @retval true  - Threads need to be terminated.
    * @retval false - Threads need to be running.
    *
    * @details
    * Helper method used to determine exit from thread loop.
    *
    */
    bool DoneProcessing();

   /**
    * @brief
    * AllFramesRendered
    *
    * @return boolean
    * @retval true  - All video frames are rendered.
    * @retval false - All frames not rendered, more frames available for rendering.
    *
    * @details
    * Helper method used to determine whether all video frames are rendered.
    *
    */
    bool AllFramesRendered() { return m_AllFramesRendered; }

   /**
    * @brief
    * AllFramesRendered
    *
    * @return boolean
    * @retval true  - Video frame need to be dropped (late)
    * @retval false - Render video frame now.
    *
    * @details
    * Helper method used to perform AV sync. If video is late it need to be dropped.
    *
    */
    bool AVSync(int64_t presentationTimeUs);

   /**
    * @brief
    * GetRealTimeUs
    *
    * @param[in] mediaTimeUs                          Media time in micro seconds.
    * @param[in] nowUs                                Current time in micro seconds.
    *
    * @return timeUs - Realtime for the media time in micro seconds.
    *
    * @details
    * Helper method to find Realtime for the given media time.
    *
    */
    int64_t GetRealTimeUs(int64_t mediaTimeUs, int64_t nowUs);

   /**
    * @brief
    * GetVideoPostDelayUs
    *
    * @return sleepTimeMs - Sleep time in milliseconds.
    *
    * @details
    * API returns render thread sleep time for smooth video playback.
    *
    */
    int64_t GetVideoPostDelayUs();

   /**
    * @brief
    * IsPaused
    *
    * @return boolean
    * @retval true  - If video rendering is paused.
    * @retval false - If video is playing.
    *
    * @details
    * API to get MovieVideoOutputHandler renderer pause state.
    *
    */
    bool IsPaused() { return m_IsPaused; }

   /**
    * @brief
    * GetPlaybackPosition
    * @param[out] playbackPositionUs             Playback position in micro seconds.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    *
    * @details
    * Returns current video playback position in micro seconds.
    *
    */
    movie::Status GetPlaybackPosition(int64_t *playbackPositionUs) const;

   /**
    * @brief
    * CanStartRendering
    *
    * @return boolean
    * @retval true  - If video rendering can be started.
    * @retval false - If video rendering cannot be started(not enough video data).
    *
    * @details
    * Helper method used to determine starting of video renderer.
    *
    */
    bool CanStartRendering();

private:
    nn::vi::NativeWindowHandle m_NativeWindow;
    nn::os::EventType *m_VideoRenderComplete;
    movie::Decoder* m_VideoDecoder;
    NvnVideoRenderer* m_NvnVideoRenderer;

    bool m_ThreadStarted;
    bool m_ThreadDone;
    bool m_ThreadCreated;
    int64_t m_PresentationTimeUs;
    size_t m_ThreadstackSize;
    void *m_Threadstack;
    nn::os::ThreadType  m_ThreadType;

    struct VideoData
    {
        int32_t index;
        int64_t presentationTimeUs;
        uint32_t flags;
    };

    std::vector<VideoData> m_VideoBuffers;
    nn::os::MutexType m_VideoMutex;
    size_t m_YuvBufferSize;
    int32_t m_Width;
    int32_t m_Height;
private:
    bool m_AllFramesRendered;
    MediaClock *m_MediaClock;
    int64_t m_VideoNeedDelayTimeUs;
    int64_t m_VideoStartTime;
    int m_DroppedVideoFrames;
    bool m_UpdateVideoStartTime;
    int64_t m_LastPresentationTimeUs;
    bool m_IsPaused;
    int64_t m_PlaybackPositionUs;
    int64_t m_VideoOutputBuffersReceived;
    movie::DecoderMode m_VideoDecoderMode;
    movie::DecoderOutputFormat m_VideoDecoderOutputFormat;
};

