﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
 /**
 * @file mm_HidHandler.h
 * @brief HID controller handling.
 */
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <cstdlib>
#include <map>

/**
 * @brief
 * HidHandler class for processing controller input data. Registered clients will be signalled when a controller
 * registered key press is detected.
 *
 */
void HidHandlerThreadFunction(void *arg);
class HidHandler
{
public:
   /**
    * @brief
    * Enumeration of HidHandlerKeyType types.
    *
    * @details
    * HID KeyType types recognized by HidHandler.
    */
    enum HidHandlerKeyType
    {
        HidHandlerKeyType_Unknown    = 0,       //!< Unknown type.
        HidHandlerKeyType_ButtonA    = 1,       //!< Button A.
        HidHandlerKeyType_ButtonB    = 2,       //!< Button B.
        HidHandlerKeyType_ButtonX    = 3,       //!< Button X.
        HidHandlerKeyType_ButtonZr   = 4,       //!< Button Zr.
        HidHandlerKeyType_ButtonL    = 5,       //!< Button L.
        HidHandlerKeyType_ButtonR    = 6,       //!< Button R.
    };

   /**
    * @brief
    * HidHandler constructor.
    *
    */
    HidHandler();

   /**
    * @brief
    * Initialize
    *
    * @return Boolean
    * @retval true  - Initialize success.
    * @retval false - Initialize failed.
    *
    * @details
    * API to initialize HidHandler.
    */
    bool Initialize();

   /**
    * @brief
    * Finalize HidHandler.
    *
    * @return None
    *
    * @details
    * This API will stop and destroy the HidHandler thread. All the resources will be freed.
    */
    void Finalize();

   /**
    * @brief
    * ProcessHidEvent.
    *
    * @return None
    *
    * @details
    * HidHandler thread loop will call this API to sample controller key press. If any key registered with HidHandler
    * is pressed, it will signal clients event.
    */
    void ProcessHidEvent();

   /**
    * @brief
    * Thread function for HidHandler thread.
    *
    * @param[in] arg                            Arguments for thread.
    *
    * @return None
    *
    * @details
    * This is HidHandler thread loop function. This thread loop will call ProcessHidEvent() API and then will sleep.
    * The sleep duration is determined by calling GetThreadSleepTimeMs(). This threads samples keys at 16ms interval.
    *
    */
    friend void HidHandlerThreadFunction(void *arg);

   /**
    * @brief
    * GetThreadSleepTimeMs.
    *
    * @return sleep time in milliseconds.
    *
    * @details
    * HidHandler thread loop function use this API get sleep time. This is set to 16ms.
    *
    */
    int64_t GetThreadSleepTimeMs();

   /**
    * @brief
    * Exit thread
    *
    * @return Boolean
    * @retval true  - Threads need to be terminated.
    * @retval false - Threads need to be running.
    *
    * @details
    * API to get thread loop exit condition.
    *
    */
    bool ExitThread();

   /**
    * @brief
    * RegisterEvent.
    *
    * @param[in] nnOsEvent           nn::os::EventType to register
    * @param[in] keyType             HidHandlerKeyType to register
    *
    * @return None
    *
    * @details
    * Client can call RegisterEvent() API to register an event and a type of controller key type. When the key is
    * pressed the registered event is signaled.
    *
    */
    void RegisterEvent(nn::os::EventType *nnOsEvent, HidHandlerKeyType keyType);

private:
    bool m_ThreadCreated;
    bool m_ThreadStarted;
    bool m_ThreadDone;
    size_t m_Threadstacksize;
    void *m_Threadstack;
    nn::os::ThreadType  m_ThreadType;
    int64_t m_ThreadSleepTimeMs;

    int m_NpadIdCount;
    int64_t m_PrevPadSamplingNumber;
    nn::hid::DebugPadState m_DebugPadState[ nn::hid::DebugPadStateCountMax ];
    nn::hid::NpadIdType *m_NpadIds;
    nn::hid::NpadJoyDualState *m_OldNpadJoyDualState;
    nn::hid::NpadJoyDualState *m_CurrentNpadJoyDualState;
    nn::hid::NpadButtonSet m_AllCurrentNpadButtonsPressed;
    nn::hid::NpadButtonSet m_AllCurrentNpadButtonsHeld;
    std::map<HidHandlerKeyType, nn::os::EventType*> m_EventMap;
};
