﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Ronde に関する API の宣言
 */

#pragma once

#include <nn/hidbus/hidbus.h>

namespace nns { namespace hidbus {

/**
* @brief Ronde センサーのデータサイズ (Byte) です。
*/
const size_t RondeSensorDataSize = 2;

/**
* @brief Ronde の Unique ID のサイズ (Byte) です。
*/
const size_t RondeUniqueIdSize = 12;

/**
* @brief Ronde FW のバージョンです。
*/
struct RondeFwVersion
{
    uint8_t mainVersion;     //!< メインバージョン
    uint8_t subVersion;      //!< サブバージョン
};

/**
* @brief Ronde センサーの受信データです。
*/
struct RondeSensorData
{
    int16_t data;                    //!< 受信データです。
    int64_t samplingNumber;          //!< 受信データのサンプリングナンバーです。
};

/**
* @brief Ronde の工程キャル値を列挙した構造体です。
*/
struct RondeManufactureCalibrationData
{
    int16_t osMax;     //!< osMax
    int16_t hkMax;     //!< hkMax
    int16_t zeroMin;   //!< zeroMin
    int16_t zeroMax;   //!< zeroMax
};

/**
* @brief Ronde の内部エラー発生回数の構造体です。
*/
struct RondeInternalErrorInfo
{
    uint32_t invalidCommandCount;     //!< invalidCommandCount
    uint32_t invalidDataNameCount;    //!< invalidDataNameCount
    uint32_t invalidDataSizeCount;    //!< invalidDataSizeCount

    uint32_t uartParityErrorCount;    //!< uartParityErrorCount
    uint32_t uartNoiseErrorCount;     //!< uartNoiseErrorCount
    uint32_t uartFrameErrorCount;     //!< uartFrameErrorCount
    uint32_t uartOverrunErrorCount;   //!< uartOverrunErrorCount
    uint32_t uartDmaErrorCount;       //!< uartDmaErrorCount
    uint32_t uartBusyErrorCount;      //!< uartBusyErrorCount

    uint32_t resetAdsCount;           //!< resetAdsCount
};

/**
 * @brief       Ronde を有効化します。
 *
 * @details     Joy-Con レールに　Ronde　が接続された状態で本関数を呼ぶと Ronde が有効化され、電源供給されます。
 *              有効化状態になると、 NFC / IR 等の排他が必要な機能が使用できなくなります@n
 *              Joy-Con レールから Ronde が取り外された場合や有効化したアプリケーションがインフォーカス状態でなくなると自動で無効化され、電源供給がなくなります。
 *              インフォーカス状態以外で本 API を呼ぶと、ResultNotInForcus が返り有効化できません。
 *              インフォーカス状態への遷移時に再度有効化してください。@n
 *              拡張バス機能を使用するために対応コントローラのファームウェアに更新が必要な場合に本関数を呼び出すと、
 *              ファームウェア更新を行う UI を表示し、ファームウェアを更新した上で有効化処理を行います。@n
 *              本機能はファームウェア更新中や、Joy-Con レールへのデバイス接続時にデバイスとの通信準備を行う際にブロッキングしますのでご注意ください。
 *              各ハンドルごとにワークバッファを渡してください。
 *              ワークバッファには JoyPollingModeWorkBufferSize 以上のサイズが必要です。
 *              本関数に渡したワークバッファは DisableRonde() の呼び出し、 Ronde の無効化、 handle の無効化時に再利用可能となります。
 *              再利用可能になるまでは、ワークバッファへアクセスしないでください。
 *
 * @param[in]   handle                    拡張バスのハンドル
 * @param[in]   workBuffer                ワークバッファ
 * @param[in]   workBufferSize            ワークバッファのサイズ
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                            処理に成功しました。}
 *      @handleresult{nn::hidbus::ResultFWUpdateFailed,             拡張バス機能を使用するためのコントローラのアップデートに失敗しました。}
 *      @handleresult{nn::hidbus::ResultInvalidHandle,              無効な handle です。}
 *      @handleresult{nn::hidbus::ResultNotInForcus,                インフォーカス状態でないため有効化できません。インフォーカス状態になった後、再度本 API を呼び出してください。}
 *      @handleresult{nn::hidbus::ResultActivateFailureNpadBusy,    拡張バス機能と排他で制御されるべき他の機能が有効なため有効化できませんでした。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceNotAttached,  拡張バスにデバイスが接続されていません。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceReadyTimeout, 拡張バスに接続されたデバイスとの通信準備のタイムアウトです。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,      拡張バスに接続されたデバイスとの通信のタイムアウトです。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceInvalidId,    Ronde と異なるデバイスが拡張バスに接続されています。}
 * @endretresult
 *
 * @pre
 *  - nn::hidbus::Initialize() の呼び出しが完了している
 *  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
 *  - workBuffer != nullptr
 *  - workBuffer が nn::os::MemoryPageSize でアラインされている
 */
nn::Result EnableRonde(const nn::hidbus::BusHandle& handle, void* workBuffer, size_t workBufferSize) NN_NOEXCEPT;

/**
 * @brief       Ronde を無効化します。
 *
 * @details     Joy-Con レールに　Ronde　が接続された状態で本関数を呼ぶと Ronde が無効化されます。
 *              無効化することで、 NFC / IR 等の排他が必要な機能が使用できるようになります。
 *              本関数の呼び出し後、ワークバッファが使用可能になります。
 *
 * @param[in]   handle          　　　拡張バスのハンドル
 *
 * @retresult
 *      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
 *      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
 *      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
 * @endretresult
 *
 * @pre
 *  - nn::hidbus::Initialize() の呼び出しが完了している
 *  - EnableRonde() の呼び出しが完了している
 *  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
 */
nn::Result DisableRonde(const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;


/**
* @brief       Ronde センサーの最新の値を取得します。
*
* @details     指定の handle と対応する Joy-Con レールに接続された Ronde から GetRondeSensorData() で 1 つの受信データを読み出した時と同じ値が返ります。
*
* @param[in]   pOutData                　最新の受信データ
* @param[in]   handle                    受信データをコピーする拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - nn::hidbus::Initialize() の呼び出しが完了している。
*  - EnableRonde() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetRondeSensorData(RondeSensorData* pOutData, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       Ronde センサーの値を過去に遡って読み出します。
*
* @details     最新のものから過去に遡って利用可能な数だけ順に、指定の handle と対応する Joy-Con レールに接続された Ronde からセンサーデータを読み出します。
*              利用可能な入力状態の数より大きなバッファ（配列）が指定された場合、余った領域に対しては何も行いません。
*              読み出し可能な入力状態の最大数は 10 です。
*              利用可能な入力状態には読み出し済みのものも含まれます。
*              差分だけを利用したい場合は RondeSensorData::samplingNumber を参照してください。
*              内部で接続されたデバイスとの通信のタイムアウトが発生した場合、データ更新されません。
*
* @param[out]  pOutData                　受信データを受け取るバッファへのポインタ
* @param[in]   count                　   受信データを受け取る数
* @param[in]   handle                   受信データをコピーする拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - nn::hidbus::Initialize() の呼び出しが完了している。
*  - EnableRonde() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetRondeSensorData(RondeSensorData* pOutData, int count, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       Ronde センサーの Thermistor の値を取得します。
*
* @details     Ronde センサーの Thermistor の値を取得します。
*
* @param[in]   pOutData                　最新の値
* @param[in]   handle                   受信データをコピーする拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - nn::hidbus::Initialize() の呼び出しが完了している。
*  - EnableRonde() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetRondeThermisterData(int16_t* pOutData, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       Ronde センサーの工程キャルの値を取得します。
*
* @details     Ronde センサーの工程キャルの値を取得します。
*              本関数はデバイスとの通信が発生するため、データの受信完了までブロッキングします。
*
* @param[in]   pOutData                　工程キャルの値をまとめた構造体
* @param[in]   handle                   拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - nn::hidbus::Initialize() の呼び出しが完了している。
*  - EnableRonde() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetRondeManufactureCalibrationData(RondeManufactureCalibrationData* pOutData, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       Ronde の Unique ID を取得します。
*
* @details     Ronde の Unique ID を取得します。
*              Unique ID は 96 bit です。
*              RondeUniqueIdSize 以上のバッファを渡してください。
*              本関数はデバイスとの通信が発生するため、データの受信完了までブロッキングします。
*
* @param[in]   pOutData                　工程キャルの値をまとめた構造体
* @param[in]   handle                   拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - nn::hidbus::Initialize() の呼び出しが完了している。
*  - EnableRonde() の呼び出しが完了している
*  - bufferSize >= RondeUniqueIdSize
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetRondeUniqueId(uint8_t* pOutBuffer, size_t bufferSize, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       Ronde のファームウェアバージョンを取得します。
*
* @details     Ronde のファームウェアバージョンを取得します。
*              ファームウェアバージョンはメインバージョンとサブバージョンの 2つで表されます。
*              本関数はデバイスとの通信が発生するため、データの受信完了までブロッキングします。
*
* @param[in]   pOutFwVersion            ファームウェアバージョン構造体へのポインタ
* @param[in]   handle                   拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - nn::hidbus::Initialize() の呼び出しが完了している。
*  - EnableRonde() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetRondeFirmwareVersion(RondeFwVersion* pOutFwVersion, const nn::hidbus::BusHandle& handle) NN_NOEXCEPT;

/**
* @brief       Ronde 内部で発生したエラー情報の統計を取得します。
*
* @details     Ronde 内部で発生したエラー情報の統計を取得します。
*              本関数はエラー内容によってリアルタイムに処理を分岐するためのものではありません。
*              エラーレポート等によって、ハードウェアのエラー情報を解析するために使用することを想定しています。
*              他の関数内で発生した内部エラーをライブラリ内で保持しており、その値が本関数で取得されます。
*              本エラー情報はアプリを終了するとリセットされます。
*              本関数はデバイス別での取得には対応していません。
*              また、複数の Ronde で発生したエラー情報は合算して取得されます。
*
* @param[in]   pOutInternalErrorInfo    内部エラー構造体へのポインタ
* @param[in]   handle                   拡張バスへのハンドル
*
*/
void GetRondeInternalErrorInfo(RondeInternalErrorInfo* pOutInternalErrorInfo) NN_NOEXCEPT;

}} // namespace nns::hidbus
