﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       キーボードとマウスの機能を持つコントローラに関する API の宣言
 */

#pragma once

#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Button.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerId.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_KeyboardAsset.h>
#include <nns/hid/hid_MouseAsset.h>

namespace nns { namespace hid {

/**
 * @brief       キーアクション定義です。
 */
struct KeyAction
{
    typedef nn::util::BitFlagSet<32, KeyAction>::Flag<0>
            Down;   //!< キーが押下されました。
    typedef nn::util::BitFlagSet<32, KeyAction>::Flag<1>
            Up;     //!< キーが開放されました。
    typedef nn::util::BitFlagSet<32, KeyAction>::Flag<8>
            Repeat; //!< キーがリピートされました。
};

/**
 * @brief       キーアクションの集合を扱う型です。
 */
typedef nn::util::BitFlagSet<32, KeyAction> KeyActionSet;

/**
 * @brief       キーイベントを表す構造体です。
 */
struct KeyEvent
{
    KeyActionSet action;                    //!< キーアクションです。
    nn::hid::KeyboardModifierSet modifiers; //!< キー修飾です。
    bool isToggleOn;                        //!< トグル状態です。
    int32_t usageId;                        //!< Usage ID です。

    //! @name   キーアクションの取得
    //! @{

    /**
     * @brief       キーの押下によるイベントか否かを表す値を返します。
     *
     * @return      キーの押下によるイベントか否かを表す値です。
     */
    bool IsKeyDown() const NN_NOEXCEPT
    {
        return this->action.Test<KeyAction::Down>();
    }

    /**
     * @brief       キーの開放によるイベントか否かを表す値を返します。
     *
     * @return      キーの開放によるイベントか否かを表す値です。
     */
    bool IsKeyUp() const NN_NOEXCEPT
    {
        return this->action.Test<KeyAction::Up>();
    }

    /**
     * @brief       キーのリピートによるイベントか否かを表す値を返します。
     *
     * @return      キーのリピートによるイベントか否かを表す値です。
     */
    bool IsKeyRepeat() const NN_NOEXCEPT
    {
        return this->action.Test<KeyAction::Repeat>();
    }

    //! @}

    //! @name   キー修飾の取得
    //! @{

    /**
     * @brief       Control キーが押下中か否かを表す値を返します。
     *
     * @return      Control キーが押下中か否かを表す値です。
     */
    bool IsControlKeyPressed() const NN_NOEXCEPT
    {
        return this->modifiers.Test<nn::hid::KeyboardModifier::Control>();
    }

    /**
     * @brief       Shift キーが押下中か否かを表す値を返します。
     *
     * @return      Shift キーが押下中か否かを表す値です。
     */
    bool IsShiftKeyPressed() const NN_NOEXCEPT
    {
        return this->modifiers.Test<nn::hid::KeyboardModifier::Shift>();
    }

    /**
     * @brief       CapsLock が有効か否かを表す値を返します。
     *
     * @return      CapsLock が有効か否かを表す値です。
     */
    bool IsCapsLockOn() const NN_NOEXCEPT
    {
        return this->modifiers.Test<nn::hid::KeyboardModifier::CapsLock>();
    }

    /**
     * @brief       NumLock が有効か否かを表す値を返します。
     *
     * @return      NumLock が有効か否かを表す値です。
     */
    bool IsNumLockOn() const NN_NOEXCEPT
    {
        return this->modifiers.Test<nn::hid::KeyboardModifier::NumLock>();
    }

    //! @}

    //! @name   キー属性の取得
    //! @{

    /**
     * @brief       キーのトグル状態が有効か否かを表す値を返します。
     *
     * @return      キーのトグル状態が有効か否かを表す値です。
     */
    bool IsToggleOn() const NN_NOEXCEPT
    {
        return this->isToggleOn;
    }

    /**
     * @brief       キーが印字可能な文字を生成するか否かを表す値を返します。
     *
     * @return      キーが印字可能な文字を生成するか否かを表す値です。
     */
    bool IsPrintingKey() const NN_NOEXCEPT;

    /**
     * @brief       キーの Usage ID を返します。
     *
     * @return      キーの Usage ID です。
     */
    int GetUsageId() const NN_NOEXCEPT
    {
        return this->usageId;
    }

    //! @}
};

/**
 * @brief   キーボードとマウスの機能を持つコントローラクラスです。
 */
class KeyboardAndMouse : public Controller
{
    NN_DISALLOW_COPY(KeyboardAndMouse);
    NN_DISALLOW_MOVE(KeyboardAndMouse);

public:
    /**
     * @brief       KeyboardAndMouse のコンストラクタです。
     *
     * @param[in]   pManager                    KeyboardAndMouse の管理者です。
     *
     * @pre
     *              - pManager != NULL
     */
    explicit KeyboardAndMouse(ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       KeyboardAndMouse のデストラクタです。
     */
    virtual ~KeyboardAndMouse() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       KeyboardAndMouse の状態を更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       KeyboardAndMouse のコントローラ識別子を返します。
     *
     * @return      KeyboardAndMouse のコントローラ識別子です。
     */
    virtual ControllerId GetControllerId() NN_NOEXCEPT NN_OVERRIDE
    {
        return ControllerId_KeyboardAndMouse;
    }

    /**
     * @brief       キーボードかマウスのいずれかが接続状態にあるか否かを返します。
     *
     * @return      キーボードかマウスのいずれかが接続状態にあるか否かを表す値です。
     */
    virtual bool IsConnected() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsKeyboardConnected || m_IsMouseConnected;
    }

    //! @name   操作情報の取得
    //! @{

    /**
     * @brief       押下中のキーを返します。
     *
     * @return      押下中のキーです。
     */
    const nn::hid::KeyboardKeySet& GetKeys() const NN_NOEXCEPT
    {
        return m_Keys;
    }

    /**
     * @brief       最後の更新で新たに押下されたキーを返します。
     *
     * @return      最後の更新で新たに押下されたキーです。
     */
    const nn::hid::KeyboardKeySet& GetKeysDown() const NN_NOEXCEPT
    {
        return m_KeysDown;
    }

    /**
     * @brief       最後の更新で新たに開放されたキーを返します。
     *
     * @return      最後の更新で新たに開放されたキーです。
     */
    const nn::hid::KeyboardKeySet& GetKeysUp() const NN_NOEXCEPT
    {
        return m_KeysUp;
    }

    /**
     * @brief       リピートされたキーを返します。
     *
     * @return      リピートされたキーです。
     */
    const nn::hid::KeyboardKeySet& GetKeysRepeat() const NN_NOEXCEPT
    {
        return m_KeysRepeat;
    }

    /**
     * @brief       トグル状態にあるキーを返します。
     *
     * @return      トグル状態にあるキーです。
     */
    const nn::hid::KeyboardKeySet& GetKeysToggle() const NN_NOEXCEPT
    {
        return m_KeysToggle;
    }

    /**
     * @brief       押下中のマウスボタンを返します。
     *
     * @return      押下中のマウスボタンです。
     */
    const nn::hid::MouseButtonSet& GetMouseButtons() const NN_NOEXCEPT
    {
        return m_MouseButtons;
    }

    /**
     * @brief       最後の更新で新たに押下されたマウスボタンを返します。
     *
     * @return      最後の更新で新たに押下されたマウスボタンです。
     */
    const nn::hid::MouseButtonSet& GetMouseButtonsDown() const NN_NOEXCEPT
    {
        return m_MouseButtonsDown;
    }

    /**
     * @brief       最後の更新で新たに開放されたマウスボタンを返します。
     *
     * @return      最後の更新で新たに開放されたマウスボタンです。
     */
    const nn::hid::MouseButtonSet& GetMouseButtonsUp() const NN_NOEXCEPT
    {
        return m_MouseButtonsUp;
    }

    /**
     * @brief       ダブルクリックされたマウスボタンを返します。
     *
     * @return      ダブルクリックされたマウスボタンです。
     */
    const nn::hid::MouseButtonSet& GetMouseButtonsDoubleClick(
        ) const NN_NOEXCEPT
    {
        return m_MouseButtonsDoubleClick;
    }

    /**
     * @brief       最後の更新で得られたマウスのホイールの回転量を返します。
     *
     * @return      最後の更新で得られたマウスのホイールの回転量です。
     */
    int32_t GetMouseWheel() const NN_NOEXCEPT
    {
        return m_MouseWheel;
    }

    //! @}

    //! @name   キーの押下判定
    //! @{

    /**
     * @brief       指定されたキーの何れかが押下中か否かを返します。
     *
     * @param[in]   keys                        キーの指定です。
     *
     * @return      指定されたキーの何れかが押下中か否かを表す値です。
     */
    bool HasAnyKeys(const nn::hid::KeyboardKeySet& keys) const NN_NOEXCEPT
    {
        return (m_Keys & keys).IsAnyOn();
    }

    /**
     * @brief       指定されたキーの何れかが最後の更新で新たに押下されたか否かを返します。
     *
     * @param[in]   keys                        キーの指定です。
     *
     * @return      指定されたキーの何れかが最後の更新で新たに押下されたか否かを表す値です。
     */
    bool HasAnyKeysDown(const nn::hid::KeyboardKeySet& keys) const NN_NOEXCEPT
    {
        return (m_KeysDown & keys).IsAnyOn();
    }

    /**
     * @brief       指定されたキーの何れかが最後の更新で新たに開放されたか否かを返します。
     *
     * @param[in]   keys                        キーの指定です。
     *
     * @return      指定されたキーの何れかが最後の更新で新たに開放されたか否かを表す値です。
     */
    bool HasAnyKeysUp(const nn::hid::KeyboardKeySet& keys) const NN_NOEXCEPT
    {
        return (m_KeysUp & keys).IsAnyOn();
    }

    /**
     * @brief       指定されたキーの何れかがリピートされたか否かを返します。
     *
     * @param[in]   keys                        キーの指定です。
     *
     * @return      指定されたキーの何れかがリピートされたか否かを表す値です。
     */
    bool HasAnyKeysRepeat(const nn::hid::KeyboardKeySet& keys) const NN_NOEXCEPT
    {
        return (m_KeysRepeat & keys).IsAnyOn();
    }

    /**
     * @brief       指定されたキーの何れかがトグル状態にあるか否かを返します。
     *
     * @param[in]   keys                        キーの指定です。
     *
     * @return      指定されたキーの何れかがトグル状態にあるか否かを表す値です。
     */
    bool HasAnyKeysToggle(const nn::hid::KeyboardKeySet& keys) const NN_NOEXCEPT
    {
        return (m_KeysToggle & keys).IsAnyOn();
    }

    /**
     * @brief       指定されたキーの全てが押下中か否かを返します。
     *
     * @param[in]   keys                        キーの指定です。
     *
     * @return      指定されたキーの全てが押下中か否かを表す値です。
     */
    bool HasAllKeys(const nn::hid::KeyboardKeySet& keys) const NN_NOEXCEPT
    {
        return ((m_Keys & keys) == keys);
    }

    //! @}

    //! @name   キー修飾の取得
    //! @{

    /**
     * @brief       キー修飾を返します。
     *
     * @return      キー修飾です。
     */
    nn::hid::KeyboardModifierSet GetKeyModifiers() const NN_NOEXCEPT
    {
        return m_KeyModifiers;
    }

    /**
     * @brief       Control キーが押下中か否かを表す値を返します。
     *
     * @return      Control キーが押下中か否かを表す値です。
     */
    bool IsControlKeyPressed() const NN_NOEXCEPT
    {
        return m_KeyModifiers.Test<nn::hid::KeyboardModifier::Control>();
    }

    /**
     * @brief       Shift キーが押下中か否かを表す値を返します。
     *
     * @return      Shift キーが押下中か否かを表す値です。
     */
    bool IsShiftKeyPressed() const NN_NOEXCEPT
    {
        return m_KeyModifiers.Test<nn::hid::KeyboardModifier::Shift>();
    }

    /**
     * @brief       CapsLock が有効か否かを表す値を返します。
     *
     * @return      CapsLock が有効か否かを表す値です。
     */
    bool IsCapsLockOn() const NN_NOEXCEPT
    {
        return m_KeyModifiers.Test<nn::hid::KeyboardModifier::CapsLock>();
    }

    /**
     * @brief       NumLock が有効か否かを表す値を返します。
     *
     * @return      NumLock が有効か否かを表す値です。
     */
    bool IsNumLockOn() const NN_NOEXCEPT
    {
        return m_KeyModifiers.Test<nn::hid::KeyboardModifier::NumLock>();
    }

    //! @}

    //! @name   キー属性情報の取得
    //! @{

    /**
     * @brief       指定されたキーの何れかが印字可能な文字を生成するか否かを表す値を返します。
     *
     * @param[in]   keys                        キーの指定です。
     *
     * @return      指定されたキーの何れかが印字可能な文字を生成するか否かを表す値です。
     */
    bool HasAnyPrintingKey(
        const nn::hid::KeyboardKeySet& keys) const NN_NOEXCEPT;

    //! @}

    //! @name   キーイベントの取得
    //! @{

    /**
     * @brief       キーイベントを返します。
     */
    const std::vector<KeyEvent>& GetKeyEvents() const
    {
        return m_KeyEvents;
    }

    //! @}

    //! @name   コントローラの設定
    //! @{

    /**
     * @brief       キーにリピートを設定します。
     *
     * @details     intervalCount に 0 を指定するとリピートは行われなくなります。
     *
     * @param[in]   keys                        キーの指定です。
     * @param[in]   delayCount                  リピートを開始するまでのフレーム数です。
     * @param[in]   intervalCount               リピートの間隔となるフレーム数です。
     */
    void SetKeysRepeatOption(
        const nn::hid::KeyboardKeySet& keys,
        uint8_t delayCount, uint8_t intervalCount) NN_NOEXCEPT;

    /**
     * @brief       キーのトグル状態を設定します。
     *
     * @details     ロックキーについては設定後もロック状態が優先されます。
     *
     * @param[in]   flags                       キーのトグル状態です。
     */
    void SetKeysToggle(const nn::hid::KeyboardKeySet& flags) NN_NOEXCEPT;

    /**
     * @brief       マウスボタンにダブルクリックを設定します。
     *
     * @details     waitCount に 0 を指定するとダブルクリックは検出されなくなります。
     *
     * @param[in]   buttons                     マウスボタンの指定です。
     * @param[in]   waitCount                   ダブルクリックの待ち受けフレーム数です。
     */
    void SetMouseButtonsDoubleClickOption(
        const nn::hid::MouseButtonSet& buttons, uint8_t waitCount) NN_NOEXCEPT;

    /**
     * @brief       マウスカーソルの絶対座標を使用するか否かを設定します。
     *
     * @param[in]   used                        マウスカーソルの絶対座標を使用するか否かを表す値です。
     */
    void SetMouseCursorAbsoluteCoordinateUsed(bool used) NN_NOEXCEPT
    {
        m_IsMouseCursorAbsoluteCoordinateUsed = used;
    }

    //! @}

    //! @name   コントローラの状態の取得
    //! @{

    /**
     * @brief       キーボードが接続状態にあるか否かを返します。
     *
     * @return      キーボードが接続状態にあるか否かを表す値です。
     */
    bool IsKeyboardConnected() const NN_NOEXCEPT
    {
        return m_IsKeyboardConnected;
    }

    /**
     * @brief       マウスが接続状態にあるか否かを返します。
     *
     * @return      マウスが接続状態にあるか否かを表す値です。
     */
    bool IsMouseConnected() const NN_NOEXCEPT
    {
        return m_IsMouseConnected;
    }

    //! @}

private:
    static const size_t KeyCountMax = 256;  //!< キーの数です。

    static const size_t MouseButtonCountMax = 32;   //!< マウスボタンの数です。

    static const uint8_t DefaultDoubleClickWaitCount = 25;  //!< ダブルクリックの待ち受けフレーム数のデフォルト値です。

    KeyboardAsset* m_pKeyboardAsset;    //!< 対応するキーボードのデバイスアセットです。

    bool m_IsKeyboardConnected; //!< キーボードが接続状態にあるか否かを表す値です。

    nn::hid::KeyboardKeySet m_Keys;         //!< 押下中のキーです。
    nn::hid::KeyboardKeySet m_KeysDown;     //!< 最後の更新で新たに押下されたキーです。
    nn::hid::KeyboardKeySet m_KeysUp;       //!< 最後の更新で新たに開放されたキーです。
    nn::hid::KeyboardKeySet m_KeysRepeat;   //!< リピートされたキーです。
    nn::hid::KeyboardKeySet m_KeysToggle;   //!< トグルされたキーです。

    uint32_t m_KeysDurationCounts[KeyCountMax];     //!< キーが押下中になってから経過したフレーム数です。

    uint8_t m_RepeatDelayCounts[KeyCountMax];       //!< リピートを開始するまでのフレーム数です。
    uint8_t m_RepeatIntervalCounts[KeyCountMax];    //!< リピートの間隔となるフレーム数です。

    nn::hid::KeyboardModifierSet m_KeyModifiers;    //!< キー修飾です。

    std::vector<KeyEvent> m_KeyEvents;  //!< キーイベントです。

    MouseAsset* m_pMouseAsset;  //!< 対応するマウスのデバイスアセットです。

    bool m_IsMouseConnected;    //!< マウスが接続状態にあるか否かを表す値です。

    bool m_IsMouseCursorAvailable;  //!< マウスのカーソル座標が利用可能か否かを表す値です。

    nn::hid::MouseButtonSet m_MouseButtons;     //!< 押下中のマウスボタンです。
    nn::hid::MouseButtonSet m_MouseButtonsDown; //!< 最後の更新で新たに押下されたマウスボタンです。
    nn::hid::MouseButtonSet m_MouseButtonsUp;   //!< 最後の更新で新たに開放されたマウスボタンです。
    nn::hid::MouseButtonSet m_MouseButtonsDoubleClick;  //!< ダブルクリックされたマウスボタンです。

    uint8_t m_MouseButtonsUpDurationCounts[MouseButtonCountMax];    //!< マウスボタンが開放されてから経過したフレーム数です。

    uint8_t m_DoubleClickWaitCounts[MouseButtonCountMax];   //!< ダブルクリックの待ち受けフレーム数です。

    bool m_IsMouseCursorAbsoluteCoordinateUsed; //!< マウスカーソルの絶対座標を使用するか否かを表す値です。

    nn::util::Float2 m_MouseCursor; //!< マウスのカーソル座標です。

    int32_t m_MouseWheel; //!< マウスのホイールの回転量です。

    /**
     * @brief       指定されたキーにリピートが必要か否かを返します。
     *
     * @param[in]   usageId                     キーの Usage ID です。
     *
     * @return      指定されたキーにリピートが必要か否かを表す値です。
     *
     * @pre
     *              - 0 <= usageId && usageId < KeyCountMax
     */
    bool NeedsKeyRepeat(int usageId) const NN_NOEXCEPT;

    /**
     * @brief       トグルされたキーを更新します。
     */
    void UpdateKeysToggle() NN_NOEXCEPT;

    /**
     * @brief       キーイベントを更新します。
     */
    void UpdateKeyEvents() NN_NOEXCEPT;

    /**
     * @brief       キーボードの状態を更新します。
     *
     * @param[out]  pOutButtons                 デジタルボタンの押下状態です。
     *
     * @pre
     *              - pOutButtons != NULL
     */
    void UpdateKeyboard(ButtonSet* pOutButtons) NN_NOEXCEPT;

    /**
     * @brief       ダブルクリックされたマウスボタンを更新します。
     */
    void UpdateMouseButtonsDoubleClick() NN_NOEXCEPT;

    /**
     * @brief       マウスの状態を更新します。
     */
    void UpdateMouse() NN_NOEXCEPT;
};

}} // namespace nns::hid
