﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d.h>
#include <nns/g3d/g3d_System.h>
#include <nns/g3d/g3d_RenderModel.h>
#include <nns/g3d/g3d_RenderModelObj.h>

namespace nns { namespace g3d {

//--------------------------------------------------------------------------------------------------
//! @brief モデルオブジェクトやアニメーションオブジェクトを管理するためのクラスです。
class ModelAnimObj
{
    NN_DISALLOW_COPY(ModelAnimObj);
public:
    //! @brief アニメーションの状態
    enum class AnimPlayState
    {
        Play, //!< 再生
        Stop //!< 停止
    };

    //! @brief アニメーションのタイプ
    enum class AnimType
    {
        Skeletal = 0x1, //!< スケルタルアニメーション
        Material, //!< マテリアルアニメーション
        Shape, //!< シェイプアニメーション
        BoneVisibility, //!< ボーンビジビリティーアニメーション
        Invalid //!< 無効なタイプ
    };

    //! @brief 無効なアニメーション id
    static const int InvalidAnimId = -1;

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    ModelAnimObj() :
        m_pModelObj(NULL),
        m_pBindModel(NULL),
        m_pSkeletalAnimList(NULL),
        m_pMaterialAnimList(NULL),
        m_pShapeAnimList(NULL),
        m_pBoneVisibilityAnimList(NULL),
        m_HandleSkeletalAnimCount(0),
        m_HandleMaterialAnimCount(0),
        m_HandleShapeAnimCount(0),
        m_HandleBoneVisibilityAnimCount(0),
        m_BindBoneIndex(-1),
        m_IsInitialized(false)
    {
        VectorSet(&m_Scale,     1.0f, 1.0f, 1.0f);
        VectorSet(&m_Rotate,    0.0f, 0.0f, 0.0f);
        VectorSet(&m_Translate, 0.0f, 0.0f, 0.0f);
        MatrixIdentity(&m_BaseMtx);
    }
    //! @brief ModelAnimObj を初期化します。
    //!
    //! @param[in] pModelObj nn::g3d::ModelObj へのポインター。
    //! @param[in] handleSkeletalAnimCount モデルで扱う最大スケルタルアニメーション数。
    //! @param[in] handleMaterialAnimCount モデルで扱う最大マテリアルアニメーション数。
    //! @param[in] handleShapeAnimCount モデルで扱う最大シェイプアニメーション数。
    //! @param[in] handleBoneVisibilityAnimCount モデルで扱う最大ボーンビジビリティーアニメーション数。
    //!
    //! @pre
    //! - ModelAnimObj が未初期化である。
    //!
    //! @post
    //! - ModelAnimObj が初期化されている。
    //!
    void Initialize(nn::g3d::ModelObj* pModelObj, int handleSkeletalAnimCount, int handleMaterialAnimCount, int handleShapeAnimCount, int handleBoneVisibilityAnimCount) NN_NOEXCEPT;

    //! @brief ModelAnimObj を破棄します。
    //!
    //! @pre
    //! - ModelAnimObj が初期化されている。
    //!
    //! @post
    //! - ModelAnimObj が未初期化である。
    //!
    void Finalize() NN_NOEXCEPT;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief nn::g3d::ModelObj を取得します。
    //!
    //! @return nn::g3d::ModelObj へのポインター。
    //!
    nn::g3d::ModelObj* GetModelObj() NN_NOEXCEPT
    {
        return m_pModelObj;
    }

    //! @brief nn::g3d::ModelObj を取得します。
    //!
    //! @return nn::g3d::ModelObj へのポインター。
    //!
    const nn::g3d::ModelObj* GetModelObj() const NN_NOEXCEPT
    {
        return m_pModelObj;
    }

    //! @brief モデルのバインド対象を設定します。
    //!
    //! @param[in] pModelObj バインド対象のモデル。
    //! @param[in] boneIndex バインド対象のボーンのインデックス。
    //!
    void BindTo(const nn::g3d::ModelObj* pModelObj, int boneIndex) NN_NOEXCEPT
    {
        m_pBindModel = pModelObj;
        m_BindBoneIndex = boneIndex;
    }

    //! @brief バインド対象に設定されているモデルを取得します。
    //!
    //! @return nn::g3d::ModelObj へのポインター。
    //!
    //! @details
    //! バインド対象にモデルが設定されていない場合には、NULL が返ります。
    //!
    const nn::g3d::ModelObj* GetBindModelObj() const NN_NOEXCEPT
    {
        return m_pBindModel;
    }

    //! @brief バインド対象に設定されているボーンのインデックスを取得します。
    //!
    //! @return ボーンのインデックス。
    //!
    //! @details
    //! バインド対象にモデルが設定されていない場合には、-1 が返ります。
    //!
    int GetBindBoneIndex() const NN_NOEXCEPT
    {
        return m_BindBoneIndex;
    }

    //! @brief nn::g3d::SkeletalAnimBlender を取得します。
    //!
    //! @return nn::g3d::SkeletalAnimBlender へのポインター。
    //!
    nn::g3d::SkeletalAnimBlender* GetSkeletalAnimBlender() NN_NOEXCEPT
    {
        return &m_SkeletalAnimBlender;
    }

    //! @brief nn::g3d::SkeletalAnimBlender を取得します。
    //!
    //! @return nn::g3d::SkeletalAnimBlender へのポインター。
    //!
    const nn::g3d::SkeletalAnimBlender* GetSkeletalAnimBlender() const NN_NOEXCEPT
    {
        return &m_SkeletalAnimBlender;
    }

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResSkeletalAnim nn::g3d::ResSkeletalAnim へのポインター。
    //! @param[in] state 作成直後のアニメーションの状態を指定。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    int CreateAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim, AnimPlayState state) NN_NOEXCEPT
    {
        nn::g3d::SkeletalAnimObj::BindArgument bindArg;
        bindArg.SetResource(m_pModelObj->GetResource()->GetSkeleton());
        return CreateAnim(pResSkeletalAnim, bindArg, state);
    }

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResSkeletalAnim nn::g3d::ResSkeletalAnim へのポインター。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    //! @details 作成直後のアニメーションの状態は再生状態になります。
    //!
    int CreateAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim) NN_NOEXCEPT
    {
        return CreateAnim(pResSkeletalAnim, AnimPlayState::Play);
    }

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResSkeletalAnim nn::g3d::ResSkeletalAnim へのポインター。
    //! @param[in] state 作成直後のアニメーションの状態を指定。
    //! @param[in] bindArgument nn:g3d::SkeletalAnimObj::Bind() に使用する BindArgument。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    int CreateAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim, const nn::g3d::SkeletalAnimObj::BindArgument& bindArgument, AnimPlayState state) NN_NOEXCEPT;

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResSkeletalAnim nn::g3d::ResSkeletalAnim へのポインター。
    //! @param[in] bindArgument nn:g3d::SkeletalAnimObj::Bind() に使用する BindArgument。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    //! @details 作成直後のアニメーションの状態は再生状態になります。
    //!
    int CreateAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim, const nn::g3d::SkeletalAnimObj::BindArgument& bindArgument) NN_NOEXCEPT
    {
        return CreateAnim(pResSkeletalAnim, bindArgument, AnimPlayState::Play);
    }

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResMaterialAnim nn::g3d::ResMaterialAnim へのポインター。
    //! @param[in] state 作成直後のアニメーションの状態を指定。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    int CreateAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim, AnimPlayState state) NN_NOEXCEPT;

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResMaterialAnim nn::g3d::ResMaterialAnim へのポインター。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    //! @details 作成直後のアニメーションの状態は再生状態になります。
    //!
    int CreateAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim) NN_NOEXCEPT
    {
        return CreateAnim(pResMaterialAnim, AnimPlayState::Play);
    }

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResShapeAnim nn::g3d::ResShapeAnim へのポインター。
    //! @param[in] state 作成直後のアニメーションの状態を指定。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    int CreateAnim(const nn::g3d::ResShapeAnim* pResShapeAnim, AnimPlayState state) NN_NOEXCEPT;

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResShapeAnim nn::g3d::ResShapeAnim へのポインター。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    //! @details 作成直後のアニメーションの状態は再生状態になります。
    //!
    int CreateAnim(const nn::g3d::ResShapeAnim* pResShapeAnim) NN_NOEXCEPT
    {
        return CreateAnim(pResShapeAnim, AnimPlayState::Play);
    }

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResBoneVisibilityAnim nn::g3d::ResBoneVisibilityAnim へのポインター。
    //! @param[in] state 作成直後のアニメーションの状態を指定。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    int CreateAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim, AnimPlayState state) NN_NOEXCEPT;

    //! @brief モデルに適用するアニメーションを作成します。
    //!
    //! @param[in] pResBoneVisibilityAnim nn::g3d::ResBoneVisibilityAnim へのポインター。
    //!
    //! @return 作成に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成したアニメーションを識別するための ID を返します。
    //!
    //! @details 作成直後のアニメーションの状態は再生状態になります。
    //!
    int CreateAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim) NN_NOEXCEPT
    {
        return CreateAnim(pResBoneVisibilityAnim, AnimPlayState::Play);
    }

    //! @brief モデルに適用されているアニメーションのインデックスを取得します。
    //!
    //! @param[in] pResSkeletalAnim モデルに適用されている nn::g3d::ResSkeletalAnim へのポインター。
    //!
    //! @return 取得に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成されているアニメーションを識別するための ID を返します。
    //!
    int FindAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim) const NN_NOEXCEPT;

    //! @brief モデルに適用されているアニメーションのインデックスを取得します。
    //!
    //! @param[in] pResMaterialAnim モデルに適用されている nn::g3d::ResMaterialAnim へのポインター。
    //!
    //! @return 取得に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成されているアニメーションを識別するための ID を返します。
    //!
    int FindAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim) const NN_NOEXCEPT;

    //! @brief モデルに適用されているアニメーションのインデックスを取得します。
    //!
    //! @param[in] pResShapeAnim モデルに適用されている nn::g3d::ResShapeAnim へのポインター。
    //!
    //! @return 取得に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成されているアニメーションを識別するための ID を返します。
    //!
    int FindAnim(const nn::g3d::ResShapeAnim* pResShapeAnim) const NN_NOEXCEPT;

    //! @brief モデルに適用されているアニメーションのインデックスを取得します。
    //!
    //! @param[in] pResBoneVisibilityAnim モデルに適用されている nn::g3d::ResBoneVisibilityAnim へのポインター。
    //!
    //! @return 取得に失敗した場合は nns::g3d::ModelAnimObj::InvalidAnimId、成功した場合は作成されているアニメーションを識別するための ID を返します。
    //!
    int FindAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim) const NN_NOEXCEPT;

    //! @brief 指定したアニメーションを破棄します。
    //!
    //! @param[in] id アニメーションの ID。
    //!
    //! @details
    //! id には CreateAnim() で返された値を指定します。
    //!
    void DestroyAnim(int id) NN_NOEXCEPT;

    //! @brief アニメーションの状態を取得します。
    //!
    //! @param[in] id アニメーションの ID。
    //!
    //! @return アニメーションの状態。
    //!
    //! @details
    //! id には CreateAnim() で返された値を指定します。
    //!
    AnimPlayState GetAnimState(int id) const NN_NOEXCEPT;

    //! @brief アニメーションの状態を設定します。
    //!
    //! @param[in] id アニメーションの ID。
    //! @param[in] state 新しいアニメーションの状態。
    //!
    //! @details
    //! id には CreateAnim() で返された値を指定します。
    //!
    void SetAnimState(int id, AnimPlayState state) NN_NOEXCEPT;

    //! @brief スケルタルアニメーションの重みを取得します。
    //!
    //! @param[out] pOutWeight スケルタルアニメーションの重みを格納する変数への参照。
    //! @param[in] id アニメーションの ID。
    //!
    //! @return 指定された id がスケルタルアニメーションの場合は true、 そうでない場合は false を返します。
    //!
    //! @details
    //! id には CreateAnim() で返された値を指定します。
    //!
    bool GetSkeletalAnimWeight(float* pOutWeight, int id) const NN_NOEXCEPT
    {
        AnimType type = GetAnimType(id);
        if (type == AnimType::Skeletal)
        {
            int index = GetAnimIndex(id);
            *pOutWeight = GetSkeletalAnimWeightImpl(index);
            return true;
        }
        return false;
    }

    //! @brief スケルタルアニメーションの重みを設定します。
    //!
    //! @param[in] id アニメーションの ID。
    //! @param[in] weight スケルタルアニメーションの重み。
    //!
    //! @return 指定された id がスケルタルアニメーションの場合は true、 そうでない場合は false を返します。
    //!
    //! @details
    //! id には CreateAnim() で返された値を指定します。
    //!
    bool SetSkeletalAnimWeight(int id, float weight) NN_NOEXCEPT
    {
        AnimType type = GetAnimType(id);
        if (type == AnimType::Skeletal)
        {
            int index = GetAnimIndex(id);
            SetSkeletalAnimWeightImpl(index, weight);
            return true;
        }
        return false;
    }

    //! @brief id に対応する nn::g3d::ModelAnimObj を取得します。
    //!
    //! @param[in] id アニメーションの ID。
    //!
    //! @return nn::g3d::ModelAnimObj へのポインター。
    //!
    //! @details
    //! id には CreateAnim() で返された値を指定します。
    //!
    nn::g3d::ModelAnimObj* GetAnimObj(int id) NN_NOEXCEPT;

    //! @brief id に対応する nn::g3d::ModelAnimObj を取得します。
    //!
    //! @param[in] id アニメーションの ID。
    //!
    //! @return nn::g3d::ModelAnimObj へのポインター。
    //!
    //! @details
    //! id には CreateAnim() で返された値を指定します。
    //!
    const nn::g3d::ModelAnimObj* GetAnimObj(int id) const NN_NOEXCEPT;

    //! @brief モデル行列のスケール成分を取得します。
    //!
    //! @return nn::util::Vector3fType への参照。
    //!
    const nn::util::Vector3fType& GetScale() const  NN_NOEXCEPT
    {
        return m_Scale;
    }

    //! @brief モデル行列のスケール成分を設定します。
    //!
    //! @param[in] scale nn::util::Vector3fType への参照。
    //!
    void SetScale(const nn::util::Vector3fType& scale) NN_NOEXCEPT
    {
        m_Scale = scale;
    }

    //! @brief モデル行列の回転成分を取得します。
    //!
    //! @return nn::util::Vector3fType への参照。
    //!
    const nn::util::Vector3fType& GetRotate() const  NN_NOEXCEPT
    {
        return m_Rotate;
    }

    //! @brief モデル行列の回転成分を設定します。
    //!
    //! @param[in] rotate nn::util::Vector3fType への参照。
    //!
    void SetRotate(const nn::util::Vector3fType& rotate) NN_NOEXCEPT
    {
        m_Rotate = rotate;
    }

    //! @brief モデル行列の移動成分を取得します。
    //!
    //! @return nn::util::Vector3fType への参照。
    //!
    const nn::util::Vector3fType& GetTranslate() const  NN_NOEXCEPT
    {
        return m_Translate;
    }

    //! @brief モデル行列の移動成分を設定します。
    //!
    //! @param[in] translate nn::util::Vector3fType への参照。
    //!
    void SetTranslate(const nn::util::Vector3fType& translate) NN_NOEXCEPT
    {
        m_Translate = translate;
    }

    //! @brief アニメーションの id からアニメーションタイプを取得します。
    //!
    //! @param[in] id アニメーションの ID。
    //!
    //! @return アニメーションの ID。
    //!
    static AnimType GetAnimType(int id) NN_NOEXCEPT
    {
        return static_cast<AnimType>(id >> AnimIndexBit);
    }

    //! @brief 初期化済みかを取得します。
    //!
    //! @return 初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    //! @brief モデルにバインド対象が設定されているかを取得します
    //!
    //! @return バインド対象が設定されている場合は true、設定されていない場合は false を返します。
    //!
    bool IsBound() const NN_NOEXCEPT
    {
        return m_pBindModel && m_BindBoneIndex >= 0;
    }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief ビューに関連する計算を行います。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //! @param[in] viewMtx ビュー行列への参照。
    //!
    //! @details
    //! リジッドボディかつビューに依存するシェイプに対して計算を行います。
    //! シェイプのユニフォームブロックに計算結果は反映されます。
    //!
    void CalculateView(int viewIndex, int bufferIndex, const nn::util::Matrix4x3fType& viewMtx) NN_NOEXCEPT;

    //! @brief アニメーションの更新を行います。
    void CalculateAnimations() NN_NOEXCEPT;

    //! @brief ワールド変換行列の更新処理を行います。
    void CalculateWorld() NN_NOEXCEPT;

    //! @brief バウンディングの更新処理を行います。
    void CalculateBounding() NN_NOEXCEPT;

    //! @brief GPU処理に影響しないモデルおよびアニメーションの更新処理を行います。
    //!
    //! @details
    //! この関数は以下の関数呼び出しを行うことに相当します。
    //! CalculateAnimations()
    //! CalculateWorld()
    //! CalculateBounding()
    //!
    //! ビューアーライブラリによってアニメーションが更新された後にこの関数を呼び出すと、
    //! クラス内で管理しているアニメーションで上書きされるため、注意する必要があります。
    //!
    void Calculate() NN_NOEXCEPT;

    //! @brief GPUが参照するユニフォームブロックを更新します。
    //!
    //! @param[in] bufferIndex 更新対象のバッファーインデックス。
    //!
    //! @details
    //! マテリアル、シェイプ、スケルトンのユニフォームブロックを更新します。
    //!
    void CalculateUniformBlock(int bufferIndex) NN_NOEXCEPT;

private:
    static const int AnimTypeMask  = 0xFF000000;
    static const int AnimIndexMask = 0x00FFFFFF;
    static const int AnimIndexBit = 24;
    static const int InvalidAnimIndex = -1;

    //! @brief アニメーションの id を生成します。
    //!
    //! @param[in] type アニメーションのタイプ。
    //! @param[in] index アニメーションのインデックス。
    //!
    //! @return 生成したアニメーション ID。
    //!
    int CreateAnimId(AnimType type, int index) const NN_NOEXCEPT
    {
        return (static_cast<int>(type) << AnimIndexBit | index);
    }

    //! @brief アニメーションの id からアニメーションのインデックスを取得します。
    //!
    //! @param[in] id アニメーションの ID。
    //!
    //! @return アニメーションのインデックス。
    //!
    int GetAnimIndex(int id) const NN_NOEXCEPT
    {
        return id & AnimIndexMask;
    }

    //! @brief アニメーションのタイプを識別し、返します。
    //!
    //! @param[in] pResAnim アニメーションリソースへのポインター。
    //!
    //! @return 識別されたアニメーションタイプ。識別出来ない場合、 nns::g3d::AnimType::Invalid を返します。
    //!
    AnimType GetAnimType(void* pResAnim) NN_NOEXCEPT;

    //! @brief モデルに適用するスケルタルアニメーションを作成します。
    //!
    //! @param[in] pResSkeletalAnim モデルに適用する nn::g3d::ResSkeletalAnim へのポインター。
    //! @param[in] bindArgument nn:g3d::SkeletalAnimObj::Bind() に使用する BindArgument。
    //! @param[in] weight スケルタルアニメブレンド時の重み。
    //!
    //! @return 作成したスケルタルアニメーションを識別するためのインデックス。
    //!
    int CreateSkeletalAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim, const nn::g3d::SkeletalAnimObj::BindArgument& bindArgument, float weight) NN_NOEXCEPT;

    //! @brief 指定したスケルタルアニメーションを破棄します。
    //!
    //! @param[in] index スケルタルアニメーションを識別するためのインデックス。
    //!
    //! @details
    //! index には CreateSkeletalAnim() で返された値を指定します。
    //!
    void DestroySkeletalAnim(int index) NN_NOEXCEPT;

    //! @brief モデルに適用されているスケルタルアニメーションのインデックスを取得します。
    //!
    //! @param[in] pResSkeletalAnim モデルに適用されている nn::g3d::ResSkeletalAnim のポインター。
    //!
    //! @return モデルに適用されているスケルタルアニメーションのインデックス。
    //!
    //! @details
    //! pResSkeletalAnim がモデルに適用されていなかった場合、-1 が返ります。
    //!
    int FindSkeletalAnim(const nn::g3d::ResSkeletalAnim* pResSkeletalAnim) const NN_NOEXCEPT;

    //! @brief スケルタルアニメーションの状態を取得します。
    //!
    //! @param[in] index スケルタルアニメーションを識別するためのインデックス。
    //!
    //! @return スケルタルアニメーションの状態。
    //!
    //! @details
    //! index には CreateSkeletalAnim() で返された値を指定します。
    //!
    AnimPlayState GetSkeletalAnimState(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleSkeletalAnimCount);
        NN_ASSERT(m_pSkeletalAnimList[index].IsInitialized());
        return m_pSkeletalAnimList[index].GetState();
    }

    //! @brief スケルタルアニメーションの状態を設定します。
    //!
    //! @param[in] index スケルタルアニメーションを識別するためのインデックス。
    //! @param[in] state 新しいスケルタルアニメーションの状態。
    //!
    //! @details
    //! index には CreateSkeletalAnim() で返された値を指定します。
    //!
    void SetSkeletalAnimState(int index, AnimPlayState state) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleSkeletalAnimCount);
        NN_ASSERT(m_pSkeletalAnimList[index].IsInitialized());
        m_pSkeletalAnimList[index].SetState(state);
    }

    //! @brief スケルタルアニメーションの重みを取得します。
    //!
    //! @param[in] index スケルタルアニメーションを識別するためのインデックス。
    //!
    //! @return スケルタルアニメーションの重み。
    //!
    //! @details
    //! index には CreateSkeletalAnim() で返された値を指定します。
    //!
    float GetSkeletalAnimWeightImpl(int index) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(index, 0, m_HandleSkeletalAnimCount);
        NN_ASSERT(m_pSkeletalAnimList[index].IsInitialized());
        return m_pSkeletalAnimList[index].GetWeight();
    }

    //! @brief スケルタルアニメーションの重みを設定します。
    //!
    //! @param[in] index スケルタルアニメーションを識別するためのインデックス。
    //! @param[in] weight スケルタルアニメーションの重み。
    //!
    //! @details
    //! index には CreateSkeletalAnim() で返された値を指定します。
    //!
    void SetSkeletalAnimWeightImpl(int index, float weight) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(index, 0, m_HandleSkeletalAnimCount);
        NN_ASSERT(m_pSkeletalAnimList[index].IsInitialized());
        m_pSkeletalAnimList[index].SetWeight(weight);
    }

    //! @brief nn::g3d::SkeletalAnimObj を取得します。
    //!
    //! @return nn::g3d::SkeletalAnimObj へのポインター。
    //!
    //! @details
    //! index には CreateSkeletalAnim() で返された値を指定します。
    //!
    nn::g3d::SkeletalAnimObj* GetSkeletalAnimObj(int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleSkeletalAnimCount);
        NN_ASSERT(m_pSkeletalAnimList[index].IsInitialized());
        return m_pSkeletalAnimList[index].GetAnimObj();
    }

    //! @brief nn::g3d::SkeletalAnimObj を取得します。
    //!
    //! @return nn::g3d::SkeletalAnimObj へのポインター。
    //!
    //! @details
    //! index には CreateSkeletalAnim() で返された値を指定します。
    //!
    const nn::g3d::SkeletalAnimObj* GetSkeletalAnimObj(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleSkeletalAnimCount);
        NN_ASSERT(m_pSkeletalAnimList[index].IsInitialized());
        return m_pSkeletalAnimList[index].GetAnimObj();

    }
    //! @brief モデルに適用するマテリアルアニメーションを作成します。
    //!
    //! @param[in] pResMaterialAnim モデルに適用する nn::g3d::ResMaterialAnim へのポインター。
    //!
    //! @return 作成したマテリアルアニメーションを識別するためのインデックス。
    //!
    int CreateMaterialAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim) NN_NOEXCEPT;

    //! @brief 指定したマテリアルアニメーションを破棄します。
    //!
    //! @param[in] index マテリアルアニメーションを識別するためのインデックス。
    //!
    //! @details
    //! index には CreateMaterialAnim() で返された値を指定します。
    //!
    void DestroyMaterialAnim(int index) NN_NOEXCEPT;

    //! @brief モデルに適用されているマテリアルアニメーションのインデックスを取得します。
    //!
    //! @param[in] pResMaterialAnim モデルに適用されている nn::g3d::ResMaterialAnim のポインター。
    //!
    //! @return モデルに適用されているマテリアルアニメーションのインデックス。
    //!
    //! @details
    //! pResMaterialAnim がモデルに適用されていなかった場合、-1 が返ります。
    //!
    int FindMaterialAnim(const nn::g3d::ResMaterialAnim* pResMaterialAnim) const NN_NOEXCEPT;

    //! @brief マテリアルアニメーションの状態を取得します。
    //!
    //! @param[in] index マテリアルアニメーションを識別するためのインデックス。
    //!
    //! @return マテリアルアニメーションの状態。
    //!
    //! @details
    //! index には CreateMaterialAnim() で返された値を指定します。
    //!
    AnimPlayState GetMaterialAnimState(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleMaterialAnimCount);
        NN_ASSERT(m_pMaterialAnimList[index].IsInitialized());
        return m_pMaterialAnimList[index].GetState();
    }

    //! @brief マテリアルアニメーションの状態を設定します。
    //!
    //! @param[in] index マテリアルアニメーションを識別するためのインデックス。
    //! @param[in] state 新しいマテリアルアニメーションの状態。
    //!
    //! @details
    //! index には CreateMaterialAnim() で返された値を指定します。
    //!
    void SetMaterialAnimState(int index, AnimPlayState state) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleMaterialAnimCount);
        NN_ASSERT(m_pMaterialAnimList[index].IsInitialized());
        m_pMaterialAnimList[index].SetState(state);
    }

    //! @brief nn::g3d::MaterialAnimObj を取得します。
    //!
    //! @return nn::g3d::MaterialAnimObj へのポインター。
    //!
    //! @details
    //! index には CreateMaterialAnim() で返された値を指定します。
    //!
    nn::g3d::MaterialAnimObj* GetMaterialAnimObj(int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleMaterialAnimCount);
        NN_ASSERT(m_pMaterialAnimList[index].IsInitialized());
        return m_pMaterialAnimList[index].GetAnimObj();
    }

    //! @brief nn::g3d::MaterialAnimObj を取得します。
    //!
    //! @return nn::g3d::MaterialAnimObj へのポインター。
    //!
    //! @details
    //! index には CreateMaterialAnim() で返された値を指定します。
    //!
    const nn::g3d::MaterialAnimObj* GetMaterialAnimObj(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleMaterialAnimCount);
        NN_ASSERT(m_pMaterialAnimList[index].IsInitialized());
        return m_pMaterialAnimList[index].GetAnimObj();
    }

    //! @brief モデルに適用するシェイプアニメーションを作成します。
    //!
    //! @param[in] pResShapeAnim モデルに適用する nn::g3d::ResShapeAnim へのポインター。
    //!
    //! @return 追加したシェイプアニメーションを識別するためのインデックス。
    //!
    int CreateShapeAnim(const nn::g3d::ResShapeAnim* pResShapeAnim) NN_NOEXCEPT;

    //! @brief 指定したシェイプアニメーションを破棄します。
    //!
    //! @param[in] index シェイプアニメーションを識別するためのインデックス。
    //!
    //! @details
    //! index には CreateShapeAnim() で返された値を指定します。
    //!
    void DestroyShapeAnim(int index) NN_NOEXCEPT;

    //! @brief モデルに適用されているシェイプアニメーションのインデックスを取得します。
    //!
    //! @param[in] pResShapeAnim モデルに適用されている nn::g3d::ResShapeAnim のポインター。
    //!
    //! @return モデルに適用されているシェイプアニメーションのインデックス。
    //!
    //! @details
    //! pResShapeAnim がモデルに適用されていなかった場合、-1 が返ります。
    //!
    int FindShapeAnim(const nn::g3d::ResShapeAnim* pResShapeAnim) const NN_NOEXCEPT;

    //! @brief シェイプアニメーションの状態を取得します。
    //!
    //! @param[in] index シェイプアニメーションを識別するためのインデックス。
    //!
    //! @return シェイプアニメーションの状態。
    //!
    //! @details
    //! index には CreateShapeAnim() で返された値を指定します。
    //!
    AnimPlayState GetShapeAnimState(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleShapeAnimCount);
        NN_ASSERT(m_pShapeAnimList[index].IsInitialized());
        return m_pShapeAnimList[index].GetState();
    }

    //! @brief シェイプアニメーションの状態を設定します。
    //!
    //! @param[in] index シェイプアニメーションを識別するためのインデックス。
    //! @param[in] state 新しいシェイプアニメーションの状態。
    //!
    //! @details
    //! index には CreateShapeAnim() で返された値を指定します。
    //!
    void SetShapeAnimState(int index, AnimPlayState state) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleShapeAnimCount);
        NN_ASSERT(m_pShapeAnimList[index].IsInitialized());
        m_pShapeAnimList[index].SetState(state);
    }

    //! @brief nn::g3d::ShapeAnimObj を取得します。
    //!
    //! @return nn::g3d::ShapeAnimObj へのポインター。
    //!
    //! @details
    //! index には CreateShapeAnim() で返された値を指定します。
    //!
    nn::g3d::ShapeAnimObj* GetShapeAnimObj(int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleShapeAnimCount);
        NN_ASSERT(m_pShapeAnimList[index].IsInitialized());
        return m_pShapeAnimList[index].GetAnimObj();
    }

    //! @brief nn::g3d::ShapeAnimObj を取得します。
    //!
    //! @return nn::g3d::ShapeAnimObj へのポインター。
    //!
    //! @details
    //! index には CreateShapeAnim() で返された値を指定します。
    //!
    const nn::g3d::ShapeAnimObj* GetShapeAnimObj(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleShapeAnimCount);
        NN_ASSERT(m_pShapeAnimList[index].IsInitialized());
        return m_pShapeAnimList[index].GetAnimObj();
    }

    //! @brief モデルに適用するボーンビジビリティーアニメーションを作成します。
    //!
    //! @param[in] pResBoneVisibilityAnim モデルに適用する nn::g3d::ResBoneVisibilityAnim へのポインター。
    //!
    //! @return 追加したボーンビジビリティーアニメーションを識別するためのインデックス。
    //!
    int CreateBoneVisibilityAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim) NN_NOEXCEPT;

    //! @brief 指定したボーンビジビリティーアニメーションを破棄します。
    //!
    //! @param[in] index ボーンビジビリティーアニメーションを識別するためのインデックス。
    //!
    //! @details
    //! index には CreateBoneVisibilityAnim() で返された値を指定します。
    //!
    void DestroyBoneVisibilityAnim(int index) NN_NOEXCEPT;

    //! @brief モデルに適用されているボーンビジビリティーアニメーションのインデックスを取得します。
    //!
    //! @param[in] pResBoneVisibilityAnim モデルに適用されている nn::g3d::ResBoneVisibilityAnim のポインター。
    //!
    //! @return モデルに適用されているボーンビジビリティーアニメーションのインデックス。
    //!
    //! @details
    //! pResBoneVisibilityAnim がモデルに適用されていなかった場合、-1 が返ります。
    //!
    int FindBoneVisibilityAnim(const nn::g3d::ResBoneVisibilityAnim* pResBoneVisibilityAnim) const NN_NOEXCEPT;


    //! @brief ボーンビジビリティーアニメーションの状態を取得します。
    //!
    //! @param[in] index ボーンビジビリティーアニメーションを識別するためのインデックス。
    //!
    //! @return ボーンビジビリティーアニメーションの状態。
    //!
    //! @details
    //! index には CreateBoneVisibilityAnim() で返された値を指定します。
    //!
    AnimPlayState GetBoneVisibilityAnimState(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleBoneVisibilityAnimCount);
        NN_ASSERT(m_pBoneVisibilityAnimList[index].IsInitialized());
        return m_pBoneVisibilityAnimList[index].GetState();
    }

    //! @brief ボーンビジビリティーアニメーションの状態を設定します。
    //!
    //! @param[in] index ボーンビジビリティーアニメーションを識別するためのインデックス。
    //! @param[in] state 新しいボーンビジビリティーアニメーションの状態。
    //!
    //! @details
    //! index には CreateBoneVisibilityAnim() で返された値を指定します。
    //!
    void SetBoneVisibilityAnimState(int index, AnimPlayState state) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleBoneVisibilityAnimCount);
        NN_ASSERT(m_pBoneVisibilityAnimList[index].IsInitialized());
        m_pBoneVisibilityAnimList[index].SetState(state);
    }

    //! @brief nn::g3d::BoneVisibilityAnimObj を取得します。
    //!
    //! @return nn::g3d::BoneVisibilityAnimObj へのポインター。
    //!
    //! @details
    //! index には CreateBoneVisibilityAnim() で返された値を指定します。
    //!
    nn::g3d::BoneVisibilityAnimObj* GetBoneVisibilityAnimObj(int index) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleBoneVisibilityAnimCount);
        NN_ASSERT(m_pBoneVisibilityAnimList[index].IsInitialized());
        return m_pBoneVisibilityAnimList[index].GetAnimObj();
    }

    //! @brief nn::g3d::BoneVisibilityAnimObj を取得します。
    //!
    //! @return nn::g3d::BoneVisibilityAnimObj へのポインター。
    //!
    //! @details
    //! index には CreateBoneVisibilityAnim() で返された値を指定します。
    //!
    const nn::g3d::BoneVisibilityAnimObj* GetBoneVisibilityAnimObj(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_HandleBoneVisibilityAnimCount);
        NN_ASSERT(m_pBoneVisibilityAnimList[index].IsInitialized());
        return m_pBoneVisibilityAnimList[index].GetAnimObj();
    }

    template <typename T>
    class Anim
    {
    public:
        Anim()
        {
            Finalize();
        }

        void Initialize(T* pAnimObj) NN_NOEXCEPT
        {
            if (!m_IsInitialized)
            {
                m_pAnimObj = pAnimObj;
                m_IsInitialized = true;
            }
        }

        void Finalize() NN_NOEXCEPT
        {
            if (m_IsInitialized)
            {
                m_pAnimObj = NULL;
                m_IsInitialized = false;
                m_PlayState = AnimPlayState::Stop;
            }
        }

        T* GetAnimObj() NN_NOEXCEPT
        {
            return m_pAnimObj;
        }

        const T* GetAnimObj() const NN_NOEXCEPT
        {
            return m_pAnimObj;
        }

        AnimPlayState GetState() const NN_NOEXCEPT
        {
            return m_PlayState;
        }

        void SetState(AnimPlayState playState) NN_NOEXCEPT
        {
            m_PlayState = playState;
        }

        int IsInitialized() const NN_NOEXCEPT
        {
            return m_IsInitialized;
        }

    private:
        T*            m_pAnimObj;
        AnimPlayState m_PlayState;
        bool          m_IsInitialized;
    };

    class SkeletalAnim
    {
    public:
        SkeletalAnim()
        {
            Finalize();
        }

        void Initialize(nn::g3d::SkeletalAnimObj* pAnimObj, float weight) NN_NOEXCEPT
        {
            if (!m_Anim.IsInitialized())
            {
                m_Anim.Initialize(pAnimObj);
                m_Weight = weight;
            }
        }

        void Finalize() NN_NOEXCEPT
        {
            if (m_Anim.IsInitialized())
            {
                m_Anim.Finalize();
                m_Weight = 0.0f;
            }
        }

        nn::g3d::SkeletalAnimObj* GetAnimObj() NN_NOEXCEPT
        {
            return m_Anim.GetAnimObj();
        }

        const nn::g3d::SkeletalAnimObj* GetAnimObj() const NN_NOEXCEPT
        {
            return m_Anim.GetAnimObj();
        }

        AnimPlayState GetState() const NN_NOEXCEPT
        {
            return m_Anim.GetState();
        }

        void SetState(AnimPlayState playState) NN_NOEXCEPT
        {
            m_Anim.SetState(playState);
        }

        int IsInitialized() const NN_NOEXCEPT
        {
            return m_Anim.IsInitialized();
        }

        float GetWeight() const NN_NOEXCEPT
        {
            return m_Weight;
        }

        void SetWeight(float weight) NN_NOEXCEPT
        {
            m_Weight = weight;
        }
    private:
        Anim<nn::g3d::SkeletalAnimObj> m_Anim;
        float                          m_Weight;
    };

    typedef Anim<nn::g3d::MaterialAnimObj> MaterialAnim;
    typedef Anim<nn::g3d::ShapeAnimObj> ShapeAnim;
    typedef Anim<nn::g3d::BoneVisibilityAnimObj> BoneVisibilityAnim;

    nn::util::Vector3fType          m_Scale;
    nn::util::Vector3fType          m_Rotate;
    nn::util::Vector3fType          m_Translate;
    nn::util::Matrix4x3fType        m_BaseMtx;
    nn::g3d::SkeletalAnimBlender    m_SkeletalAnimBlender;
    nn::g3d::ModelObj*              m_pModelObj;
    const nn::g3d::ModelObj*        m_pBindModel;
    SkeletalAnim*                   m_pSkeletalAnimList;
    MaterialAnim*                   m_pMaterialAnimList;
    ShapeAnim*                      m_pShapeAnimList;
    BoneVisibilityAnim*             m_pBoneVisibilityAnimList;
    int                             m_HandleSkeletalAnimCount;
    int                             m_HandleMaterialAnimCount;
    int                             m_HandleShapeAnimCount;
    int                             m_HandleBoneVisibilityAnimCount;
    int                             m_BindBoneIndex;
    bool                            m_IsInitialized;
};

//--------------------------------------------------------------------------------------------------
//! @brief ビュー管理のためのユーティリティクラス
class RenderView
{
    NN_DISALLOW_COPY(RenderView);
public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{
    RenderView() NN_NOEXCEPT :
        m_pView(NULL),
        m_pViewUniformBlock(NULL),
        m_pViewVolume(NULL),
        m_pViewUniformBlockOffset(NULL),
        m_ViewCount(0),
        m_BufferCount(0)
    {
    }

    //! @brief RenderView を初期化します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] viewCount ビュー数。
    //! @param[in] bufferCount バッファー数。
    //!
    //! @pre
    //! - RenderView が未初期化である。
    //!
    //! @post
    //! - RenderView が初期化されている。
    //!
    void Initialize(nn::gfx::Device* pDevice, int viewCount, int bufferCount) NN_NOEXCEPT;

    //! @brief RenderView を破棄します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //!
    //! @pre
    //! - RenderView が初期化されている。
    //!
    //! @post
    //! - RenderView が未初期化である。
    //!
    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief プロジェクション行列を取得します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //!
    //! @return const nn::util::Matrix4x4fType への参照。
    //!
    const nn::util::Matrix4x4fType& GetProjectionMtx(int viewIndex) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        return m_pView[viewIndex].projectionMtx;
    }

    //! @brief プロジェクション行列を設定します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] projectionMtx 設定するプロジェクション行列。
    //!
    void SetProjectionMtx(int viewIndex, const nn::util::Matrix4x4fType& projectionMtx) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        m_pView[viewIndex].projectionMtx = projectionMtx;
    }

    //! @brief ビュー行列を取得します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //!
    //! @return const nn::util::Matrix4x3fType への参照。
    //!
    const nn::util::Matrix4x3fType& GetViewMtx(int viewIndex) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        return m_pView[viewIndex].viewMtx;
    }

    //! @brief ビュー行列を設定します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] viewMtx 設定するビュー行列。
    //!
    void SetViewMtx(int viewIndex, const nn::util::Matrix4x3fType& viewMtx) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        m_pView[viewIndex].viewMtx = viewMtx;
    }

    //! @brief ビューボリュームを取得します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //!
    //! @return nn::g3d::ViewVolume への参照。
    //!
    nn::g3d::ViewVolume& GetViewVolume(int viewIndex) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        return m_pViewVolume[viewIndex];
    }

    //! @brief ビューボリュームを取得します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //!
    //! @return const nn::g3d::ViewVolume への参照。
    //!
    const nn::g3d::ViewVolume& GetViewVolume(int viewIndex) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        return m_pViewVolume[viewIndex];
    }

    //! @brief ビュー用ユニフォームブロックの nn::gfx::Buffer オブジェクトを取得します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //!
    //! @return const nn::gfx::Buffer へのポインター。
    //!
    const nn::gfx::Buffer* GetViewUniformBlock(int viewIndex, int bufferIndex) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        NN_ASSERT_RANGE(bufferIndex, 0, m_BufferCount);
        return &m_pViewUniformBlock[m_BufferCount * viewIndex + bufferIndex];
    }

    //! @brief ビュー用ユニフォームブロックのサイズを取得します。
    //!
    //! @return ビュー用ユニフォームブロックのサイズ。
    //!
    size_t GetViewUniformBlockSize() const NN_NOEXCEPT
    {
        return sizeof(ViewUniformBlock);
    }

    //! @brief ビューの逆行列を取得します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //!
    //! @return nn::g3d::Matrix4x3fType への参照。
    //!
    nn::util::Matrix4x3fType& GetInverseViewMtx(int viewIndex) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        return m_pView[viewIndex].inverseViewMtx;
    }

    //! @brief ビューの逆行列を取得します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //!
    //! @return const nn::g3d::Matrix4x3fType への参照。
    //!
    const nn::util::Matrix4x3fType& GetInverseViewMtx(int viewIndex) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(viewIndex, 0, m_ViewCount);
        return m_pView[viewIndex].inverseViewMtx;
    }

    //! @brief 更新を行います。
    //!
    //! @param[in] bufferIndex バッファーインデックス。
    //!
    //! @details
    //! プロジェクション行列、ビュー行列、ビュー逆行列をバッファーインデックスで指定された
    //! ビュー用ユニフォームブロックの nn::gfx::Buffer に反映します。
    //!
    void Calculate(int bufferIndex) NN_NOEXCEPT
    {
        for (int viewIndex = 0; viewIndex < m_ViewCount; ++viewIndex)
        {
            Calculate(viewIndex, bufferIndex);
        }
    }

    //! @brief 更新を行います。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //!
    //! @details
    //! ビューインデックスで指定されたビューのプロジェクション行列、ビュー行列、ビュー逆行列をバッファーインデックスで指定された
    //! ビュー用ユニフォームブロックの nn::gfx::Buffer に反映します。
    //!
    void Calculate(int viewIndex, int bufferIndex) NN_NOEXCEPT;

    //! @brief ビュー数を取得します。
    //!
    //! @return ビュー数。
    //!
    int GetViewCount() const NN_NOEXCEPT
    {
        return m_ViewCount;
    }

private:
    enum MemoryBlock
    {
        MemoryBlock_View,
        MemoryBlock_ViewUniformBlock,
        MemoryBlock_ViewVolume,
        MemoryBlock_ViewUniformBlockOffset,
        MemoryBlock_Max,
    };
    struct View
    {
        nn::util::Matrix4x4fType projectionMtx;
        nn::util::Matrix4x3fType viewMtx;
        nn::util::Matrix4x3fType inverseViewMtx;
    };

    struct ViewUniformBlock
    {
        nn::util::FloatColumnMajor4x4 projectionMtx;
        nn::util::FloatColumnMajor4x3 viewMtx;
        nn::util::FloatColumnMajor4x3 inverseViewMtx;
    };

    View*                m_pView;
    nn::gfx::Buffer*     m_pViewUniformBlock;
    nn::g3d::ViewVolume* m_pViewVolume;
    ptrdiff_t*           m_pViewUniformBlockOffset;
    int                  m_ViewCount;
    int                  m_BufferCount;
};

//! @brief nn::g3d::ModelObj を作成し、返します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] builder nn::g3d::ModelObj::Builder への参照。
//!
//! @return nn::g3d::ModelObj へのポインター。
//!
//! @details
//! 指定された builder で nn::g3d::ModelObj を作成し、返します。
//!
nn::g3d::ModelObj* CreateModelObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj::Builder& builder) NN_NOEXCEPT;

//! @brief CreateModelObj() で作成した nn::g3d::ModelObj を破棄します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] pModelObj 破棄する対象の nn::g3d::ModelObj へのポインター。
//!
void DestroyModelObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj* pModelObj) NN_NOEXCEPT;

//! @brief nn::g3d::ModelAnimObj を作成し、返します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] builder nn::g3d::ModelObj::Builder への参照。
//!
//! @return nn::g3d::ModelAnimObj へのポインター。
//!
//! @details
//! 指定された builder で nn::g3d::ModelAnimObj を作成し、返します。
//!
nns::g3d::ModelAnimObj* CreateModelAnimObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj::Builder& builder) NN_NOEXCEPT;

//! @brief nn::g3d::ModelAnimObj を作成し、返します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] builder nn::g3d::ModelObj::Builder への参照。
//! @param[in] handleSkeletalAnimCount 扱う最大スケルタルアニメ数。
//! @param[in] handleMaterialAnimCount 扱う最大マテリアルアニメ数。
//! @param[in] handleShapeAnimCount 扱う最大シェイプアニメ数。
//! @param[in] handleBoneVisibilityAnimCount 扱う最大ボーンビジビリティーアニメ数。
//!
//! @return nn::g3d::ModelAnimObj へのポインター。
//!
//! @details
//! 指定された builder で nn::g3d::ModelAnimObj を作成し、返します。
//!
nns::g3d::ModelAnimObj* CreateModelAnimObj(nn::gfx::Device* pDevice, nn::g3d::ModelObj::Builder& builder, int handleSkeletalAnimCount, int handleMaterialAnimCount, int handleShapeAnimCount, int handleBoneVisibilityAnimCount) NN_NOEXCEPT;

//! @brief CreateModelAnimObj() で作成した nn::g3d::ModelAnimObj を破棄します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] pModelAnimObj 破棄する対象の nn::g3d::ModelAnimObj へのポインター。
//!
void DestroyModelAnimObj(nn::gfx::Device* pDevice, nns::g3d::ModelAnimObj* pModelAnimObj) NN_NOEXCEPT;

//! @brief nn::g3d::ShadingModelObj を作成し、返します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] builder nn::g3d::ShadingModelObj::Builder への参照。
//!
//! @return nn::g3d::ShadingModelObj へのポインター。
//!
//! @details
//! 指定された builder で nn::g3d::ShadingModelObj を作成し、返します。
//!
nn::g3d::ShadingModelObj* CreateShadingModelObj(nn::gfx::Device* pDevice, nn::g3d::ShadingModelObj::Builder& builder) NN_NOEXCEPT;

//! @brief CreateShadingModelObj() で作成した nn::g3d::ShadingModelObj を破棄します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] pShadingModelObj 破棄する対象の nn::g3d::ShadingModelObj へのポインター。
//!
void DestroyShadingModelObj(nn::gfx::Device* pDevice, nn::g3d::ShadingModelObj* pShadingModelObj) NN_NOEXCEPT;

//! @brief nn::g3d::ShaderSelector を作成し、返します。
//!
//! @param[in] builder nn::g3d::ShaderSelector::Builder への参照。
//!
//! @return nn::g3d::ShaderSelector へのポインター。
//!
//! @details
//! 指定された builder で nn::g3d::ShaderSelector を作成し、返します。
//!
nn::g3d::ShaderSelector* CreateShaderSelector(nn::g3d::ShaderSelector::Builder& builder) NN_NOEXCEPT;

//! @brief CreateShaderSelector() で作成した nn::g3d::ShaderSelector を破棄します。
//!
//! @param[in] pShaderSelector 破棄する対象の nn::g3d::ShaderSelector へのポインター。
//!
void DestroyShaderSelector(nn::g3d::ShaderSelector* pShaderSelector) NN_NOEXCEPT;

//! @brief nns::g3d::RenderModel を作成し、返します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] pResModel nn::g3d::ResModel へのポインター。
//! @param[in] passCount 描画パスの数。
//!
//! @return nns::g3d::RenderModel へのポインター。
//!
RenderModel* CreateRenderModel(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel, int passCount) NN_NOEXCEPT;

//! @brief nns::g3d::RenderModel を作成し、返します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] pResModel nn::g3d::ResModel へのポインター。
//!
//! @return nns::g3d::RenderModel へのポインター。
//!
//! @details
//! 描画パスの数は1つとして設定し作成します。
//!
RenderModel* CreateRenderModel(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel) NN_NOEXCEPT;

//! @brief CreateRenderModel() で作成した nns::g3d::RenderModel を破棄します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] pRenderModel 破棄する対象の nns::g3d::RenderModel へのポインター。
//!
void DestroyRenderModel(nn::gfx::Device* pDevice, RenderModel* pRenderModel) NN_NOEXCEPT;

//! @brief nns::g3d::RenderModelObj を作成し、返します。
//!
//! @param[in] pModelObj nn::g3d::ModelObj へのポインター。
//! @param[in] pRenderModel nns::g3d::RenderModel へのポインター。
//!
//! @return nns::g3d::RenderModelObj へのポインター。
//!
RenderModelObj* CreateRenderModelObj(nn::g3d::ModelObj* pModelObj, RenderModel* pRenderModel) NN_NOEXCEPT;

//! @brief CreateRenderModelObj() で作成した nns::g3d::RenderModelObj を破棄します。
//!
//! @param[in] pRenderModelObj 破棄する対象の nns::g3d::RenderModelObj へのポインター。
//!
void DestroyRenderModelObj(RenderModelObj* pRenderModelObj) NN_NOEXCEPT;

//! @brief アニメーションオブジェクトを作成し、返します。
//!
//! @tparam T アニメーションの型。
//! @tparam U T 型のアニメーションオブジェクトを生成するための Builder 型。デフォルトテンプレート引数として T::Builder 型が指定されます。
//!
//! @param[in] builder T::Builder への参照。
//!
//! @return T 型のアニメーションオブジェクトのポインター。
//!
template<typename T, typename U = typename T::Builder>
T* CreateAnimObj(U& builder) NN_NOEXCEPT
{
    NN_ASSERT(builder.IsValid());

    {
        builder.CalculateMemorySize();
    }

    T* pAnimObj = NULL;
    {
        // Tのサイズ + Tオブジェクト作成に必要なサイズ分のメモリーを確保する
        size_t alignment = T::Alignment_Buffer;
        size_t bufferSize = builder.GetWorkMemorySize();
        size_t objectSize = nn::util::align_up(sizeof(T), alignment);
        void* ptr = Allocate(objectSize + bufferSize, alignment);

        pAnimObj = new(ptr) T();
        ptr = nn::g3d::AddOffset(ptr, objectSize);

        bool result = builder.Build(pAnimObj, ptr, bufferSize);
        NN_UNUSED(result);
        NN_ASSERT(result == true);
    }

    return pAnimObj;
}

//! @brief CreateAnimObj() で作成した T 型のアニメーションオブジェクトを破棄します。
//!
//! @tparam T アニメーションの型。
//!
//! @param[in] pAnimObj T 型のアニメーションへのポインター。
//!
template<typename T>
void DestroyAnimObj(T* pAnimObj) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pAnimObj);

    pAnimObj->~T();
    Free(pAnimObj);
}

}}
//--------------------------------------------------------------------------------------------------
