﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_Mutex.h>

#include <nns/console/detail/console_ConsoleGraphicsFramework.h>

namespace nns {
namespace console {

// DefaultConsoleManagerSettings : 最大4つのコンソールを扱うことができる ConsoleManager の生成用 Settings
struct DefaultConsoleManagerSettings
{
    // この値は今のところ経験的に決めるしかなさそう
    static const size_t TotalMemorySize = 128 * 1024 * 1024;

    static const int ConsoleCount = 4; // 最大コンソール数
    static const int CharactorCountMax = 4096; // 1つのコンソール当たりに保持できる文字数
    static const size_t ScreenWidth = 1280; // レンダリング解像度(横)
    static const size_t ScreenHeight = 720; // レンダリング解像度(縦)
    static const size_t FrameworkMemorySize = 8 * 1024 * 1024; // nns::gfx::GraphicsFramework 用のバッファサイズ
};

// ConsoleManager : コンソールの管理体
template <typename ConsoleManagerSettings>
class ConsoleManager final
{
    NN_DISALLOW_COPY(ConsoleManager);
    NN_DISALLOW_MOVE(ConsoleManager);

private:
    detail::ConsoleGraphicsFramework m_Framework;

    nn::gfx::util::DebugFontTextWriter m_Writers[ConsoleManagerSettings::ConsoleCount];
    detail::ConsoleBase* m_Consoles[ConsoleManagerSettings::ConsoleCount];

    static void CommandMakeFunction(nn::gfx::CommandBuffer* pCommandBuffer, void* callbackUserData) NN_NOEXCEPT;

public:
    static const ConsoleManagerSettings Settings;

    ConsoleManager(void *buffer, size_t bufferSize) NN_NOEXCEPT;

    template <typename ConsoleType>
    void Register(ConsoleType* pConsole, int posX, int posY, size_t width, size_t height) NN_NOEXCEPT;

    template <typename ConsoleType>
    void Unregister(ConsoleType* pConsole) NN_NOEXCEPT;

    void Draw() NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;
};

// ConsoleManagerHolder : ConsoleManager 生成用のユーティリティ
template <typename ConsoleManagerSettings>
struct NN_ALIGNAS(4096) ConsoleManagerHolder
{
    char _memory[ConsoleManagerSettings::TotalMemorySize];
    typename std::aligned_storage<sizeof(ConsoleManager<ConsoleManagerSettings>), std::alignment_of<ConsoleManager<ConsoleManagerSettings>>::value>::type _storage;
    ConsoleManager<ConsoleManagerSettings>* _pManager;
    nn::os::MutexType _lock;
    bool _isInitialized;

    ConsoleManager<ConsoleManagerSettings>& GetManager() NN_NOEXCEPT;
};

// ConsoleManagerHolder 初期化用のマクロ
#define NNS_CONSOLE_CONSOLE_MANAGER_HOLDER_INITIALIZER {{}, {}, nullptr, NN_OS_MUTEX_INITIALIZER(false), false}

// DefaultConsoleManagerHolder : DefaultConsoleManagerSettings を使用した ConsoleManager 生成用のユーティリティ
typedef ConsoleManagerHolder<DefaultConsoleManagerSettings> DefaultConsoleManagerHolder;

} // ~namespace nns::console
}

// ---------------------------------------------------------------------------------------------

#include <mutex>
#include <new>

namespace nns {
namespace console {

template <typename ConsoleManagerSettings>
const ConsoleManagerSettings ConsoleManager<ConsoleManagerSettings>::Settings = {};

template <typename ConsoleManagerSettings>
inline ConsoleManager<ConsoleManagerSettings>& ConsoleManagerHolder<ConsoleManagerSettings>::GetManager() NN_NOEXCEPT
{
    nn::os::LockMutex(&_lock);
    if (!_isInitialized)
    {
        _pManager = new(&_storage)ConsoleManager<ConsoleManagerSettings>(_memory, sizeof(_memory));
        _isInitialized = true;
    }
    nn::os::UnlockMutex(&_lock);
    return *_pManager;
}

template <typename ConsoleManagerSettings>
inline void ConsoleManager<ConsoleManagerSettings>::CommandMakeFunction(nn::gfx::CommandBuffer* pCommandBuffer, void* callbackUserData) NN_NOEXCEPT
{
    auto& obj = *reinterpret_cast<ConsoleManager<ConsoleManagerSettings>*>(callbackUserData);
    for (auto i = 0; i < ConsoleManagerSettings::ConsoleCount; ++ i)
    {
        auto& c = obj.m_Consoles[i];
        if (c != nullptr)
        {
            c->MakeCommand(&obj.m_Writers[i], pCommandBuffer);
        }
    }
}

template <typename ConsoleManagerSettings>
inline ConsoleManager<ConsoleManagerSettings>::ConsoleManager(void *buffer, size_t bufferSize) NN_NOEXCEPT
    : m_Framework(buffer, bufferSize)
{
    for (auto& c : m_Consoles)
    {
        c = nullptr;
    }

    nn::util::Color4u8Type clearColor = {{4, 4, 4, 255}};
    m_Framework.Initialize(
        CommandMakeFunction, this,
        ConsoleManagerSettings::ScreenWidth, ConsoleManagerSettings::ScreenHeight,
        clearColor,
        ConsoleManagerSettings::FrameworkMemorySize);
}

template <typename ConsoleManagerSettings>
template <typename ConsoleType>
inline void ConsoleManager<ConsoleManagerSettings>::Register(ConsoleType* pConsole, int posX, int posY, size_t width, size_t height) NN_NOEXCEPT
{
    NN_STATIC_ASSERT((std::is_base_of<detail::ConsoleBase, ConsoleType>::value));

    int index = 0;
    for (; index < ConsoleManagerSettings::ConsoleCount; ++ index)
    {
        if (m_Consoles[index] == nullptr)
        {
            break;
        }
    }
    const auto ConsoleCount = ConsoleManagerSettings::ConsoleCount;
    NN_ABORT_UNLESS_LESS(index, ConsoleCount);

    auto heightDiff = static_cast<int>(ConsoleManagerSettings::ScreenHeight) - static_cast<int>(height);

    nn::gfx::ViewportStateInfo viewport;
    viewport.SetDefault();
    viewport.SetOriginX(static_cast<float>(posX));
    viewport.SetOriginY(static_cast<float>(- posY + heightDiff));
    viewport.SetWidth(static_cast<float>(width));
    viewport.SetHeight(static_cast<float>(height));

    nn::gfx::ScissorStateInfo scissor;
    scissor.SetDefault();
    scissor.SetOriginX(posX);
    scissor.SetOriginY(- posY + heightDiff);
    scissor.SetWidth(static_cast<int>(width));
    scissor.SetHeight(static_cast<int>(height));

    m_Consoles[index] = static_cast<detail::ConsoleBase*>(pConsole);
    m_Consoles[index]->Initialize(m_Framework.GetDevice(), viewport, scissor);

    m_Framework.InitializeDebugTextWriter(&m_Writers[index], width, height, ConsoleManagerSettings::CharactorCountMax);
}

template <typename ConsoleManagerSettings>
template <typename ConsoleType>
inline void ConsoleManager<ConsoleManagerSettings>::Unregister(ConsoleType* pConsole) NN_NOEXCEPT
{
    NN_STATIC_ASSERT((std::is_base_of<detail::ConsoleBase, ConsoleType>::value));

    int index = 0;
    for (; index < ConsoleManagerSettings::ConsoleCount; ++ index)
    {
        if (m_Consoles[index] == pConsole)
        {
            break;
        }
    }
    const auto ConsoleCount = ConsoleManagerSettings::ConsoleCount;
    NN_ABORT_UNLESS_LESS(index, ConsoleCount);

    m_Framework.FinalizeDebugTextWriter(&m_Writers[index]);

    m_Consoles[index]->Finalize();
    m_Consoles[index] = nullptr;
}

template <typename ConsoleManagerSettings>
inline void ConsoleManager<ConsoleManagerSettings>::Update() NN_NOEXCEPT
{
    for (auto& c : m_Consoles)
    {
        if (c != nullptr)
        {
            c->Update();
        }
    }
}

template <typename ConsoleManagerSettings>
inline void ConsoleManager<ConsoleManagerSettings>::Draw() NN_NOEXCEPT
{
    m_Framework.ProcessFrame();
}

} // ~namespace nns::console
}
