﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Quaternion.h>

#include <nns/ble/ble_Types.h>

namespace nns { namespace ble {

//!< Npad Service を表す UUID
const nn::bluetooth::GattAttributeUuid HidServiceUuid =
{
    .length = nn::bluetooth::GATT_ATTR_UUID_LEN_128,
    .uu.uuid128 = { 0xE5, 0x23, 0x7A, 0xE2, 0x51, 0x6B, 0x55, 0xBB, 0x67, 0x45, 0x6D, 0xF3, 0x6C, 0xE1, 0x75, 0x66 }
};

//!< HID Input Report Characteristic を表す UUID
const nn::bluetooth::GattAttributeUuid HidInputReportUuid
{
    .length = nn::bluetooth::GATT_ATTR_UUID_LEN_128,
    .uu.uuid128 = { 0xE6, 0x23, 0x7A, 0xE2, 0x51, 0x6B, 0x55, 0xBB, 0x67, 0x45, 0x6D, 0xF3, 0x6C, 0xE1, 0x75, 0x66 }
};

//!< Client Characteristic Config Descriptor を表すUUID
const nn::bluetooth::GattAttributeUuid CccUuid
{
    .length = nn::bluetooth::GATT_ATTR_UUID_LEN_16,
    .uu.uuid16 = 0x2902
};

//!< HID Report Reference Descriptor を表すUUID
const nn::bluetooth::GattAttributeUuid ReportReferenceUuid
{
    .length = nn::bluetooth::GATT_ATTR_UUID_LEN_16,
    .uu.uuid16 = 0x2908
};

/**
 * @brief   HID デバイスの入力状態を表す構造体です。
 */
struct HidState
{
    int64_t                     samplingNumber; //!< 入力状態が更新されるたびに増加する値です。
    nn::hid::NpadButtonSet      buttons;        //!< Npad のデジタルボタンの状態です。
    nn::hid::AnalogStickState   analogStick;    //!< アナログスティックの状態です。
    nn::hid::SixAxisSensorState sixAxisState;   //!< 6 軸センサーの状態です。
    NN_ALIGNAS(16)
        nn::util::Quaternion    quaternion;     //!< デバイスの 3 次元姿勢を表すクォータニオンです。
};

/**
 * @brief   指定されたクライアントが HID デバイスかどうか判定します。
 *
 * @param[in]   clientIndex     対象のクライアントの番号
 */
bool IsHidClient(uint8_t clientIndex) NN_NOEXCEPT;

/**
 * @brief   指定された HID クライアントデバイスの入力状態を取得します。
 *
 * @param[out]  pOutState       入力状態の格納先
 * @param[in]   clientIndex     対象のクライアントの番号
 *
 * @pre
 *  - @a pOutState != nullptr
 */
void GetHidState(HidState* pOutState, uint8_t clientIndex) NN_NOEXCEPT;

}}  // nns::ble
