﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/demo/demo_System.h>
#include <nw/demo/demo_DefaultAllocator.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>

#include <nw/dw.h>
#include <nw/font.h>

#include <nw/snd.h>
#include <nw/snd/util/sndutil_FileReader.h>
#include <nw/snddw.h>
#include "common.fsid"

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
#define DEMO_WIN
using namespace nw::internal::winext;
#include <windows.h>
#endif

//------------------------------------------------------------------------------

// 定数
const f32 WINDOW_WIDTH = 1280.0f;
const f32 WINDOW_HEIGHT = 720.0f;

// パッド
PADStatus s_PadStatus;

// アロケータ
#define HEAP_SIZE ( 16 * 1024 * 1024 )
nw::ut::MemoryAllocator s_Allocator;

// デモフレームワーク
nw::demo::DemoSystem* s_Demo = NULL;

// サウンドデバッグウィンドウフレームワーク
nw::snd::dw::SoundWindowSystem s_SoundWindowSystem;

void* s_pFontBinary;

// ウィンドウ
nw::snd::dw::VoicesWindow s_VoicesWindow;
nw::snd::dw::PerformanceWindow s_PerformanceWindow;
nw::snd::dw::MasterOutputWindow s_MasterOutputWindow;

void* s_pMemoryForSoundWindowSystem;
void* s_pMemoryForVoicesWindow;
void* s_pMemoryForPerformanceWindow;
void* s_pMemoryForMasterOutputWindow;

// ファイルシステム
const char CONTENT_DIR[] = "/vol/content";

// サウンド
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;
nw::snd::SoundHandle        s_SoundHandleHold;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
void* s_pMemoryForFsSoundArchive;
#endif

//==============================================================================
// 初期化、終了処理
//==============================================================================

//------------------------------------------------------------------------------
// サウンド

void
InitializeSdkSound()
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
#endif
}

void
FinalizeSdkSound()
{
#if defined( NW_PLATFORM_CAFE )
    AXQuit();
#endif
}

void
InitializeNwSound()
{
    nw::demo::DefaultAllocator allocator;

    // サウンドシステムの初期化
    {
        nw::snd::SoundSystem::SoundSystemParam param;
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
        s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

        nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );
    }

    // サウンドアーカイブの初期化
    {
#if defined(NW_PLATFORM_CAFE)
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = MEMAllocFromDefaultHeap(size);
        if ( ! s_SoundArchive.Open(
                    s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#else
        if ( ! s_SoundArchive.Open(SOUND_ARC_PATH) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#endif
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = allocator.Alloc( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }

    if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
    }
    if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
    }
}

void
FinalizeNwSound()
{
    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    nw::demo::DefaultAllocator allocator;
    allocator.Free( s_pMemoryForSoundHeap );
    allocator.Free( s_pMemoryForSoundArchivePlayer );
    allocator.Free( s_pMemoryForStreamBuffer );
    allocator.Free( s_pMemoryForSoundDataManager );
    allocator.Free( s_pMemoryForInfoBlock );
#if defined(NW_PLATFORM_CAFE)
    allocator.Free( s_pMemoryForFsSoundArchive );
#endif
    allocator.Free( s_pMemoryForSoundSystem );
}

//------------------------------------------------------------------------------
// デバッグウィンドウ
void InitializeDebugWindow()
{
    // フォントバイナリの読み込み
    nw::snd::util::FileReader fileReader;
    nw::snd::util::FileReader::Result result = fileReader.Initialize(
#if defined(NW_PLATFORM_CAFE)
        s_pFsClient
#endif
    );
    NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

#if defined(NW_PLATFORM_CAFE)
    result = fileReader.Open("common/fonts/nintendo_NTLG-DB_002.bffnt");
#else
    result = fileReader.Open("common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt");
#endif
    NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

    u32 fontBinarySize = fileReader.GetSize();
    s_pFontBinary = s_Allocator.Alloc(fontBinarySize, nw::font::RESOURCE_ALIGNMENT);
    s32 readSize = 0;
    fileReader.Read(s_pFontBinary, fontBinarySize, &readSize);
    NW_ASSERT(fontBinarySize == static_cast<u32>(readSize));

    result = fileReader.Close();
    NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);
    result = fileReader.Finalize();
    NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

    // SoundWindowSystemの初期化
    {
        u32 memorySize = s_SoundWindowSystem.GetRequiredSize();
        s_pMemoryForSoundWindowSystem = s_Allocator.Alloc(memorySize);

        nw::snd::dw::SoundWindowSystem::Param param;
        param.memory = s_pMemoryForSoundWindowSystem;
        param.memorySize = memorySize;
        param.fontBinary = s_pFontBinary;
        param.fontBinarySize = fontBinarySize;
        param.width = WINDOW_WIDTH;
        param.height = WINDOW_HEIGHT;

        s_SoundWindowSystem.Initialize(param);

        //s_SoundWindowSystem.SetModifierKey(nw::demo::Pad::MASK_R | nw::demo::Pad::MASK_L);
    }

    // VoicesWindowの初期化
    {
        u32 memorySize = s_VoicesWindow.GetRequiredSize();
        s_pMemoryForVoicesWindow = s_Allocator.Alloc(memorySize);
        s_VoicesWindow.Initialize(s_pMemoryForVoicesWindow, memorySize);
    }

    // PerformanceWindowの初期化
    {
        u32 memorySize = s_PerformanceWindow.GetRequiredSize();
        s_pMemoryForPerformanceWindow = s_Allocator.Alloc(memorySize);
        s_PerformanceWindow.Initialize(s_pMemoryForPerformanceWindow, memorySize);
    }

    // MasterOutputWindowの初期化
    {
        u32 memorySize = s_MasterOutputWindow.GetRequiredSize();
        s_pMemoryForMasterOutputWindow = s_Allocator.Alloc(memorySize);
        s_MasterOutputWindow.Initialize(s_pMemoryForMasterOutputWindow, memorySize);
    }

    // ウィンドウを生成
    f32 margin = 5.0f;
    s_SoundWindowSystem.AddWindow(
        margin,
        WINDOW_HEIGHT - s_PerformanceWindow.GetHeight() - margin,
        s_PerformanceWindow
    );
    s_SoundWindowSystem.AddWindow(
        WINDOW_WIDTH - s_MasterOutputWindow.GetWidth() - margin,
        margin,
        s_MasterOutputWindow
    );
    s_SoundWindowSystem.AddWindow(
        WINDOW_WIDTH - s_VoicesWindow.GetWidth() - margin,
        WINDOW_HEIGHT - s_VoicesWindow.GetHeight() - margin,
        s_VoicesWindow
    );
}

void FinalizeDebugWindow()
{
    s_MasterOutputWindow.Finalize();
    s_PerformanceWindow.Finalize();
    s_VoicesWindow.Finalize();
    s_SoundWindowSystem.Finalize();
    s_Allocator.Free(s_pMemoryForMasterOutputWindow);
    s_Allocator.Free(s_pMemoryForPerformanceWindow);
    s_Allocator.Free(s_pMemoryForVoicesWindow);
    s_Allocator.Free(s_pMemoryForSoundWindowSystem);

    s_Allocator.Free(s_pFontBinary);
}


//------------------------------------------------------------------------------
/*
 *  デモの初期化処理
 */
void
InitializeDemo()
{
    InitializeSdkSound();
    InitializeNwSound();
    InitializeDebugWindow();

    PADInit();
}

//------------------------------------------------------------------------------
/*
 *  デモの終了処理
 */
void
FinalizeDemo()
{
    FinalizeDebugWindow();
    FinalizeNwSound();
    FinalizeSdkSound();
}

//------------------------------------------------------------------------------
/*
 *  入力処理
 */
void
UpdateInput()
{
    nw::demo::Pad* pad = s_Demo->GetPad();
    pad->GetPadStatus(&s_PadStatus);
    s_SoundWindowSystem.UpdateInputs(s_PadStatus);

    if (pad->IsTrig(nw::demo::Pad::MASK_A))
    {
        s_SoundHandle.Stop( 0 );
        bool result = s_SoundArchivePlayer
            .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
        NW_LOG("[SEQ] StartSound(SEQ_MARIOKART) ... (%d)\n", result);
    }
    else if (pad->IsTrig(nw::demo::Pad::MASK_X))
    {
        s_SoundHandle.Stop( 0 );
        bool result = s_SoundArchivePlayer
            .StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
        NW_LOG("[SEQ] StartSound(STRM_MARIOKART) ... (%d)\n", result);
    }
    else if (pad->IsTrig(nw::demo::Pad::MASK_B))
    {
        s_SoundHandle.Stop( 0 );
    }
}


//------------------------------------------------------------------------------
/*
 *  計算処理
 */
void
ProcCalc()
{
    s_SoundWindowSystem.Update();

    // 入力処理
    UpdateInput();

    s_SoundArchivePlayer.Update();
}

//------------------------------------------------------------------------------
/*
 *  描画処理
 */
void
ProcDraw()
{
    s_SoundWindowSystem.Draw();

#if defined(DEMO_WIN)
    NW_GL_ASSERT();
#endif
}

//------------------------------------------------------------------------------
/*
 *  1 フレーム分の処理
 */
void
ProcFrame()
{
    // Pad の計算
    s_Demo->UpdatePad();

    // 計算処理
    ProcCalc();

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        // 画面のクリア
        s_Demo->ClearFrameBuffers();

        // 描画処理
        ProcDraw();
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    // バッファのスワップ
    s_Demo->SwapBuffer();

    //  VBlank待ち
    s_Demo->WaitForVBlank();
}

void
PrintUsage()
{
    NW_LOG("-------------------------------------------------------------------------------\n");
    NW_LOG("NW4F Demo/snd/snddw\n");
    NW_LOG("-------------------------------------------------------------------------------\n");
    NW_LOG("[A]                                         StartSound SEQ  (SEQ_MARIOKART)\n");
    NW_LOG("[X]                                         StartSound STRM (STRM_MARIOKART)\n");
    NW_LOG("[B]                                         Stop Sound\n");
    NW_LOG("[R + ZR]                                    Toggle visibility of all windows\n");
    NW_LOG("[R + Left Stick]                            Move Cursor\n");
    NW_LOG("[R + Right Stick]                           Adjust transparency of the window\n");
    NW_LOG("[R + A(On a Window) + Left Stick]           Move the window\n");
    NW_LOG("[R + X(On a Window) + Left Stick]           Adjust the window size\n");
    NW_LOG("[R + A(On a Window) + UP/DOWN/LEFT/RIGHT]   Move the window \n");
    NW_LOG("[R + Y(On a Window)]                        Change WindowMode\n");
    NW_LOG("[R + UP/DOWN]                               Change ActiveWindow\n");
    NW_LOG("[R + UP/DOWN + B]                           Toggle visibility of selected window\n");
    NW_LOG("[R + A + UP/DOWN]                           Change order of the window\n");
    NW_LOG("-------------------------------------------------------------------------------\n");
}

//------------------------------------------------------------------------------
/*
 *  main() 関数
 */
int
NwDemoMain(int /* argc */, char ** /* argv */)
{
    // アロケーターの初期化
#if defined(DEMO_WIN)
    void* addr = malloc( HEAP_SIZE );
#else
    nw::demo::DefaultAllocator allocator;
    void* addr = allocator.Alloc( HEAP_SIZE );
#endif
    s_Allocator.Initialize( addr, HEAP_SIZE );

    // ファイルシステムの初期化
#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(MEMAllocFromDefaultHeap(sizeof(FSClient)));
        NW_ASSERT_NOT_NULL(s_pFsClient);
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(MEMAllocFromDefaultHeap(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        MEMFreeToDefaultHeap(block);
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#endif

    // フレームワークの作成
    nw::demo::DemoSystem::CreateArg arg;
    arg.allocator = &s_Allocator;
    arg.waitVBlank = 1;
    arg.width = static_cast<s32>(WINDOW_WIDTH);
    arg.height = static_cast<s32>(WINDOW_HEIGHT);
    s_Demo = new( s_Allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) ) nw::demo::DemoSystem( arg );

    // フレームワークの初期化
    s_Demo->Initialize();

    // グラフィックシステムの初期化
    s_Demo->InitializeGraphicsSystem();

    // デモの初期化
    InitializeDemo();

    // 操作説明の表示
    PrintUsage();

    // メインループ
    NW_LOG("Start demo.\n");
    while ( !s_Demo->IsExiting() )
    {
        ProcFrame();
    }

    // デモの終了処理
    FinalizeDemo();

    // グラフィックスシステムの終了処理
    s_Demo->FinalizeGraphicsSystem();

    // フレームワークの終了処理
    s_Demo->Finalize();

    // フレームワークの破棄
#if defined(DEMO_WIN)
    s_Demo->~SystemWinGL();
#else
    s_Demo->~SystemCafe();
#endif
    nw::ut::SafeFree( s_Demo, &s_Allocator );

#if defined( NW_PLATFORM_CAFE )
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    MEMFreeToDefaultHeap(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();

    AXQuit();
#endif

    // アロケータの終了処理
    s_Allocator.Finalize();
#if defined(DEMO_WIN)
    free( addr );
#else
    allocator.Free( addr );
#endif

    NW_LOG("End demo.\n");
    return 0;
}

