﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SoundDemoUtility.h"
#include "ModelUtility.h"

namespace {

const f32 DEFAULT_CAMERA_POS_X = 10.0f;
const f32 DEFAULT_CAMERA_POS_Z = 0.0f;

}

SoundDemoSystem::SoundDemoSystem()
: m_Allocator(NULL)
, m_Demo(NULL)
, m_CameraPosX(DEFAULT_CAMERA_POS_X)
, m_CameraPosZ(DEFAULT_CAMERA_POS_Z)
{
}

SoundDemoSystem::~SoundDemoSystem()
{
}

//------------------------------------------------------------------------------
//      デモ 初期化処理
//------------------------------------------------------------------------------
void SoundDemoSystem::Initialize(nw::ut::IAllocator& allocator)
{
    m_Allocator = &allocator;

    // ファイルシステムの初期化
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
    fileDeviceArg.allocator = m_Allocator;
    fileSystem->Initialize( fileDeviceArg );

    // デモシステムの作成
#if defined(USE_DRC)
    SimpleDemoSystemDRC::CreateArg arg;
#else
    SimpleDemoSystem::CreateArg    arg;
#endif
    arg.allocator   = m_Allocator;
    arg.waitVBlank  = 1;
    arg.width       = 1280;
    arg.height      = 720;
    arg.clearColor  = nw::ut::Color4u8::GRAY;
    arg.clearDepth  = 1.0f;
    arg.drawMeter   = false;
#if defined(NW_PLATFORM_CAFE)
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002.bffnt";
    arg.fontShaderPath = "common/shaders/font_BuildinShader.gsh";
#else
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt";
#endif

#if defined(USE_DRC)
    m_Demo = new SimpleDemoSystemDRC( arg );
#else
    m_Demo = new SimpleDemoSystem( arg );
#endif

    // デモシステムの初期化
    m_Demo->Initialize();

    // グラフィックシステムの初期化
    m_Demo->InitializeGraphicsSystem();

    // PrimitiveRenderer の初期化
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        renderer->Initialize( m_Allocator );
#else
        // PrimitiveRenderer で用いるシェーダーバイナリへのパスを指定する。
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "common/shaders/dev_PrimitiveRenderer.gsh";
        loadArg.allocator = m_Allocator;
        u8* binary = fileSystem->Load( loadArg );
        renderer->InitializeFromBinary( m_Allocator, binary, loadArg.readSize );
        fileSystem->Unload( loadArg, binary );
#endif
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    // ビュー行列の初期化
    {
        const f32 width  = static_cast<f32>( m_Demo->GetWidth() );
        const f32 height = static_cast<f32>( m_Demo->GetHeight() );
        nw::math::MTX44Perspective( &m_ProjctionMatrix, nw::math::F_PI/4, width/(f32)height, 0.1f, 1000.f);
        nw::math::MTX34Identity( &m_ViewMatrix );
    }
#if defined(USE_DRC)
    {
        const f32 width  = static_cast<f32>( m_Demo->GetDRCWidth() );
        const f32 height = static_cast<f32>( m_Demo->GetDRCHeight() );
        nw::math::MTX44 viewMat;
        nw::math::MTX44Perspective( &m_ProjctionMatrixDRC, nw::math::F_PI/4, width/(f32)height, 0.1f, 1000.f);
        nw::math::MTX34Identity( &m_ViewMatrixDRC );
    }
#endif

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    // GL では座標を Cafe と同じ中心に設定する。
    nw::ut::DynamicCast<nw::demo::WPadWin*>( m_Demo->GetWPad() )->SetPointerCenterOrigin( true );
    nw::ut::DynamicCast<nw::demo::MouseWin*>( m_Demo->GetMouse() )->SetPointerCenterOrigin( true );
#endif
}


//------------------------------------------------------------------------------
//      デモ 終了処理
//------------------------------------------------------------------------------
void SoundDemoSystem::Finalize()
{
    // PrimitiveRenderer終了処理
    nw::dev::PrimitiveRenderer::GetInstance()->Finalize( m_Allocator );

    // デモシステム終了処理
    m_Demo->FinalizeGraphicsSystem();
    m_Demo->Finalize();

    // ファイルシステム終了処理
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    fileSystem->Finalize();

    m_Allocator = NULL;
}

//------------------------------------------------------------------------------
//      カメラ処理
//------------------------------------------------------------------------------
void SoundDemoSystem::ProcCamera()
{
    // TVカメラ処理
    {
        if (m_Demo->GetPad()->IsHold(nw::demo::Pad::MASK_UP))
        {
            m_CameraPosX += 0.2f;
        }
        else if (m_Demo->GetPad()->IsHold(nw::demo::Pad::MASK_DOWN))
        {
            m_CameraPosX -= 0.2f;
        }

        if (m_Demo->GetPad()->IsHold(nw::demo::Pad::MASK_RIGHT))
        {
            m_CameraPosZ += 0.2f;
        }
        else if (m_Demo->GetPad()->IsHold(nw::demo::Pad::MASK_LEFT))
        {
            m_CameraPosZ -= 0.2f;
        }

        if (m_Demo->GetPad()->IsTrig(nw::demo::Pad::MASK_X))
        {
            m_CameraPosX = DEFAULT_CAMERA_POS_X;
            m_CameraPosZ = DEFAULT_CAMERA_POS_Z;
        }

        m_CameraPosition     = nw::math::VEC3(m_CameraPosX, 3.0f, m_CameraPosZ);
        m_CameraLookAt       = nw::math::VEC3(m_CameraPosX - DEFAULT_CAMERA_POS_X, 3.0f, m_CameraPosZ - DEFAULT_CAMERA_POS_Z);
        nw::math::MTX34LookAt(
            &m_ViewMatrix,
            m_CameraPosition,
            nw::math::VEC3(0, 1, 0),
            m_CameraLookAt
        );
    }

#ifdef USE_DRC
    // DRCカメラ処理
    {
        const f32 rotRadius = 15.f;
        const f32 cameraRot = - nw::ut::Tick::GetSystemCurrent().ToTimeSpan().GetMilliSeconds() / 50.f;
        m_CameraPositionDRC  = nw::math::VEC3(rotRadius * nw::math::CosDeg(cameraRot), -5, rotRadius * nw::math::SinDeg(cameraRot));
        m_CameraLookAtDRC    = nw::math::VEC3(0, 0, 0);
        nw::math::MTX34LookAt( &m_ViewMatrixDRC, m_CameraPositionDRC, nw::math::VEC3(0,1,0), m_CameraLookAtDRC );
    }
#endif
}

//------------------------------------------------------------------------------
//      計算処理
//------------------------------------------------------------------------------
void SoundDemoSystem::ProcFrame()
{
    // フレームの開始処理
    m_Demo->BeginFrame();

    // デモパッドの更新
    m_Demo->UpdatePad();

    // カメラの更新
    ProcCamera();
}

//------------------------------------------------------------------------------
//      背景処理
//------------------------------------------------------------------------------
void SoundDemoSystem::ProcBackGroundDraw()
{
    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
    nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();
    graphics->SetBlendEnable(false);

    {
        renderer->SetProjectionMtx( &m_ProjctionMatrix );
        renderer->SetViewMtx( &m_ViewMatrix );

        renderer->Begin();
        {
            graphics->SetDepthTestEnable( true );
            graphics->SetDepthWriteEnable( true );

            {
                nw::dev::PrimitiveRenderer::CubeArg arg;
                arg.SetCenter( 0, -3, 0.0f );
                arg.SetSize(10, 0.2f, 10);
                arg.SetColor( nw::ut::Color4u8( 255, 0, 0, 255 ), nw::ut::Color4u8( 0, 0, 255, 255 ) );
                renderer->DrawCube( arg );
            }
        }
        renderer->End();
    }
}

void SoundDemoSystem::UpdateViewBlock(SimpleModel& model)
{
    model.CalcBlock(
        reinterpret_cast<const nw::g3d::math::Mtx34*>(&m_ViewMatrix),
        reinterpret_cast<const nw::g3d::math::Mtx44*>(&m_ProjctionMatrix)
    );

    nw::g3d::CPUCache::Sync();
    nw::g3d::GPUCache::InvalidateAll();
}
