﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "soundActor";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;
    nw::snd::SoundHandle s_SoundHandleHold;
    nw::snd::SoundActor         s_SoundActor;

    f32 s_ActorPan;
    f32 s_ActorVolume;

    const f32 ACTOR_PAN_MIN = -1.0f;
    const f32 ACTOR_PAN_MAX =  1.0f;
    const f32 ACTOR_PAN_DEFAULT = 0.0f;
    const f32 ACTOR_VOLUME_MIN = 0.1f;
    const f32 ACTOR_VOLUME_MAX = 2.0f;
    const f32 ACTOR_VOLUME_DEFAULT = 1.0f;
    const f32 CONTROL_PAD_STEP = 0.05f;

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        // データのロード
        if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
        }

        // サウンドアクターの初期化
        {
            s_ActorPan = ACTOR_PAN_DEFAULT;
            s_ActorVolume = ACTOR_VOLUME_DEFAULT;

            s_SoundActor.Initialize( s_SoundArchivePlayer );
            s_SoundActor.SetPlayableSoundCount( 0, 2 );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        s_SoundActor.Finalize();

        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]          Start Sound\n");
        NW_LOG("[B]          Stop Sound\n");
        NW_LOG("[LEFT/RIGHT] Modify Actor Pan\n");
        NW_LOG("[UP/DOWN]    Modify Actor Volume\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]       Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]          Exit Application\n");
#endif
        NW_LOG("[R]          Print Usage (this)\n");
        NW_LOG("---------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundActor
                .StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            NW_LOG("StartSound(SE_YOSHI) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundActor.StopAllSound( 3 );
        }

        // 左右で、パンを変更しながらホールド再生
        if ( pad->IsHold( nw::demo::Pad::MASK_LEFT ) )
        {
            s_ActorPan = s_ActorPan <= ACTOR_PAN_MIN ?
                ACTOR_PAN_MIN : s_ActorPan - CONTROL_PAD_STEP;
            bool result = s_SoundActor
                .HoldSound( &s_SoundHandleHold, SE_WIHAHO ).IsSuccess();
            s_SoundActor.SetOutputPan( nw::snd::OUTPUT_DEVICE_MAIN, s_ActorPan );
            NW_LOG("HoldSound(SE_WIHAHO) ... (%d) pan(%.2f)\n", result, s_ActorPan);
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_RIGHT ) )
        {
            s_ActorPan = s_ActorPan >= ACTOR_PAN_MAX ?
                ACTOR_PAN_MAX : s_ActorPan + CONTROL_PAD_STEP;
            bool result = s_SoundActor
                .HoldSound( &s_SoundHandleHold, SE_WIHAHO ).IsSuccess();
            s_SoundActor.SetOutputPan( nw::snd::OUTPUT_DEVICE_MAIN, s_ActorPan );
            NW_LOG("HoldSound(SE_WIHAHO) ... (%d) pan(%.2f)\n", result, s_ActorPan);
        }

        // 上下で、ボリューム変更
        if ( pad->IsHold( nw::demo::Pad::MASK_UP ) )
        {
            s_ActorVolume = s_ActorVolume >= ACTOR_VOLUME_MAX ?
                ACTOR_VOLUME_MAX : s_ActorVolume + CONTROL_PAD_STEP;
            bool result = s_SoundActor
                .HoldSound( &s_SoundHandleHold, SE_WIHAHO ).IsSuccess();
            s_SoundActor.SetVolume( s_ActorVolume );
            NW_LOG("HoldSound(SE_WIHAHO) ... (%d) vol(%.2f)\n", result, s_ActorVolume );
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_DOWN ) )
        {
            s_ActorVolume = s_ActorVolume <= ACTOR_VOLUME_MIN ?
                ACTOR_VOLUME_MIN : s_ActorVolume - CONTROL_PAD_STEP;
            bool result = s_SoundActor
                .HoldSound( &s_SoundHandleHold, SE_WIHAHO ).IsSuccess();
            s_SoundActor.SetVolume( s_ActorVolume );
            NW_LOG("HoldSound(SE_WIHAHO) ... (%d) vol(%.2f)\n", result, s_ActorVolume );
        }

        // Usage 表示
        if ( pad->IsTrig( nw::demo::Pad::MASK_R ) ) {
            PrintUsage();
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) )
        {
            return false;
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void SoundActorDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
