﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "drc";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;

    const f32 PARAM_DELTA = 0.2f;
    const f32 PARAM_MIN = 0.2f;
    const f32 PARAM_MAX = 4.0f;
    const f32 PARAM_DEFAULT= 1.0f;

    f32 s_Volume = PARAM_DEFAULT;
    f32 s_Pitch = PARAM_DEFAULT;

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        // データロード
        if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
        }
        if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]     StartSound SEQ  (SEQ_MARIOKART)\n");
        NW_LOG("[X]     StartSound WSD  (SE_YOSHI)\n");
        NW_LOG("[Y]     StartSound STRM (STRM_MARIOKART)\n");
        NW_LOG("[B]     Stop Sound\n");
        NW_LOG("[R]     Reset Volume/Pitch\n");
        NW_LOG("[LEFT/RIGHT] Change DrcOutVolume (0.2-4.0)\n");
        NW_LOG("[UP/DOWN]    Change Pitch (0.2-4.0)\n");
        NW_LOG("[HOME]       Exit Application\n");
        NW_LOG("[L]          Print Usage (this)\n");
        NW_LOG("---------------------------------------\n");
    }

    void StartSound( u32 soundId )
    {
        s_SoundHandle.Stop( 0 );
        bool result = s_SoundArchivePlayer.StartSound( &s_SoundHandle, soundId ).IsSuccess();
        if ( result )
        {
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_DRC0 );
            s_SoundHandle.SetOutputVolume( nw::snd::OUTPUT_DEVICE_DRC, s_Volume );
            s_SoundHandle.SetPitch( s_Pitch );
            NW_LOG("StartSound(%08x) volume(%.2f) pitch(%.2f)\n", soundId, s_Volume, s_Pitch );
        }
        else
        {
            NW_LOG("StartSound(%08x) failed.\n", soundId );
        }
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            StartSound( SEQ_MARIOKART );
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            StartSound( SE_YOSHI );
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            StartSound( STRM_MARIOKART );
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // パラメータ変更
        if ( pad->IsTrig( nw::demo::Pad::MASK_LEFT ) )
        {
            s_Volume -= PARAM_DELTA;
            if ( s_Volume < PARAM_MIN )
            {
                s_Volume = PARAM_MIN;
            }

            s_SoundHandle.SetOutputVolume( nw::snd::OUTPUT_DEVICE_DRC, s_Volume );
            NW_LOG("DccOutVolume(%.2f)\n", s_Volume);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_RIGHT ) )
        {
            s_Volume += PARAM_DELTA;
            if ( s_Volume > PARAM_MAX )
            {
                s_Volume = PARAM_MAX;
            }

            s_SoundHandle.SetOutputVolume( nw::snd::OUTPUT_DEVICE_DRC, s_Volume );
            NW_LOG("DrcOutVolume(%.2f)\n", s_Volume);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_UP ) )
        {
            s_Pitch += PARAM_DELTA;
            if ( s_Pitch > PARAM_MAX )
            {
                s_Pitch = PARAM_MAX;
            }

            s_SoundHandle.SetPitch( s_Pitch );
            NW_LOG("Pitch(%.2f)\n", s_Pitch);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_DOWN ) )
        {
            s_Pitch -= PARAM_DELTA;
            if ( s_Pitch < PARAM_MIN )
            {
                s_Pitch = PARAM_MIN;
            }

            s_SoundHandle.SetPitch( s_Pitch );
            NW_LOG("Pitch(%.2f)\n", s_Pitch);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_R ) )
        {
            s_Volume = s_Pitch = PARAM_DEFAULT;

            s_SoundHandle.SetOutputVolume( nw::snd::OUTPUT_DEVICE_DRC, s_Volume );
            s_SoundHandle.SetPitch( s_Pitch );
            NW_LOG("DrcOutVolume(%.2f) Pitch(%.2f)\n", s_Volume, s_Pitch);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_L ) )
        {
            PrintUsage();
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            return false;
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void DrcDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
