﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/////////////////////////////////////////////////////////////////////////////
//
// simple.cpp
//
// スケーラブルフォントエンジンで文字ビットマップを
// 生成してテクスチャに貼って表示します。
//
// 本デモは、基本的な処理の流れだけを示しています。
// より実戦的な利用については、NW4F の ScFontモジュールのソースコードを参考
// にしてください。
//////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>

#include <cafe/demo.h>
#include <cafe/gx2.h>
#include <nw/scfont/scffnd_ScalableFontEngine.h>

////////////////////////////////////////////////////
//
// Assets data, types and interface for demos
//
////////////////////////////////////////////////////

// ----- Font
static scffnd::ScalableFontEngine* pFontEngine = NULL;

static const int WORK_MEMORY_SIZE 		= 1024 * 1024 * 2;
static const int TEXTURE_H 				= 256;
static const int TEXTURE_W 				= 256;
static const int FONT_SIZE 				= 192;

static float sAscent_ratio 			= 1.0f;
static u8* spTexAddr 				= NULL;
static u32 sCharCode 				= 'a';

// ----- Shader information
static const char * const GSH_SHADER_FILE = "assets/shaders/texture2D.gsh";

static void * gshBuf;

static GX2VertexShader * pVertexShader;
static GX2PixelShader * pPixelShader;
static GX2FetchShader fetchShader;
static void * fsBuf;

static GX2Texture *pTexture;

static u32 offsetLoc;
static u32 posLoc;
static u32 tcLoc;
static u32 samplerLoc;

// ----- GX2 Triangle Data

#define AR (16.0f/9.0f)

static DEMO_F32x3 posDataGX2[] =
{
    {-0.3f,  0.3f*AR, 0.0f},
    { 0.3f,  0.3f*AR, 0.0f},
    {-0.3f, -0.3f*AR, 0.0f},
    { 0.3f, -0.3f*AR, 0.0f}
};

static DEMO_F32x2 tcDataGX2[] =
{
     {0.0f,  0.0f},
     {1.0f,  0.0f},
     {0.0f,  1.0f},
     {1.0f,  1.0f}
};

static u32 idxDataGX2[] = { 0, 1, 2, 3 };

#define NUM_ATTRIB 2
#define NUM_VERTEX 4

#define POSITION_BUFFER_IDX 0
#define TEXCOORD_BUFFER_IDX 1

static GX2AttribStream attribs[NUM_ATTRIB];

// ----- Attribute buffers
static GX2RBuffer vertexBuffer   = { GX2R_RESOURCE_FLAGS_NONE };
static GX2RBuffer texcoordBuffer = { GX2R_RESOURCE_FLAGS_NONE };
static GX2RBuffer indexBuffer    = { GX2R_RESOURCE_FLAGS_NONE };

// ----- Texture & sampler objects
static GX2Texture myTex1;
static GX2Sampler mySampler1;

////////////////////////////////////////////////////
//
// Prototypes
//
////////////////////////////////////////////////////

static BOOL TexInit(void);
static BOOL SceneInit(void);
static BOOL SceneDraw(void);
static void SceneShutdown(void);

////////////////////////////////////////////////////
//
// Functions
//
////////////////////////////////////////////////////

//----------------------------------------------------------
static void AssertFsError(const char* api_name)
{
    if(pFontEngine->GetError() != scffnd::NO_ERR)
    {
        DEMOPrintf("%s error: %d", api_name, pFontEngine->GetError());
    }
}

//----------------------------------------------------------
static BOOL TexInit(void)
{
    GX2InitTexture(&myTex1, TEXTURE_H, TEXTURE_W, 1, 0, GX2_SURFACE_FORMAT_TC_R8_UNORM, GX2_SURFACE_DIM_2D);
    myTex1.surface.tileMode = GX2_TILE_MODE_LINEAR_ALIGNED;

    GX2InitTextureCompSel(&myTex1, GX2_GET_COMP_SEL(GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R, GX2_COMPONENT_X_R));

    GX2CalcSurfaceSizeAndAlignment(&myTex1.surface);
    GX2InitTextureRegs(&myTex1);

    spTexAddr = (u8*)DEMOGfxAllocMEM2(myTex1.surface.imageSize, myTex1.surface.alignment);
    GX2InitTexturePtrs(&myTex1, spTexAddr, 0);
    memset(spTexAddr, 0, myTex1.surface.imageSize);
    GX2Invalidate(GX2_INVALIDATE_CPU_TEXTURE, myTex1.surface.imagePtr, myTex1.surface.imageSize);

    GX2InitSampler(&mySampler1, GX2_TEX_CLAMP_CLAMP, GX2_TEX_XY_FILTER_BILINEAR);

    return 1;
}

//----------------------------------------------------------

static void CreateGlyphTexture(u32 charCode)
{
    // グリフビットマップを取得します。
    // エンジン内部ワークにビットマップを生成しています。
    // MAP_GRAYMAP8 だけが指定可能です。
    scffnd::ScGlyphMap *pMap = pFontEngine->GetGlyphmap(charCode, scffnd::MAP_GRAYMAP8);
    AssertFsError("GetGlyphmap");
    if(pMap == NULL)
    {
        return;
    }

    // 表示確認用に グリフビットマップ を テクスチャ にコピーします。
    {

        // 全体をクリア
        std::memset(spTexAddr, 32, myTex1.surface.imageSize);

        const u32 cache_flush_size 	= pMap->width + 32;
        const int font_ascent 		= static_cast<int>(sAscent_ratio * FONT_SIZE + 0.5f);
        const int offsetFromTop 	= font_ascent - pMap->hiY;
        const int render_origin_y 	= offsetFromTop;
        const int render_origin_x 	= static_cast<int>((TEXTURE_W - pMap->width) * 0.5 + pMap->loX);

        // ベースラインと、文字のトップにラインを描画する
        std::memset(&spTexAddr[(render_origin_y) * TEXTURE_W], 255, TEXTURE_W);
        std::memset(&spTexAddr[(font_ascent) * TEXTURE_W], 255, TEXTURE_W);

        // エンジンが生成したグリフビットマップをテクスチャにコピーする
        for (s32 l = 0; l < pMap->height; l++)
        {
            u8* line_start = &spTexAddr[(l + render_origin_y) * TEXTURE_W + render_origin_x];
            std::memcpy(line_start, &pMap->bits[l * pMap->width], pMap->width);

            // DCFlushRange(line_start, copy_size)と同等の処理
            for (u32 dot = 0; dot < cache_flush_size; dot += 32)
            {
                asm ( "dcbf		%0, %1" : "+g"(line_start), "+g"(dot) );
            }
        }

        GX2Invalidate(GX2_INVALIDATE_CPU_TEXTURE, myTex1.surface.imagePtr, myTex1.surface.imageSize);
    }

    // エンジンのグリフ情報を破棄します。
    pFontEngine->ReleasesGlyph(pMap);
    AssertFsError("ReleasesGlyph");
}

//----------------------------------------------------------
static BOOL FontInit()
{
    // エンジン用ワークメモリの確保
    pFontEngine = static_cast<scffnd::ScalableFontEngine*>(DEMOGfxAllocMEM2(sizeof(scffnd::ScalableFontEngine), 4));
    pFontEngine->Initialize(DEMOGfxAllocMEM2(WORK_MEMORY_SIZE, 4), WORK_MEMORY_SIZE);

    // フォントを読み込み、エンジンに設定します。
    {
        u32 fontFileSize;
        void* fontData;

#if 1
        // 共有リソースから内蔵フォントを取得します。
        BOOL result = OSGetSharedData(SHARED_FONT_CAFESTD, 0, &fontData, &fontFileSize);
        ASSERT(result);
#else
        // ファイルから bfttf を読み込みます。
        // bfttf は Load前にデコード処理をする
        const char * const FONT_FILE = "assets/fonts/nintendo_NTLG-DB_002.bfttf";
        void* pRawFontFile 	= DEMOFSSimpleRead(FONT_FILE, &fontFileSize);
        fontData 		= scffnd::ScalableFontEngineHelper::Decode(pRawFontFile, fontFileSize);
#endif

        char* pMyFontName = "MyFontName";
        pFontEngine->LoadFont(NULL, fontData, 0, scffnd::MAX_FONT_NAME_LEN, pMyFontName);
        AssertFsError("LoadFont");

        pFontEngine->SetFont(pMyFontName);
        AssertFsError("SetFont");
    }

    // フォントサイズを設定します。
    pFontEngine->SetScale(FONT_SIZE << 16, 0, 0, FONT_SIZE << 16);
    AssertFsError("SetScale");

    // グリフの配置用にフォント情報を取得して必要な情報を記憶しておきます。
    {
        scffnd::ScMetrics metrics;
        pFontEngine->GetFontMetrics(&metrics);
        AssertFsError("GetFontMetrics");

        sAscent_ratio = static_cast<float>(metrics.os2WinAscent) / metrics.metricsResolution;
    }

    // グリフビットマップを生成します。
    CreateGlyphTexture(sCharCode);

    return 1;
}

//----------------------------------------------------------
// 初期化
static BOOL SceneInit(void)
{
    // シェーダバイナリの込みこみ
    u32 gshLen;
    {
        gshBuf = DEMOFSSimpleRead(GSH_SHADER_FILE, &gshLen);
        ASSERT(NULL != gshBuf && "Unable to load the shader file");
        BOOL fOKv = DEMOGFDReadVertexShader(&pVertexShader, 0, gshBuf);
        BOOL fOKp = DEMOGFDReadPixelShader(&pPixelShader, 0, gshBuf);
        if (FALSE == fOKv)
        {
            ASSERT(fOKv && "The shader file did not contain a vertex shader");
            DEMOPrintf("The file %s did not contain a vertex shader\n", GSH_SHADER_FILE);
            exit(-1);
        }
        if (FALSE == fOKp)
        {
            ASSERT(fOKp && "The shader file did not contain a pixel shader");
            DEMOPrintf("The file %s did not contain a pixel shader\n", GSH_SHADER_FILE);
            exit(-1);
        }
    	DEMOFree(gshBuf);
    }

    // Uniform Location ,Texture Sampler ,Attribute Location ...
    offsetLoc = (u32)GX2GetVertexUniformVarOffset(pVertexShader, "u_offset");
    samplerLoc = (u32)GX2GetPixelSamplerVarLocation(pPixelShader, "s_texture");
    posLoc = (u32)GX2GetVertexAttribVarLocation(pVertexShader, "a_position");
    tcLoc  = (u32)GX2GetVertexAttribVarLocation(pVertexShader, "a_texCoord");

    // Attribute Format 初期化
    GX2InitAttribStream(&attribs[0], posLoc, POSITION_BUFFER_IDX, 0, GX2_ATTRIB_FORMAT_32_32_32_FLOAT);
    GX2InitAttribStream(&attribs[1], tcLoc, TEXCOORD_BUFFER_IDX, 0, GX2_ATTRIB_FORMAT_32_32_FLOAT);

    // GX2 Fetch Shader 初期化
    fsBuf = DEMOGfxAllocMEM2(GX2CalcFetchShaderSize(NUM_ATTRIB), GX2_SHADER_ALIGNMENT);
    GX2InitFetchShader(&fetchShader, fsBuf, NUM_ATTRIB, attribs);

    // 頂点バッファ
    GX2UTCreateVertexBuffer(&vertexBuffer, sizeof(posDataGX2[0]),  sizeof(posDataGX2) / sizeof(posDataGX2[0]));
    GX2UTFillBuffer(&vertexBuffer, (void*)posDataGX2);

    // テクスチャ座標バッファ
    GX2UTCreateVertexBuffer(&texcoordBuffer, sizeof(tcDataGX2[0]), sizeof(tcDataGX2) / sizeof(tcDataGX2[0]));
    GX2UTFillBuffer(&texcoordBuffer, (void*)tcDataGX2);

    // インデックスバッファ
    GX2UTCreateIndexBuffer(&indexBuffer, sizeof(idxDataGX2[0]), sizeof(idxDataGX2) / sizeof(idxDataGX2[0]));
    GX2UTFillBuffer(&indexBuffer, (void*)idxDataGX2);

    // Texture setup
    TexInit();

    // フォントレンダラの初期化
    FontInit();

    return 1;
}

//----------------------------------------------------------

static void FontShutdown(void)
{
    void* work = reinterpret_cast<void*>(pFontEngine->GetPointerToWorkBuffer());

    pFontEngine->Finalize();
    AssertFsError("Exit");

    DEMOGfxFreeMEM2(pFontEngine);
    DEMOGfxFreeMEM2(work);
}

//----------------------------------------------------------

static void SceneShutdown(void)
{
    // フォントレンダラの終了処理。
    FontShutdown();

    // shader
    DEMOGFDFreeVertexShader(pVertexShader);
    DEMOGFDFreePixelShader(pPixelShader);

    // texture
    DEMOGFDFreeTexture(pTexture);
    DEMOGfxFreeMEM2(myTex1.surface.imagePtr);
    DEMOGfxFreeMEM2(fsBuf);

    // attrib buffers
    GX2RDestroyBuffer(&vertexBuffer);
    GX2RDestroyBuffer(&texcoordBuffer);
    GX2RDestroyBuffer(&indexBuffer);
}

//----------------------------------------------------------
static BOOL SceneDraw(void)
{
    static f32 gray = 0.3f;
    static f32 mOffset[4] = { -0.5f, 0.0f, 0.0f, 0.0f };

    // 表示グリフの変更
    {
        DEMOPadRead();
        if(DEMOPadGetButtonDown(0) & DEMO_PAD_BUTTON_A)
        {
            sCharCode++;
        }

        if(DEMOPadGetButtonDown(0) & DEMO_PAD_BUTTON_B)
        {
            sCharCode--;
        }

        if(sCharCode < 0x20)
        {
            sCharCode = 0x7F;
        }

        if(sCharCode > 0x7F)
        {
            sCharCode = 0x20;
        }
    }

    // グリフの更新
    CreateGlyphTexture(sCharCode);

    // グリフテクスチャをポリゴンに貼って表示
    {
        DEMOGfxBeforeRender();

        GX2ClearColor(&DEMOColorBuffer, gray, gray, gray, 1.0f);
        GX2ClearDepthStencil(&DEMODepthBuffer, GX2_CLEAR_BOTH);

        DEMOGfxSetContextState();

        GX2SetShaders(&fetchShader, pVertexShader, pPixelShader);

        GX2UTSetAttributeBuffer(&vertexBuffer,   POSITION_BUFFER_IDX, 0);
        GX2UTSetAttributeBuffer(&texcoordBuffer, TEXCOORD_BUFFER_IDX, 0);

        mOffset[0] = 0.0f;
        GX2SetVertexUniformReg(offsetLoc, 1*4, mOffset);

        GX2SetPixelTexture(&myTex1, samplerLoc);
        GX2SetPixelSampler(&mySampler1, samplerLoc);

        GX2UTDrawIndexed(GX2_PRIMITIVE_TRIANGLE_STRIP, &indexBuffer);

        // 情報の表示
        {
            DEMOFontSetContextState();
            DEMOFontSetColor(1.0f, 0.0f, 0.0f, 1.0f);
            DEMOFontPrintf(3, 1, "SCFFND SAMPLE");
            DEMOFontPrintf(3, 3, "CODE = [%x]", sCharCode);
            DEMOFontPrintf(3, 4, "(Push A/B to change code)");
            DEMOGfxSetContextState();
        }

        DEMOGfxDoneRender();
    }

    return 1;
}

//----------------------------------------------------------
int main(int argc, char **argv)
{
    DEMOInit();
    DEMOTestInit(argc, argv);
    DEMOGfxInit(argc, argv);
    DEMOFontInit();

    SceneInit();

    while (DEMOIsRunning())
    {
        SceneDraw();
    }
    GX2DrawDone();

    SceneShutdown();

    DEMOFontShutdown();
    DEMOGfxShutdown();
    DEMOShutdown();

    return DEMOTestResult();
}
