﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//----------------------------------------
// McsHID Demo
//

#define NW_CONSOLE_ENABLE

#include <nw/demo.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>
#include <nw/mcs.h>

namespace {

const u32 SCREEN_WIDTH = 1280;
const u32 SCREEN_HEIGHT = 720;

// アロケーター
#define HEAP_SIZE ( 16 * 1024 * 1024 )
nw::ut::MemoryAllocator s_Allocator;

// デモシステム
nw::demo::DemoSystem* s_Demo = NULL;

// PrimitiveRenderer
nw::dev::PrimitiveRenderer* s_Renderer = NULL;
static nw::math::MTX44 s_ProjMtx;
static nw::math::MTX34 s_ViewMtx;

// mcsHID
const u32 MCS_INPUT_DEVICE_BUF_SIZE = 64 * 1024; //!< mcsHID のバッファサイズです。
u8* s_InputDeviceBuf = NULL;                     //!< mcsHID のバッファです。
bool s_Captured = false;                         //!< mcsHID ポーリングに成功したかを表すフラグです。

//---------------------------------------------------------------------------
//! @brief    mcsHID イベントハンドラクラスです。
//---------------------------------------------------------------------------
#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4512) // 'class' : assignment operator could not be generated
#endif
class KeyboardMouseInputEventHandler : public nw::mcs::InputEventHandler
{
public:
    //! @brief コンストラクタです。
    KeyboardMouseInputEventHandler( const u32 screenWidth, const u32 screenHeight )
      : nw::mcs::InputEventHandler(),
        m_PosX( 0.f ),
        m_PosY( 0.f ),
        m_ScreenWidth( screenWidth ),
        m_ScreenHeight( screenHeight ),
        m_Hold( false )
    {}

    //! @brief 現在のマウスポインタの X 座標位置を取得します。
    f32 GetPosX() const { return m_PosX; }
    //! @brief 現在のマウスポインタの Y 座標位置を取得します。
    f32 GetPosY() const { return m_PosY; }
    //! @brief 現在マウスボタンが押されているかを取得します。
    bool IsButtonHold() const { return m_Hold; }

private:
    //! @brief マウスが移動したときに呼び出されます。
    virtual void MouseMoveEvent(const nw::mcs::MouseEventArg& arg)
    {
        NW_LOG("MouseMoveEvent:\n");
        NW_LOG("  GetButton() = %x\n", arg.GetButton());
        NW_LOG("  GetMoveX() = %d\n", arg.GetMoveX());
        NW_LOG("  GetMoveY() = %d\n", arg.GetMoveY());
        NW_LOG("  GetWheel() = %d\n", arg.GetWheel());

        m_PosX += arg.GetMoveX();
        m_PosY += arg.GetMoveY();

        const f32 WIDTH = static_cast<f32>( m_ScreenWidth );
        const f32 HEIGHT = static_cast<f32>( m_ScreenHeight );

        if ( m_PosX < - WIDTH / 2.f )
        {
            m_PosX = - WIDTH / 2.f;
        }
        if ( m_PosX > WIDTH / 2.f )
        {
            m_PosX = WIDTH / 2.f;
        }
        if ( m_PosY < - HEIGHT / 2.f )
        {
            m_PosY = - HEIGHT / 2.f;
        }
        if ( m_PosY > HEIGHT / 2.f )
        {
            m_PosY = HEIGHT / 2.f;
        }
    }

    //! @brief マウスボタンが押されたときに呼び出されます。
    virtual void MouseDownEvent(const nw::mcs::MouseEventArg& arg)
    {
        NW_LOG("MouseDownEvent:\n");
        NW_LOG("  GetButton() = %x\n", arg.GetButton());
        NW_LOG("  GetMoveX() = %d\n", arg.GetMoveX());
        NW_LOG("  GetMoveY() = %d\n", arg.GetMoveY());
        NW_LOG("  GetWheel() = %d\n", arg.GetWheel());

        m_Hold = true;
    }

    //! @brief マウスボタンが離されたときに呼び出されます。
    virtual void MouseUpEvent(const nw::mcs::MouseEventArg& arg)
    {
        NW_LOG("MouseUpEvent:\n");
        NW_LOG("  GetButton() = %x\n", arg.GetButton());
        NW_LOG("  GetMoveX() = %d\n", arg.GetMoveX());
        NW_LOG("  GetMoveY() = %d\n", arg.GetMoveY());
        NW_LOG("  GetWheel() = %d\n", arg.GetWheel());

        m_Hold = false;
    }

    //! @brief マウスボタンがダブルクリックされたときに呼び出されます。
    virtual void MouseDoubleClickEvent(const nw::mcs::MouseEventArg& arg)
    {
        NW_LOG("MouseDoubleClickEvent:\n");
        NW_LOG("  GetButton() = %x\n", arg.GetButton());
        NW_LOG("  GetMoveX() = %d\n", arg.GetMoveX());
        NW_LOG("  GetMoveY() = %d\n", arg.GetMoveY());
        NW_LOG("  GetWheel() = %d\n", arg.GetWheel());
    }

    //! @brief マウスホイールが回転したときに呼び出されます。
    virtual void MouseWheelEvent(const nw::mcs::MouseEventArg& arg)
    {
        NW_LOG("MouseWheelEvent:\n");
        NW_LOG("  GetButton() = %x\n", arg.GetButton());
        NW_LOG("  GetMoveX() = %d\n", arg.GetMoveX());
        NW_LOG("  GetMoveY() = %d\n", arg.GetMoveY());
        NW_LOG("  GetWheel() = %d\n", arg.GetWheel());
    }

    //! @brief キーボードのキーが押されたときに呼び出されます。
    virtual void KeyDownEvent(const nw::mcs::KeyEventArg& arg)
    {
        NW_LOG("KeyDownEvent:\n");
        NW_LOG("  GetModifier() = %x\n", arg.GetModifier());
        NW_LOG("  GetAttribute() = %x\n", arg.GetAttribute());
        NW_LOG("  GetCategory() = %x\n", arg.GetCategory());
        NW_LOG("  GetKeyCode() = %x\n", arg.GetKeyCode());

        if (arg.GetKeyCode() == nw::mcs::KC_RALT)
        {
            NW_LOG("Stop input-capture.\n");
            nw::mcs::McsHID_StopCapture();
        }
    }

    //! @brief キーボードのキーが離されたときに呼び出されます。
    virtual void KeyUpEvent(const nw::mcs::KeyEventArg& arg)
    {
        NW_LOG("KeyUpEvent:\n");
        NW_LOG("  GetModifier() = %x\n", arg.GetModifier());
        NW_LOG("  GetAttribute() = %x\n", arg.GetAttribute());
        NW_LOG("  GetCategory() = %x\n", arg.GetCategory());
        NW_LOG("  GetKeyCode() = %x\n", arg.GetKeyCode());
    }

    //! @brief 文字が入力されたときに呼び出されます。
    virtual void CharEvent(const nw::mcs::CharEventArg& arg)
    {
        NW_LOG("CharEvent:\n");
        int c = arg.GetKeyChar();
        if (isascii(c))
        {
            NW_LOG("  GetKeyChar() = %02x (%c)\n", c, c);
        }
        else
        {
            NW_LOG("  GetKeyChar() = %02x\n", c);
        }
    }

    //! @brief ホスト上でキャプチャが開始されたときに呼び出されます。
    virtual void CaptureStartEvent(const nw::mcs::CaptureEventArg& arg)
    {
        NW_LOG("CaptureStartEvent:\n");
        NW_LOG("  (no additional info)\n");
        (void)arg;
    }

    //! @brief ホスト上でキャプチャが停止されたときに呼び出されます。
    virtual void CaptureStopEvent(const nw::mcs::CaptureEventArg& arg)
    {
        NW_LOG("CaptureStopEvent:\n");
        NW_LOG("  (no additional info)\n");
        (void)arg;
    }

    f32 m_PosX;
    f32 m_PosY;
    const u32 m_ScreenWidth;
    const u32 m_ScreenHeight;
    bool m_Hold;
};
#if defined(_WIN32)
#pragma warning(pop)
#endif

KeyboardMouseInputEventHandler s_InputEventHandler( SCREEN_WIDTH, SCREEN_HEIGHT ); //!< mcsHID イベントハンドラです。

} // namespace

//------------------------------------------------------------------------------
/*
 *  デモの初期化処理
 */
void
InitializeDemo()
{
    // mcs の初期化処理を行います。
    {
        nw::mcs::Mcs_Initialize();

        u32 errorCode = nw::mcs::Mcs_Open();
        NW_ASSERT( errorCode == nw::mcs::MCS_ERROR_SUCCESS );

        // mcsHID を初期化します。
        nw::mcs::McsHID_Initialize();
        if ( nw::mcs::McsHID_GetRegisteredBuffer() == NULL )
        {
            // 登録済みのバッファがない場合には、バッファを生成して登録します。
            s_InputDeviceBuf = reinterpret_cast<u8*>( s_Allocator.Alloc( MCS_INPUT_DEVICE_BUF_SIZE ) );
            nw::mcs::McsHID_RegisterBuffer( s_InputDeviceBuf, MCS_INPUT_DEVICE_BUF_SIZE );
        }

        // mcsHID イベントハンドラを登録します。
        nw::mcs::McsHID_RegisterEventHandler( &s_InputEventHandler );
    }

    s_Renderer = nw::dev::PrimitiveRenderer::GetInstance();

    // PrimitiveRenderer 用行列のセットアップ
    {
        const f32 width  = static_cast<f32>( s_Demo->GetWidth() );
        const f32 height = static_cast<f32>( s_Demo->GetHeight() );

        nw::math::MTX44Ortho(
            &s_ProjMtx,
            -width / 2.0f,  // left
             width / 2.0f,  // right
             height / 2.0f, // bottom
            -height / 2.0f, // top
            0.0f,          // near
            300.0f          // far
        );
        nw::math::MTX34Identity( &s_ViewMtx );
    }

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    // GL では座標を Cafe と同じ中心に設定する。
    nw::ut::DynamicCast<nw::demo::MouseWin*>( s_Demo->GetMouse() )->SetPointerCenterOrigin( true );
#endif
}

//------------------------------------------------------------------------------
/*
 *  デモの終了処理
 */
void
FinalizeDemo()
{
    // mcs の終了処理を行います。
    {
        if ( s_InputDeviceBuf != NULL && nw::mcs::McsHID_GetRegisteredBuffer() == s_InputDeviceBuf )
        {
            nw::mcs::McsHID_UnregisterBuffer();
            s_Allocator.Free( s_InputDeviceBuf );
        }

        nw::mcs::McsHID_Finalize();
        nw::mcs::Mcs_Finalize();
    }
}

//------------------------------------------------------------------------------
/*
 *  計算処理
 */
void
ProcCalc()
{
    // McsHID のポーリングを行います。
    if ( nw::mcs::McsHID_Polling() == nw::mcs::MCS_ERROR_SUCCESS )
    {
        s_Captured = true;
    }
    else
    {
        s_Captured = false;
    }
}

//------------------------------------------------------------------------------
/*
 *  描画処理（TV）
 */
void
ProcDrawTV()
{
    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
    graphics->SetBlendEnable( true );
    graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_NONE );

    {
        s_Renderer->SetProjectionMtx( &s_ProjMtx );
        s_Renderer->SetViewMtx( &s_ViewMtx );

        graphics->SetDepthTestEnable( true );
        graphics->SetDepthWriteEnable( true );

        s_Renderer->Begin();
        {
            const nw::math::VEC3 pointPos( s_InputEventHandler.GetPosX(), s_InputEventHandler.GetPosY(), 0.f );

            const nw::math::VEC3 circlePos[4] = {
                nw::math::VEC3( -350.f, -200.f, 0.0f ),
                nw::math::VEC3( -350.f,  200.f, 0.0f ),
                nw::math::VEC3(  350.f, -200.f, 0.0f ),
                nw::math::VEC3(  350.f,  200.f, 0.0f )
            };

            const f32 radius = 25.f;
            const f32 radiusClicked = 30.f;
            const nw::ut::Color4u8 color = nw::demo::SRGBToLinear( nw::ut::Color4u8( 40, 80, 255, 255 ) );
            const nw::ut::Color4u8 colorClicked = nw::demo::SRGBToLinear( nw::ut::Color4u8( 255, 40, 20, 255 ) );

            for ( int i = 0; i < 4; ++i )
            {
                if ( s_Captured && s_InputEventHandler.IsButtonHold() && ( pointPos - circlePos[i] ).Length() <= radius  )
                {
                    s_Renderer->DrawSphere8x16( circlePos[i], radiusClicked, colorClicked );
                }
                else
                {
                    s_Renderer->DrawSphere8x16( circlePos[i], radius, color );
                }
            }
        }
        s_Renderer->End();

        graphics->SetDepthTestEnable( false );
        graphics->SetDepthWriteEnable( false );

        if ( s_Captured )
        {
            s_Demo->DrawPointerCursor(
                s_InputEventHandler.GetPosX() / ( s_Demo->GetWidth() / 2.f ),
                s_InputEventHandler.GetPosY() / ( s_Demo->GetHeight() / 2.f ),
                static_cast<f32>( s_Demo->GetWidth() ),
                static_cast<f32>( s_Demo->GetHeight() )
            );
        }
    }
}

//------------------------------------------------------------------------------
/*
 *  1 フレーム分の処理
 */
void
ProcFrame()
{
    // フレームの開始処理
    s_Demo->BeginFrame();

    // 計算処理
    s_Demo->GetMeterCalc().BeginMeasure();
    ProcCalc();
    s_Demo->GetMeterCalc().EndMeasure();

    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
    graphics->LockDrawContext();
    {
        // TV 描画
        {
            s_Demo->SetViewport( 0.f, 0.f, static_cast<f32>( s_Demo->GetWidth() ), static_cast<f32>( s_Demo->GetHeight() ), 0.f, 1.f );
            s_Demo->SetScissor( 0.f, 0.f, static_cast<f32>( s_Demo->GetWidth() ), static_cast<f32>( s_Demo->GetHeight() ) );

            s_Demo->GetMeterGPU().BeginMeasure();
            s_Demo->GetMeterDraw().BeginMeasure();
            {
                s_Demo->ClearFrameBuffers();
                ProcDrawTV();
            }
            s_Demo->GetMeterDraw().EndMeasure();
            s_Demo->GetMeterGPU().EndMeasure();

            // 負荷計測メーターを描画します。
            s_Demo->DrawLoadMeters();
        }
    }
    graphics->UnlockDrawContext();

    // バッファのスワップ
    s_Demo->SwapBuffer();

    // フレームの終了処理
    s_Demo->EndFrame();

    //  VBlank 待ち
    s_Demo->WaitForVBlank();
}

//------------------------------------------------------------------------------
/*
 *  main() 関数
 */
int
NwDemoMain(int argc, char **argv)
{
#if defined(NW_RELEASE)
    NW_LOG("NintendoWare --- Can't execute this demo as release-build.\n");
    for(;;) { } // Release 版では ホスト通信機能 が利用できないため実行できません。
#endif

    // アロケーターの初期化
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    void* addr = malloc( HEAP_SIZE );
#else
    void* addr = MEMAllocFromDefaultHeap( HEAP_SIZE );
#endif
    s_Allocator.Initialize( addr, HEAP_SIZE );

    // デモシステムの作成
    nw::demo::DemoSystem::CreateArg arg;
    arg.allocator = &s_Allocator;
    arg.width = SCREEN_WIDTH;
    arg.height = SCREEN_HEIGHT;
    arg.usePointerCursor = true;
    arg.fileDeviceManagerInitialize = true;
    arg.drawMeter = true;
#if defined(NW_PLATFORM_CAFE)
    // デモ文字列描画用のフォントのパスを指定します。
    arg.fontPath = "common/fonts/nintendo_NTLG-DB_002.bffnt";
    // デモ文字列描画用のシェーダーのパスを指定します。（ PC 版では不要です。 ）
    arg.fontShaderPath = "common/shaders/font_BuildinShader.gsh";
    // PrimitiveRenderer で用いるシェーダーのパスを指定します。（ PC 版では不要です。 ）
    arg.primitiveRendererShaderPath = "common/shaders/dev_PrimitiveRenderer.gsh";
#else
    // デモ文字列描画用のフォントのパスを指定します。
    arg.fontPath = "common/fonts/nintendo_NTLG-DB_002_Nw4f.bffnt";
    arg.primitiveRendererInitialize = true;
#endif
    s_Demo = new( s_Allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) ) nw::demo::DemoSystem( arg );

    // デモシステムの初期化
    s_Demo->Initialize();

    // グラフィックシステムの初期化
    s_Demo->InitializeGraphicsSystem();

    // デモの初期化
    InitializeDemo();

    // メインループ
    NW_LOG("Start demo.\n");
    while ( !s_Demo->IsExiting() )
    {
        ProcFrame();
    }

    // 終了処理
    FinalizeDemo();
    s_Demo->FinalizeGraphicsSystem();
    s_Demo->Finalize();
    nw::ut::SafeFree( s_Demo, &s_Allocator );

    s_Allocator.Finalize();
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    free( addr );
#else
    MEMFreeToDefaultHeap( addr );
#endif

    NW_LOG("End demo.\n");
    (void)argc;
    (void)argv;
    return 0;
}

