﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*

  WordWrapping: レイアウトでワードラッピングを行うサンプルです。

 *---------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/demo.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>

#include <nw/lyt.h>
#include <nw/font.h>

#define ARRAY_LENGTH(a) (sizeof(a) / sizeof(a[0]))

//------------------------------------------------------------------------------
//   変数の宣言
//------------------------------------------------------------------------------

// ワードラップ処理の対象となるソース文字列です。
const char16 s_Text[] =
    L"nw::lyt::WordWrapping Demo\n"
    L"\n"
    L"Creating a Layout\n"
    L"\n"
    L"As a first step, here we provide a description of the basic work "
    L"flow, which involves:\n"
    L"\n"
    L"* Creating a new layout\n"
    L"* Positioning the pane\n"
    L"* Previewing in the Viewer\n"
    L"* Saving the data\n"
    L"\n"
    L"Creating a New Layout\n"
    L"\n"
    L"Start LayoutEditor and create a new layout file by selecting "
    L"File > Create New > Layout."
    L"A window with the title \"Untitled\" will appear on the screen. "
    L"This is called the Layout Window."
    L"With LayoutEditor, you create game screen layouts by "
    L"positioning images in the Layout Window.\n"
    L"\n"
    L"Description of the Layout Window Screen\n"
    L"\n"
    L"Look at the Layout Window and notice that a black square drawn "
    L"on the screen."
    L"This square is called the layout screen region (A)."
    L"The layout screen region corresponds to the screen shown on the "
    L"actual hardware."
    L"The red line drawn in the center of the screen is the coordinate "
    L"origin of the layout screen region (B). "
    L"(The layout coordinate system is a three-dimensional coordinate "
    L"system consisting of a two-dimensional X-Y axis plus a "
    L"one-dimensional Z axis.)\n"
    L"\n"
    L"Moving the Display Area (View)\n"
    L"\n"
    L"Pressing the space bar on the Layout window changes the figure "
    L"of the mouse cursor to a hand symbol (A)."
    L"At this point, the editor is in View Change mode ."
    L"While in this View Change mode, the display region in the window "
    L"can be changed by dragging the mouse (B).\n"
    L"\n"
    L"Changing Scale of the Display Area (View)\n"
    L"\n"
    L"Confirm that there is a combo box displaying at 100.0% and a "
    L"slider with + and - icons at the lower right of the Layout "
    L"window (A)."
    L"You can operate these to zoom in and zoom out.\n"
    L"\n"
    L"Create a New Null Pane\n"
    L"\n"
    L"Right-click in the window and select Create Pane > Null Pane.\n"
    L"\n"
    L"A green square appears in the Layout window. This is a Null pane."
    L"A pane is the smallest component of the layout data, representing "
    L"an individual image on the screen."
    L"Use the editor to position panes and create layouts.\n"
    L"\n"
    L"In addition to Null panes, there are also four other types of panes."
    L"A Null pane serves only an auxiliary purpose and is not displayed "
    L"on the actual game screen.";

// アロケーター
const u32 c_HeapSize = 16 * 1024 * 1024;
nw::ut::MemoryAllocator s_Allocator;

// デモシステム
nw::demo::DemoSystem* s_Demo = NULL;

// グラフィックスコンテキスト
nw::gfnd::GraphicsContext s_GraphicsContext;

// レイアウト
nw::lyt::Layout* s_Layout = NULL;
u8* s_LayoutArchiveBinary = NULL;
// レイアウトで使用するフォント
nw::font::ResFont* s_Font = NULL;
u8* s_FontBinary = NULL;
// レイアウトアーカイブリソースアクセサ
nw::lyt::ArcResourceAccessor* s_ArcResourceAccessor = NULL;
// レイアウトグラフィックスリソース
nw::lyt::GraphicsResource s_GraphicsResource;
// 描画時に使用する情報
nw::lyt::DrawInfo s_DrawInfo;
#if defined(NW_PLATFORM_CAFE)
// レイアウトシェーダセットアップヘルパー(Cafeでのみ必要)
nw::lyt::ShaderSetupHelper s_ShaderSetupHelper;
#endif

#if defined(NW_PLATFORM_CAFE)
static const char* const DEMO_FONT_FILENAME = "nintendo_NTLG-DB_002.bffnt";
#else
static const char* const DEMO_FONT_FILENAME = "nintendo_NTLG-DB_002_Nw4f.bffnt";
#endif

//---------------------------------------------------------------------------
// 一行に収まる文字列の範囲を求めます。
//---------------------------------------------------------------------------
class MyWordWrapCallback : public nw::font::WordWrapCallback
{
public:
    MyWordWrapCallback(nw::font::WideTextWriter* pTextWriter)
        : m_pTextWriter(pTextWriter)
    {
    }

    // 与えられた文字列のうち一行の表示に収まる範囲を求めます。
    //
    // 一行に表示できる文字列の範囲は WideTextWriter::SetWidthLimit() で
    // 表示幅を設定したのち、 WideTextWriter::FindPosOfWidthLimit() で計算することができます。
    //
    virtual const char16* GetLineBreakLimit(const char16* str, const char16* end)
    {
        // 文字列の表示幅を求めます。
        const char16* lineBreak = m_pTextWriter->FindPosOfWidthLimit(str, end - str);
        return lineBreak;
    }

    nw::font::WideTextWriter* m_pTextWriter;
};

class MyTextSearcher : public nw::lyt::TextSearcher
{
public:
    MyTextSearcher(nw::ut::IAllocator* pAllocator)
        : m_pAllocator(pAllocator)
        , m_WordWrappedText(0)
        , m_WordWrappedTextSize(0)
    {
    }

    virtual ~MyTextSearcher()
    {
        m_pAllocator->Free(m_WordWrappedText);
    }

    void AllocWordWrappedText(nw::lyt::TextBox* pTextBox)
    {
        if (m_WordWrappedText)
        {
            m_pAllocator->Free(m_WordWrappedText);
        }

        nw::font::WideTextWriter writer;
        pTextBox->SetupTextWriter(&writer);

        // ワードラップ処理された文字列を格納するバッファとして、
        // ソース文字列＋挿入されうる改行コード分のメモリを確保します。
        u32 numLines = static_cast<u32>(
            pTextBox->GetSize().height / writer.GetLineHeight());
        m_WordWrappedTextSize = ARRAY_LENGTH(s_Text) + numLines;

        // Note: フラグメンテーションを避けるため、
        // 実際にはヒープの異なるサイドから確保するなどした方が良いでしょう。
        // (nn::fnd::ExpHeap::Allocate()でalignment引数に負の値を指定します)
        m_WordWrappedText = static_cast<char16*>(
            m_pAllocator->Alloc(sizeof(char16) * m_WordWrappedTextSize));
    }

    const char16* GetMessage(const char* messageId)
    {
        if (messageId == NULL)
        {
            return NULL;
        }
        else if (strcmp(messageId, "Text") == 0)
        {
            return s_Text;
        }
        else
        {
            return L"Unexpected messageId";
        }
    }

    // ワードラップ処理が必要かどうか、拡張ユーザデータを調べます。
    bool CheckWordWrappingRequest(const nw::lyt::TextBox* pTextBox)
    {
        const nw::lyt::res::ExtUserData* pUserData = pTextBox->GetExtUserDataArray();

        for (u32 i = 0; i <pTextBox->GetExtUserDataNum(); ++i)
        {
            const nw::lyt::res::ExtUserData& data = pUserData[i];

            if (data.GetType() == nw::lyt::EXTUSERDATATYPE_INT &&
                strcmp(data.GetName(), "WordWrapping") == 0 &&
                data.GetNum() > 0)
            {
                return (data.GetIntArray()[0] > 0);
            }
        }

        return false;
    }

    virtual void SearchText(
        TextInfo* textInfo,
        const char* messageId,
        nw::lyt::Layout* layout,
        nw::lyt::TextBox* textBox,
        nw::lyt::Layout* resourceLayout)
    {
        NW_UNUSED_VARIABLE(layout);
        NW_UNUSED_VARIABLE(resourceLayout);

        textInfo->text = GetMessage(messageId);
        textInfo->textLength = 0;

        // ユーザデータでワードラップ処理が指定された場合には、
        // メッセージにワードラップ処理を施します。
        if (textInfo->text && this->CheckWordWrappingRequest(textBox))
        {
            AllocWordWrappedText(textBox);

            // TextWriter に TextBox の描画に必要な設定を行います。
            // TextBox::SetupTextWriter() により WideTextWriter::SetWidthLimit()
            // が設定されるので、 WideTextWriter::GetLineBreak() により
            // 一行に表示できる文字列の範囲を得ることができます。
            nw::font::WideTextWriter textWriter;
            textBox->SetupTextWriter(&textWriter);

            // 一行に収まる文字列の範囲を判定するコールバックです。
            MyWordWrapCallback lineBreakFunc(&textWriter);

            // ペインの大きさを超えないように行数を指定します。
            nw::font::WordWrapConfig config;
            u32 numLines = static_cast<u32>(textBox->GetSize().height / textWriter.GetLineHeight());
            config.SetMaxLineNum(numLines);

            // ワードラップ処理を行います。
            nw::font::WordWrapping::CalcWordWrapping(
                /* writeSize */ NULL,
                m_WordWrappedText,
                m_WordWrappedTextSize,
                textInfo->text,
                lineBreakFunc,
                config);

            // ワードラップ処理した文字列をTextBoxに設定します。
            textInfo->text = m_WordWrappedText;
        }
    }

protected:
    nw::ut::IAllocator* m_pAllocator;

    // ワードラップ処理を施した文字列を格納するバッファです。
    char16* m_WordWrappedText;

    // s_WordWrappedTextに格納できる文字数（ヌル文字を含む）です。
    u32 m_WordWrappedTextSize;
};

//------------------------------------------------------------------------------
//   レイアウトの処理
//------------------------------------------------------------------------------
/*
 *  レイアウトの初期化処理
 */
void
InitializeLayout()
{
    // レイアウトライブラリの初期化
    nw::lyt::Initialize(&s_Allocator);

    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

    // グラフィックスリソースの初期化
    {
        u8* pFontBinary = NULL;
        u8* pLayoutBinary = NULL;
#if defined(NW_PLATFORM_CAFE)
        // Cafeではシェーダバイナリを読み込んで初期化する必要がある
        nw::ut::MemoryRange fontShaderBinary;
        nw::ut::MemoryRange lytShaderBinary;
        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = "font_BuildinShader.gsh";
            loadArg.allocator = &s_Allocator;
            pFontBinary = fileSystem->Load( loadArg );
            fontShaderBinary = nw::ut::MakeMemoryRange(pFontBinary, loadArg.readSize);
        }
        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = "lyt_BuildinShader.gsh";
            loadArg.allocator = &s_Allocator;
            pLayoutBinary = fileSystem->Load( loadArg );
            lytShaderBinary = nw::ut::MakeMemoryRange(pLayoutBinary, loadArg.readSize);
        }
#endif
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        {
#if defined(NW_PLATFORM_CAFE)
            s_GraphicsResource.Setup(4096, fontShaderBinary);
            s_ShaderSetupHelper.Setup(&s_GraphicsResource, lytShaderBinary);
#else
            s_GraphicsResource.Setup(4096);
#endif
        }

        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
        nw::ut::SafeFree(pFontBinary, &s_Allocator);
        nw::ut::SafeFree(pLayoutBinary, &s_Allocator);
    }

    // レイアウトアーカイブの読み込みとリソースアクセサへのアタッチ
    {
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "WordWrapping.arc";
        loadArg.allocator = &s_Allocator;
        loadArg.alignment = nw::lyt::ARCHIVE_RESOURCE_ALIGNMENT;
        s_LayoutArchiveBinary = fileSystem->Load( loadArg );

        s_ArcResourceAccessor = new( s_Allocator.Alloc( sizeof( nw::lyt::ArcResourceAccessor ) ) ) nw::lyt::ArcResourceAccessor();
        bool ret = s_ArcResourceAccessor->Attach(s_LayoutArchiveBinary, ".");
        NW_ASSERT(ret);
    }

    // フォントの読み込み
    {
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "sample.bffnt";
        loadArg.allocator = &s_Allocator;
        loadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
        s_FontBinary = fileSystem->Load( loadArg );
    }

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    // フォントの初期化
    {
        s_Font = new( s_Allocator.Alloc( sizeof( nw::font::ResFont ) ) ) nw::font::ResFont();
        // リソースのセット。内部でテクスチャを初期化するため、コンテキストのロックが必要
        bool ret = s_Font->SetResource(s_FontBinary);
        NW_ASSERT(ret);

        // リソースアクセサにアタッチ
        s_ArcResourceAccessor->RegistFont("sample.bffnt", s_Font);
    }

    // レイアウトの初期化
    {
        MyTextSearcher textSearcher(&s_Allocator);

        s_Layout = new( s_Allocator.Alloc( sizeof( nw::lyt::Layout ) ) ) nw::lyt::Layout();
        // レイアウトファイル名とリソースアクセサを指定してレイアウトを構築
        s_Layout->BuildWithName("WordWrapping.bflyt", s_ArcResourceAccessor, NULL, &textSearcher);
    }

    // 描画に使用する情報の設定
    {
        nw::math::MTX44 projection;
        nw::ut::Rect layoutRect = s_Layout->GetLayoutRect();
        nw::math::MTX44Ortho(&projection, layoutRect.left, layoutRect.right, layoutRect.bottom, layoutRect.top, 0.0f, 300.0f);
        nw::math::MTX34 view;
        // zが1の位置からxy平面上のポリゴンを見る設定
        nw::math::VEC3 pos(0.f, 0.f, 1.f);
        nw::math::VEC3 up(0.f, 1.f, 0.f);
        nw::math::VEC3 target(0.f, 0.f, 0.f);
        nw::math::MTX34LookAt(&view, &pos, &up, &target);

        s_DrawInfo.SetGraphicsResource(&s_GraphicsResource);
        s_DrawInfo.SetProjectionMtx(projection);
        s_DrawInfo.SetViewMtx(view);
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

//------------------------------------------------------------------------------
/*
 *  レイアウトの更新処理
 */
void
UpdateLayout()
{
    // アニメーションを適用、同時にアニメを全て1フレーム進める
    s_Layout->AnimateAndUpdateAnimFrame();

    // 描画時に使用するマトリックスを計算
    s_Layout->CalculateMtx(s_DrawInfo);
}

//------------------------------------------------------------------------------
/*
 *  レイアウトの描画処理
 */
void
DrawLayout()
{
    // 描画設定
    // レイアウトは基本的にデプステストなし、カリングなしで描画する
    // これらの設定はレイアウト内では行っていないため、Drawを呼び出す前に設定しておく必要がある
    // 逆にアルファテスト及びブレンドの設定はレイアウト内で書き換えるため、注意が必要。
    s_GraphicsContext.SetDepthEnable(false, false);
    s_GraphicsContext.SetCullingMode(nw::gfnd::Graphics::CULLING_MODE_NONE);
    s_GraphicsContext.Apply();

    // レイアウトの描画
    s_Layout->Draw(s_DrawInfo);
}

//------------------------------------------------------------------------------
/*
 *  レイアウトの終了処理
 */
void
FinalizeLayout()
{
    // レイアウトの破棄、属するアニメータも一緒に破棄される
    nw::ut::SafeFreeWithDestruct( s_Layout, &s_Allocator );

    // リソースアクセサの破棄
    nw::ut::SafeFreeWithDestruct( s_ArcResourceAccessor, &s_Allocator );
    nw::ut::SafeFree(s_LayoutArchiveBinary, &s_Allocator);

    // フォントの破棄
    nw::ut::SafeFreeWithDestruct( s_Font, &s_Allocator );
    nw::ut::SafeFree(s_FontBinary, &s_Allocator);

    // グラフィックスリソースの終了処理
    s_GraphicsResource.Finalize();

#if defined(NW_PLATFORM_CAFE)
    // シェーダの破棄
    s_ShaderSetupHelper.Finalize();
#endif
}

//------------------------------------------------------------------------------
//   デモフレームワークの処理
//------------------------------------------------------------------------------
/*
 *  1 フレーム分の処理
 */
void
ProcFrame()
{
    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
    NW_NULL_ASSERT(graphics);

    // フレームの開始処理
    s_Demo->BeginFrame();

    // デモパッドの更新
    s_Demo->UpdatePad();

    // レイアウトの計算処理
    s_Demo->GetMeterCalc().BeginMeasure();
    UpdateLayout();
    s_Demo->GetMeterCalc().EndMeasure();

    graphics->LockDrawContext();
    {
        // TV 描画
        {
#if defined(NW_PLATFORM_CAFE)
            // GPUのreadキャッシュを全域クリアする
            GX2Invalidate(static_cast<GX2InvalidateType>(GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_TEXTURE | GX2_INVALIDATE_CONSTANT_BUFFER | GX2_INVALIDATE_SHADER),
                0x00000000, 0xffffffff );
#endif

            // フレームバッファのクリア
            s_Demo->ClearFrameBuffers();

            s_Demo->SetViewport( 0.f, 0.f, static_cast<f32>( s_Demo->GetWidth() ), static_cast<f32>( s_Demo->GetHeight() ), 0.f, 1.f );
            s_Demo->SetScissor( 0.f, 0.f, static_cast<f32>( s_Demo->GetWidth() ), static_cast<f32>( s_Demo->GetHeight() ) );

            // このデモのGPU負荷計測では、レイアウトの処理のみ計測する
            // 実際にはフレームバッファのクリア、スキャンバッファへの転送などの処理も必要だが、その処理は表示されない。
            s_Demo->GetMeterGPU().BeginMeasure();
            s_Demo->GetMeterDraw().BeginMeasure();
            {
                // レイアウトの描画処理
                DrawLayout();
            }
            s_Demo->GetMeterDraw().EndMeasure();
            s_Demo->GetMeterGPU().EndMeasure();

            // 負荷計測メーターを描画します。
            s_Demo->DrawLoadMeters();
        }
    }
    graphics->UnlockDrawContext();

    // バッファのスワップ
    s_Demo->SwapBuffer();

    // フレームの終了処理
    s_Demo->EndFrame();

    //  VBlank 待ち
    s_Demo->WaitForVBlank();
}

//------------------------------------------------------------------------------
/*
 *  main() 関数
 */
int
NwDemoMain(int argc, char **argv)
{
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);

    //============== 初期化処理 ==============//
    // アロケーターの初期化
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    void* addr = malloc( c_HeapSize );
#else
    void* addr = MEMAllocFromDefaultHeap( c_HeapSize );
#endif
    NW_NULL_ASSERT(addr);
    s_Allocator.Initialize( addr, c_HeapSize );

    // デモシステムの作成
    nw::demo::DemoSystem::CreateArg arg;
    arg.allocator = &s_Allocator;
    arg.waitVBlank = 1;
    arg.width = 1280;
    arg.height = 720;
    arg.drawMeter = true;
    // デモ文字列描画用のフォントのパスを指定します。
    arg.fontPath = DEMO_FONT_FILENAME;
#if defined(NW_PLATFORM_CAFE)
    // デモ文字列描画用のシェーダーのパスを指定します。（ PC 版では不要です。 ）
    arg.fontShaderPath = "font_BuildinShader.gsh";
    // PrimitiveRenderer で用いるシェーダーのパスを指定します。（ PC 版では不要です。 ）
    arg.primitiveRendererShaderPath = "dev_PrimitiveRenderer.gsh";
#else
    arg.primitiveRendererInitialize = true;
#endif
    arg.fileDeviceManagerInitialize = true;

    s_Demo = new( s_Allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) ) nw::demo::DemoSystem( arg );
    NW_NULL_ASSERT(s_Demo);
    // デモシステムの初期化
    s_Demo->Initialize();

    // グラフィックシステムの初期化
    s_Demo->InitializeGraphicsSystem();

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    // GL では座標を Cafe と同じ中心に設定する。
    nw::ut::DynamicCast<nw::demo::WPadWin*>( s_Demo->GetWPad() )->SetPointerCenterOrigin( true );
    nw::ut::DynamicCast<nw::demo::MouseWin*>( s_Demo->GetMouse() )->SetPointerCenterOrigin( true );
#endif

    // レイアウトの初期化処理
    InitializeLayout();

    //============== メインループ ==============//
    NW_LOG("Start demo.\n");
    // L,R,X同時押しで抜ける
    while ( ! s_Demo->GetPad()->IsHoldAll( nw::demo::Pad::MASK_L | nw::demo::Pad::MASK_R | nw::demo::Pad::MASK_X ) && ! s_Demo->IsExiting())
    {
        ProcFrame();
    }

    //============== 終了処理 ==============//
    // レイアウトの終了処理
    FinalizeLayout();

    // グラフィックシステムの終了処理
    s_Demo->FinalizeGraphicsSystem();
    // デモシステムの終了処理
    s_Demo->Finalize();
    nw::ut::SafeFree( s_Demo, &s_Allocator );

    // アロケータの破棄
    s_Allocator.Finalize();
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    free( addr );
#else
    MEMFreeToDefaultHeap( addr );
#endif

    NW_LOG("End demo.\n");
    return 0;
}
