﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3ddemo_DemoUtility.h>

#include <string>
#include <algorithm>
#include <map>

#include <cstdio>
#include <nw/g3d/fnd/g3d_EglUtility.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>

#include <nw/g3d/ut/g3d_Inlines.h>
#include <nw/g3d/ut/g3d_Perf.h>
#include <nw/g3d/math/g3d_MathCommon.h>
#include <nw/g3d/math/g3d_Vector3.h>
#include <nw/g3d/math/g3d_Matrix34.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>

#include <nw/g3d/edit/g3d_EditDefs.h>

namespace nw { namespace g3d { namespace demo {

namespace
{
    const int s_TblModeToSizeTV[] = {
        0, 0,
        640, 480,
        854, 480,
        1280, 720,
        1280, 720,
        1920, 1080
    };
    GX2TVRenderMode s_ModeTV = GX2_TV_RENDER_NONE;
    GX2SurfaceFormat s_FormatTV = GX2_SURFACE_FORMAT_INVALID;
    std::map<void*, void*> s_MemMap;
}

//--------------------------------------------------------------------------------------------------
// Memory

#define RoundUp(x, y) (((unsigned long)(x) + (y) - 1) & ~((y) - 1))

void* AllocMem1(size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/)
{
    NW_G3D_ASSERT(IsPowerOfTwo(alignment));
    void* ptr = NULL;
    if (alignment == 0 || alignment == 1)
    {
        ptr = malloc(size);
    }
    else
    {
        ptr = malloc(size + alignment);
    }
    NW_G3D_ASSERTMSG(ptr != NULL, "AllocMem1 failed.\n");
    void* alignedPtr = NULL;
    if (alignment == 0 || alignment == 1)
    {
        alignedPtr = ptr;
    }
    else
    {
        alignedPtr = reinterpret_cast<void*>(RoundUp(ptr, alignment));
    }
    //printf("mem1 alloc size %lu alignment %lu ptr %lx alignedPtr %lx\n", size, alignment, (unsigned long)ptr, (unsigned long)alignedPtr);
    s_MemMap.insert( std::pair<void*, void*>(alignedPtr, ptr) );
    return alignedPtr;
}

void FreeMem1(void* ptr)
{
    std::map<void*, void*>::iterator iter;
    iter = s_MemMap.find(ptr);
    if (iter != s_MemMap.end())
    {
        //printf("mem1 free ptr %lx alignedPtr %lx\n", (unsigned long)iter->second, (unsigned long)ptr);
        free(iter->second);
        s_MemMap.erase(iter);
    }
}

void* AllocMem2(size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/)
{
    NW_G3D_ASSERT(IsPowerOfTwo(alignment));
    void* ptr = NULL;
    if (alignment == 0 || alignment == 1)
    {
        ptr = malloc(size);
    }
    else
    {
        ptr = malloc(size + alignment);
    }
    NW_G3D_ASSERTMSG(ptr != NULL, "AllocMem2 failed.\n");
    void* alignedPtr = NULL;
    if (alignment == 0 || alignment == 1)
    {
        alignedPtr = ptr;
    }
    else
    {
        alignedPtr = reinterpret_cast<void*>(RoundUp(ptr, alignment));
    }
    //printf("mem2 alloc size %lu alignment %lu ptr %lx alignedPtr %lx\n", size, alignment, (unsigned long)ptr, (unsigned long)alignedPtr);
    s_MemMap.insert( std::pair<void*, void*>(alignedPtr, ptr) );
    return alignedPtr;
}

void FreeMem2(void* ptr)
{
    std::map<void*, void*>::iterator iter;
    iter = s_MemMap.find(ptr);
    if (iter != s_MemMap.end())
    {
        //printf("mem2 free ptr %lx alignedPtr %lx\n", (unsigned long)iter->second, (unsigned long)ptr);
        free(iter->second);
        s_MemMap.erase(iter);
    }
}

//--------------------------------------------------------------------------------------------------
// File
void* LoadFile(const char* path, size_t* pSize /*= NULL*/, size_t alignment /*= DEFAULT_ALIGNMENT*/)
{
    NW_G3D_ASSERT_NOT_NULL(path);
    NW_G3D_ASSERT(IsPowerOfTwo(alignment));

    alignment = std::max<size_t>(alignment, PPC_IO_BUFFER_ALIGN);

    FILE *fp = fopen(path, "rb");
    NW_G3D_ASSERTMSG(fp != NULL, "OpenFile failed(%s).\n", path);

    long size = 0;
    fseek(fp, 0, SEEK_END);
    size = ftell(fp);
    NW_G3D_ASSERTMSG(size != 0, "file size is 0.\n");
    fseek(fp, 0, SEEK_SET);

    // テキストファイル NUL 終端させるために多めにメモリ確保して 0 で埋める。
    void* ptr = AllocMem2(size + DEFAULT_ALIGNMENT, alignment);
    size_t read_size = fread(ptr, size, 1, fp);
    NW_G3D_ASSERTMSG(read_size == 1, "ReadFile failed(%s).\n", path);

    memset(nw::g3d::AddOffset(ptr, size), 0, DEFAULT_ALIGNMENT);
    fclose(fp);

    size += DEFAULT_ALIGNMENT;

    if (pSize)
    {
        *pSize = size;
    }

    NW_G3D_UNUSED(read_size);

    return ptr;
}

//--------------------------------------------------------------------------------------------------
// Display

void InitDisplay(const InitDisplayArg& arg)
{
    if (arg.modeTV != GX2_TV_RENDER_NONE)
    {
        s_ModeTV = arg.modeTV;
        s_FormatTV = arg.formatTV;

        nw::g3d::SetSwapInterval(arg.swapInterval);
    }
}

void CopyOut(const nw::g3d::GfxColorBuffer* pColorBuffer, int screenWidth, int screenHeight)
{
    int w;
    int h;
    NW_G3D_ASSERT_NOT_NULL(pColorBuffer);

    // 画面全体塗りつぶし
    //glBindFramebuffer(GL_FRAMEBUFFER, 0);
    glScissor(0, 0, screenWidth, screenHeight);
    glClearColor(0.2f, 0.2f, 0.3f, 1.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    float wRaito = static_cast<float>(s_TblModeToSizeTV[s_ModeTV * 2])/static_cast<float>(screenWidth);
    float hRaito = static_cast<float>(s_TblModeToSizeTV[s_ModeTV * 2 + 1])/static_cast<float>(screenHeight);

    if (wRaito < 1.0f && hRaito < 1.0f)
    {
        w = s_TblModeToSizeTV[s_ModeTV * 2];
        h = s_TblModeToSizeTV[s_ModeTV * 2 + 1];
    }
    else if (wRaito > hRaito)
    {
        w = screenWidth;
        h = static_cast<int>(static_cast<float>(s_TblModeToSizeTV[s_ModeTV * 2 + 1] * screenWidth) / static_cast<float>(s_TblModeToSizeTV[s_ModeTV * 2]));
    }
    else
    {
        w = static_cast<int>(static_cast<float>(s_TblModeToSizeTV[s_ModeTV * 2] * screenHeight) / static_cast<float>(s_TblModeToSizeTV[s_ModeTV * 2 + 1]));
        h = screenHeight;
    }

    int offsetW = (screenWidth - w) >> 1;
    int offsetH = (screenHeight - h) >> 1;

    nw::g3d::fnd::detail::SystemContext& system = nw::g3d::fnd::detail::GetSystemContext();
    const nw::g3d::fnd::detail::SystemShader* shader = &system.swapCopyShader;
    if (pColorBuffer->GetGX2ColorBuffer()->surface.aa != GX2_AA_MODE_1X)
    {
        shader = &system.swapResolveShader;
    }
    glUseProgram(shader->hShader);

    //glBindFramebuffer(GL_FRAMEBUFFER, 0);
    glViewport(offsetW, offsetH, w, h);
    glScissor(offsetW, offsetH, w, h);

    system.BindTexture(pColorBuffer->handle, shader->texUnit, 0);
    glUniform1f(shader->locations[shader->LOCATION_GAMMA], 1.0f);
    glDisable(GL_CULL_FACE);
    glBindVertexArray(0);
    system.DrawScreenQuad();

    NW_G3D_GL_ASSERT();
}

}}} // namespace nw::g3d::demo
