﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3ddemo_DemoUtility.h>

#include <string>
#include <algorithm>

#ifdef _WIN32
#include <cstdio>
#include <io.h>
#include <nw/g3d/fnd/g3d_WinUtility.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>
#else
#include <cafe/mem.h>
#include <cafe/pad.h>
#endif

#if defined(NW_USE_CPU_PROFILER) && !defined(_WIN32)

#include <cafe/profiler.h>

#endif

#include <nw/g3d/ut/g3d_Inlines.h>
#include <nw/g3d/ut/g3d_Perf.h>
#include <nw/g3d/math/g3d_MathCommon.h>
#include <nw/g3d/math/g3d_Vector3.h>
#include <nw/g3d/math/g3d_Matrix34.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>

#include <nw/g3d/edit/g3d_EditDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO
#if NW_G3D_IS_HOST_WIN
#define NOMINMAX
#define WIN32_LEAN_AND_MEAN
#include <winsock.h>
#pragma comment(lib, "wsock32.lib ")
#else
#include <cafe/hio.h>
#endif
#endif

namespace nw { namespace g3d { namespace demo {

namespace {

#ifdef _WIN32
HWND s_hWndTV = NULL;
HWND s_hWndDRC = NULL;

GX2TVRenderMode s_ModeTV = GX2_TV_RENDER_NONE;
GX2DRCMode s_ModeDRC = GX2_DRC_NONE;
GX2SurfaceFormat s_FormatTV = GX2_SURFACE_FORMAT_INVALID;
GX2SurfaceFormat s_FormatDRC = GX2_SURFACE_FORMAT_INVALID;
#else
MEMHeapHandle s_Mem1Heap = MEM_HEAP_INVALID_HANDLE;

void* s_ScanBufferTV = NULL;
void* s_ScanBufferDRC = NULL;

bool s_QuitMsg = false;
#endif

#if defined(NW_USE_CPU_PROFILER) && !defined(_WIN32)

u8* s_ProfilerBuffer = NULL;

#endif

const int s_TblModeToSizeTV[] = {
    0, 0,
    640, 480,
    854, 480,
    1280, 720,
    1280, 720,
    1920, 1080
};

const int s_TblModeToSizeDRC[] = {
    0, 0,
    854, 480,
    854, 960
};

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

#define POSIX_MOUNT_POINT "/cygdrive/"

void SetCurrentDir()
{
#ifdef _WIN32
    if (u32 size = GetEnvironmentVariableA("CAFE_CONTENT_DIR", NULL, 0))
    {
        std::string contentDir;
        contentDir.resize(size); // NUL 終端込みのサイズ。
        GetEnvironmentVariableA("CAFE_CONTENT_DIR", &contentDir[0], size);
        contentDir.resize(size - 1); // NULL 終端を削除。

        // CafeSDK では CAFE_CONTENT_DIR はマウントポイント /cygdrive から始まる
        // POSIX 形式である必要があるので、マウントポイント /cygdrive の場合のみ変換するようにする。
        if (contentDir.find(POSIX_MOUNT_POINT) == 0)
        {
            contentDir = contentDir.substr(strlen(POSIX_MOUNT_POINT), std::string::npos);
            // c/ を c:\ に変換。
            contentDir.insert(contentDir.begin() + 1, 1, ':');
            std::replace(contentDir.begin(), contentDir.end(), '/', '\\');
        }

        SetCurrentDirectoryA(contentDir.c_str());
    }
#endif
}

void InitLauncher()
{
#ifdef _WIN32
    _CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
    _CrtSetBreakAlloc(-1);

    setlocale(LC_CTYPE, "");

    nw::g3d::demo::SetCurrentDir();

    // FPU の精度を 32 ビットに設定し、非正規化数を 0 として扱う。
    unsigned int oldVal;
    int err = _controlfp_s(&oldVal, _DN_FLUSH | _PC_24, _MCW_DN | MCW_PC);
    (void)err;
    NW_G3D_ASSERT(err == 0);
#endif

    NW_G3D_PERF_INIT();
}

void Init(const InitArg& arg)
{
    (void)arg;
#ifdef _WIN32
    if (arg.hRC)
    {
        nw::g3d::GfxContext::Reinit(arg.hRC);
    }
#else
    static bool gx2Initialized = false;
    if (!gx2Initialized)
    {
        GX2Init(NULL);
    }
    if (s_Mem1Heap == MEM_HEAP_INVALID_HANDLE)
    {
        MEMHeapHandle hMem1 = MEMGetBaseHeapHandle(MEM_ARENA_1);
        u32 allocatableSize = MEMGetAllocatableSizeForFrmHeapEx(hMem1, DEFAULT_ALIGNMENT);
        void* pMem1 = MEMAllocFromFrmHeapEx(hMem1, allocatableSize, DEFAULT_ALIGNMENT);
        s_Mem1Heap = MEMCreateExpHeap(pMem1, allocatableSize);
        NW_G3D_ASSERT(s_Mem1Heap != MEM_HEAP_INVALID_HANDLE);
    }

#if defined( __ghs__ )
    OSInitFastCast();
#endif

#if defined(NW_USE_CPU_PROFILER)

    s_ProfilerBuffer = AllocMem2<u8>(PROFILER_MINBUFFERSIZE);
    PROFILERInitialize(s_ProfilerBuffer, PROFILER_MINBUFFERSIZE);

#endif // NW_USE_CPU_PROFILER

#if defined(NW_USE_GPU_PROFILER)

    GX2DebugCaptureInit(NULL);

#endif // NW_USE_GPU_PROFILER

#endif

#if NW_G3D_CONFIG_USE_HOSTIO
#if NW_G3D_IS_HOST_WIN
    WSADATA wsaData;
    WSAStartup(MAKEWORD(1,1), &wsaData);
#else
    HIOInit();
#endif
#endif
}

void Shutdown()
{
#ifdef _WIN32
#else

#if defined(NW_USE_CPU_PROFILER)

    PROFILERFinalize();
    FreeMem2(s_ProfilerBuffer);
#endif // NW_USE_CPU_PROFILER

    GX2Shutdown();
    if (s_Mem1Heap != MEM_HEAP_INVALID_HANDLE)
    {
        MEMDestroyExpHeap(s_Mem1Heap);
        s_Mem1Heap = MEM_HEAP_INVALID_HANDLE;
        MEMHeapHandle hMem1 = MEMGetBaseHeapHandle(MEM_ARENA_1);
        MEMFreeToFrmHeap(hMem1, MEM_FRMHEAP_FREE_ALL);
    }
#endif

#if NW_G3D_CONFIG_USE_HOSTIO
#if NW_G3D_IS_HOST_WIN
    WSACleanup();
#else
    //HIOShutdown();
#endif
#endif
}

bool ProcessMsg()
{
    NW_G3D_PERF_NEXT_FRAME();
#ifdef _WIN32
    MSG msg;
    while (PeekMessageA(&msg, NULL, 0, 0, PM_REMOVE))
    {
        if (msg.message == WM_QUIT)
        {
            return false;
        }

        TranslateMessage(&msg);
        DispatchMessageA(&msg);
    }
    return true;
#else
    if (s_QuitMsg)
    {
        s_QuitMsg = false;
        return false;
    }
#if defined(NW_USE_CPU_PROFILER)
    PROFILERHeartbeat(PROFILEREventHeartbeatName_Main, PROFILEREventHeartbeatType_Beat);
#endif
    return true;
#endif
}

void PostQuitMsg()
{
#ifdef _WIN32
    PostQuitMessage(EXIT_SUCCESS);
#else
    s_QuitMsg = true;
#endif
}

//--------------------------------------------------------------------------------------------------
// Memory

void* AllocMem1(size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/)
{
    NW_G3D_ASSERT(IsPowerOfTwo(alignment));
    void* ptr = NULL;
#ifdef _WIN32
    ptr = _aligned_malloc(size, alignment);
#else
    NW_G3D_ASSERT(s_Mem1Heap != MEM_HEAP_INVALID_HANDLE);
    ptr = MEMAllocFromExpHeapEx(s_Mem1Heap, size, alignment);
    GX2NotifyMemAlloc(ptr, size, alignment);
#endif
    NW_G3D_ASSERTMSG(ptr != NULL, "AllocMem1 failed.\n");
    return ptr;
}

void FreeMem1(void* ptr)
{
#ifdef _WIN32
    _aligned_free(ptr);
#else
    MEMFreeToExpHeap(s_Mem1Heap, ptr);
    GX2NotifyMemFree(ptr);
#endif
}

void* AllocMem2(size_t size, size_t alignment /*= DEFAULT_ALIGNMENT*/)
{
    NW_G3D_ASSERT(IsPowerOfTwo(alignment));
    void* ptr = NULL;
#ifdef _WIN32
    ptr = _aligned_malloc(size, alignment);
#else
    ptr = MEMAllocFromDefaultHeapEx(size, alignment);
#endif
    NW_G3D_ASSERTMSG(ptr != NULL, "AllocMem2 failed.\n");
    return ptr;
}

void FreeMem2(void* ptr)
{
#ifdef _WIN32
    _aligned_free(ptr);
#else
    MEMFreeToDefaultHeap(ptr);
#endif
}

//--------------------------------------------------------------------------------------------------
// Pad

bool Pad::Reset()
{
#ifndef _WIN32
    PADInit();
#endif
    if (!Read() || !Read()) // m_Last* を初期化するため2回空読みする。
    {
        nw::g3d::DebugPrint("Pad::Read Failed.\n");
        return false;
    }
    m_AnalogStick.leftX = m_AnalogStick.leftY = m_AnalogStick.rightX = m_AnalogStick.rightY = 0.0f;

    return true;
}

bool Pad::Read()
{
    bit32 button = 0;
    Stick stick = { 0, 0, 0, 0 };
#ifdef _WIN32
    HWND hWnd = GetFocus();
    if (hWnd && (hWnd == s_hWndTV || hWnd == s_hWndDRC))
    {
        static const bit16 HOLD_MASK = 0x8000;
        button |= (HOLD_MASK & GetKeyState(VK_BACK)) ? BUTTON_B : 0;
        button |= (HOLD_MASK & GetKeyState(VK_RETURN)) ? BUTTON_A : 0;
        button |= (HOLD_MASK & GetKeyState(VK_SPACE)) ? BUTTON_START : 0;
        button |= (HOLD_MASK & GetKeyState(VK_LEFT)) ? BUTTON_LEFT : 0;
        button |= (HOLD_MASK & GetKeyState(VK_UP)) ? BUTTON_UP : 0;
        button |= (HOLD_MASK & GetKeyState(VK_RIGHT)) ? BUTTON_RIGHT : 0;
        button |= (HOLD_MASK & GetKeyState(VK_DOWN)) ? BUTTON_DOWN : 0;
        button |= (HOLD_MASK & GetKeyState('A')) ? BUTTON_A : 0;
        button |= (HOLD_MASK & GetKeyState('B')) ? BUTTON_B : 0;
        button |= (HOLD_MASK & GetKeyState('X')) ? BUTTON_X : 0;
        button |= (HOLD_MASK & GetKeyState('Y')) ? BUTTON_Y : 0;
        button |= (HOLD_MASK & GetKeyState('R')) ? TRIGGER_R : 0;
        button |= (HOLD_MASK & GetKeyState('L')) ? TRIGGER_L : 0;
        button |= (HOLD_MASK & GetKeyState('Z')) ? TRIGGER_Z : 0;

        POINT point;
        RECT rect;
        if (GetCursorPos(&point) && ScreenToClient(hWnd, &point)
            && GetClientRect(hWnd, &rect))
        {
            if (0 <= point.x && point.x <= rect.right && 0 <= point.y && point.y <= rect.bottom)
            {
                static const s8 RADIUS = 127;
                float invRadius = 1.0f / std::min(rect.right, rect.bottom);
                float rx = invRadius * ((point.x << 1) - rect.right);
                float ry = invRadius * (rect.bottom - (point.y << 1));
                float lengthSq = rx * rx + ry * ry;
                if (lengthSq > 1.0f)
                {
                    float rcpLen = nw::g3d::math::Math::RSqrt(lengthSq);
                    rx *= rcpLen;
                    ry *= rcpLen;
                }
                if (HOLD_MASK & GetKeyState(VK_LBUTTON))
                {
                    stick.leftX = static_cast<s8>(rx * RADIUS);
                    stick.leftY = static_cast<s8>(ry * RADIUS);
                }
                if (HOLD_MASK & GetKeyState(VK_RBUTTON))
                {
                    stick.rightX = static_cast<s8>(rx * RADIUS);
                    stick.rightY = static_cast<s8>(ry * RADIUS);
                }
            }
        }
    }
#else
    PADStatus pads[PAD_MAX_CONTROLLERS];
    do
    {
        PADRead(pads);
        PADClampCircle(pads);
        switch (pads[PAD_CHAN0].err)
        {
        case PAD_ERR_NONE:
            break;
        case PAD_ERR_NOT_READY:
        case PAD_ERR_TRANSFER:
            continue;
        case PAD_ERR_NO_CONTROLLER:
            nw::g3d::DebugPrint("No controller.\n");
        default:
            return false;
        }
        break;
    } while (0);

    const PADStatus& pad = pads[PAD_CHAN0];
    button = pad.button;
    stick.leftX = pad.stickX;
    stick.leftY = pad.stickY;
    stick.rightX = pad.substickX;
    stick.rightY = pad.substickY;
#endif
    static const int STICK_THRESHOLD = 32;
    button |= (m_Stick.leftX < -STICK_THRESHOLD) ? STICK_L_LEFT : 0;
    button |= (m_Stick.leftX > STICK_THRESHOLD) ? STICK_L_RIGHT : 0;
    button |= (m_Stick.leftY < -STICK_THRESHOLD) ? STICK_L_DOWN : 0;
    button |= (m_Stick.leftY > STICK_THRESHOLD) ? STICK_L_UP : 0;
    button |= (m_Stick.rightX < -STICK_THRESHOLD) ? STICK_R_LEFT : 0;
    button |= (m_Stick.rightX > STICK_THRESHOLD) ? STICK_R_RIGHT : 0;
    button |= (m_Stick.rightY < -STICK_THRESHOLD) ? STICK_R_DOWN : 0;
    button |= (m_Stick.rightY > STICK_THRESHOLD) ? STICK_R_UP : 0;

    m_LastButton = m_Button;
    m_Button = button;
    m_LastStick = m_Stick;
    m_Stick = stick;

    m_AnalogStick.leftX +=
        static_cast<float>(m_Stick.leftX - m_LastStick.leftX) / 56.0f;
    m_AnalogStick.leftY +=
        static_cast<float>(m_Stick.leftY - m_LastStick.leftY) / 56.0f;
    m_AnalogStick.rightX +=
        static_cast<float>(m_Stick.rightX - m_LastStick.rightX) / 56.0f;
    m_AnalogStick.rightY +=
        static_cast<float>(m_Stick.rightY - m_LastStick.rightY) / 56.0f;

    bit32 changed = m_Button ^ m_LastButton;
    m_Triggered = changed & m_Button;
    m_Released = changed & m_LastButton;

    return true;
}

//--------------------------------------------------------------------------------------------------
// CameraController

void ControllCamera(nw::g3d::Vec3* pos, nw::g3d::Vec3* up, nw::g3d::Vec3* target, const nw::g3d::Mtx34* viewMtx, const Pad* pad)
{
    const f32 RotSpeed     = 0.05f;
    const f32 ZoomSpeed    = 30.0f;
    const f32 ZoomMin      = 0.001f;
    const f32 ZoomMax      = 1000000.0f;
    const f32 MoveScaleMin = 0.0001f;
    const f32 MoveScaleMax = 1000.0f;
    const f32 AtMoveSpeed  = 25.0f;

    float zoomInOut = 0.0f;
    float upDown = 0.0f;

    if (pad->IsHold(Pad::BUTTON_Y))
    {
        zoomInOut = 1.0f;
    }
    else if (pad->IsHold(Pad::BUTTON_X))
    {
        zoomInOut = -1.0f;
    }

    if (pad->IsHold(Pad::BUTTON_A))
    {
        upDown = 1.0f;
    }
    else if (pad->IsHold(Pad::BUTTON_B))
    {
        upDown = -1.0f;
    }

    nw::g3d::Vec3 right, camUp, look;
    right.Set(viewMtx->m00, viewMtx->m01, viewMtx->m02);
    camUp.Set(viewMtx->m10, viewMtx->m11, viewMtx->m12);
    look.Set(viewMtx->m20, viewMtx->m21, viewMtx->m22);

    nw::g3d::Vec3 diff;
    diff.Sub(*target, *pos);

    float dist = diff.Normalize(diff);
    f32 ratio     = ( dist - ZoomMin ) / ( ZoomMax - ZoomMin );
    f32 moveScale = 0.5f * ( MoveScaleMin + ( MoveScaleMax - MoveScaleMin ) * ratio );
    dist += zoomInOut * ZoomSpeed * moveScale;
    dist = ( ZoomMax <= dist ) ? ZoomMax : ( ( dist <= ZoomMin ) ? ZoomMin : dist );

    nw::g3d::Vec3 dirX, dirY, dirZ;

    dirZ.Cross(right, nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f));
    dirZ.Normalize(dirZ);

    dirY.Cross(dirZ, right);
    dirY.Normalize(dirY);

    const Pad::AnalogStick& analogStick = pad->GetAnalogStick();

    target->Add(dirX.Mul(right, AtMoveSpeed * moveScale * analogStick.rightX), *target);
    target->Add(dirY.Mul(dirY, AtMoveSpeed * moveScale * upDown), *target);
    target->Add(dirZ.Mul(dirZ, AtMoveSpeed * moveScale * ( -analogStick.rightY)), *target);

    nw::g3d::math::Quat qt, qtX, qtY;
    {
        float ang = analogStick.leftY * -RotSpeed * 0.5f;
        float c = nw::g3d::math::Math::Cos(ang);
        float s = nw::g3d::math::Math::Sin(ang);
        qtX.Set(s * right.x, s * right.y, s *right.z, c);
    }
    {
        float ang = analogStick.leftX * RotSpeed * 0.5f;
        float c = nw::g3d::math::Math::Cos(ang);
        float s = nw::g3d::math::Math::Sin(ang);
        qtY.Set(0.0f, s, 0.0f, c);
    }

    qt.Mul(qtY, qtX);

    nw::g3d::Mtx34 qtMtx;
    qtMtx.SetR(qt);

    diff.Rotate(qtMtx, diff);
    diff.Normalize(diff);

    pos->Mul(diff, dist);
    pos->Sub(*target, *pos);;

    up->Cross(right, diff);
    up->Set(0.0f, (up->y >= 0) ? 1.0f : 0.0f, 0.0f);

#if 0
    nw::g3d::DebugPrint("------values------\n");
    nw::g3d::DebugPrint("left  : %f, %f\n", analogStick.leftX, analogStick.leftY);
    nw::g3d::DebugPrint("right : %f, %f\n", analogStick.rightX, analogStick.rightY);

    nw::g3d::DebugPrint("pos   : %f, %f, %f\n", pos->x, pos->y, pos->z);
    nw::g3d::DebugPrint("up    : %f, %f, %f\n", up->x, up->y, up->z);
    nw::g3d::DebugPrint("target: %f, %f, %f\n", target->x, target->y, target->z);
#endif
}

void LookAtModel(nw::g3d::Vec3* pos, nw::g3d::Vec3* target, const nw::g3d::ModelObj* model)
{
    if (const nw::g3d::Sphere* pBounding = model->GetBounding())
    {
        float distance = pBounding->radius / ( nw::g3d::Math::Tan(nw::g3d::Math::DegToRad(45.0f) * 0.5f) );

        nw::g3d::Vec3 dir;
        dir.Sub(*pos, *target);
        dir.Normalize(dir);
        dir.Mul(dir, distance);

        *target = pBounding->center;
        pos->Add(pBounding->center, dir);
    }
}

//--------------------------------------------------------------------------------------------------
// File

int OpenFile(FileHandle* pHandle, const char* path, const char* mode)
{
    NW_G3D_ASSERT_NOT_NULL(path);
    NW_G3D_ASSERT_NOT_NULL(mode);
#ifdef _WIN32
    return fopen_s(pHandle, path, mode);
#else
    return DEMOFSOpenFileMode(path, pHandle, mode);
#endif
}

int CloseFile(FileHandle handle)
{
#ifdef _WIN32
    return fclose(handle);
#else
    return DEMOFSCloseFile(&handle);
#endif
}

int GetFileSize(FileHandle handle, size_t* pSize)
{
    NW_G3D_ASSERT_NOT_NULL(pSize);
#ifdef _WIN32
    *pSize = _filelength(_fileno(handle));
    return 0;
#else
    u32 size;
    s32 ret = DEMOFSGetLength(&handle, &size);
    *pSize = static_cast<size_t>(size);
    return ret;
#endif
}

int ReadFile(FileHandle handle, void* pDst, size_t length)
{
    NW_G3D_ASSERT_NOT_NULL(pDst);
#ifdef _WIN32
    int result = fseek(handle, 0, SEEK_SET);
    if (result != 0)
    {
        return -1;
    }
    size_t readSize = fread(pDst, length, 1, handle);
    return readSize > 0 ? 0 : -1;
#else
    return DEMOFSRead(&handle, pDst, length, 0);
#endif
}

int WriteFile(FileHandle handle, const void* pSrc, size_t length)
{
    NW_G3D_ASSERT_NOT_NULL(pSrc);
#ifdef _WIN32
    int result = fseek(handle, 0, SEEK_SET);
    if (result != 0)
    {
        return -1;
    }
    size_t writtenSize = fwrite(pSrc, length, 1, handle);
    return writtenSize > 0 ? 0 : -1;
#else
    return DEMOFSWrite(&handle, pSrc, length);
#endif
}

void* LoadFile(const char* path, size_t* pSize /*= NULL*/, size_t alignment /*= DEFAULT_ALIGNMENT*/)
{
    NW_G3D_ASSERT_NOT_NULL(path);
    NW_G3D_ASSERT(IsPowerOfTwo(alignment));

#ifndef _WIN32
    alignment = std::max<size_t>(alignment, PPC_IO_BUFFER_ALIGN);
#endif

    int result = 0;
    FileHandle handle = 0;
    result = OpenFile(&handle, path, "rb");
    NW_G3D_ASSERTMSG(result == 0, "OpenFile failed.\n");

    size_t size = 0;
    result = GetFileSize(handle, &size);
    NW_G3D_ASSERTMSG(result == 0, "GetFileSize failed.\n");
    NW_G3D_ASSERTMSG(size != 0, "file size is 0.\n");

    // テキストファイル NUL 終端させるために多めにメモリ確保して 0 で埋める。
    void* ptr = AllocMem2(size + DEFAULT_ALIGNMENT, alignment);
    result = ReadFile(handle, ptr, size);
    NW_G3D_ASSERTMSG(result == 0, "ReadFile failed.\n");

    memset(nw::g3d::AddOffset(ptr, size), 0, DEFAULT_ALIGNMENT);
    result = CloseFile(handle);
    NW_G3D_ASSERTMSG(result == 0, "CloseFile failed.\n");

    size += DEFAULT_ALIGNMENT;

    if (pSize)
    {
        *pSize = size;
    }

    NW_G3D_UNUSED(result);

    return ptr;
}

void SaveFile(const char* path, const void* pData, size_t size)
{
    NW_G3D_ASSERT_NOT_NULL(path);
    NW_G3D_ASSERT_NOT_NULL(pData);

#ifndef _WIN32
    NW_G3D_ASSERT_ADDR_ALIGNMENT(pData, PPC_IO_BUFFER_ALIGN);
#endif

    int result = 0;
    FileHandle handle = 0;
    result = OpenFile(&handle, path, "wb");
    NW_G3D_ASSERTMSG(result == 0, "OpenFile failed.\n");

    result = WriteFile(handle, pData, size);
    NW_G3D_ASSERTMSG(result == 0, "WriteFile failed.\n");

    result = CloseFile(handle);
    NW_G3D_ASSERTMSG(result == 0, "CloseFile failed.\n");

    NW_G3D_UNUSED(result);
}

//--------------------------------------------------------------------------------------------------
// Display

#ifdef _WIN32

namespace {

LRESULT CALLBACK WndProc(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    case WM_PAINT:
        {
            PAINTSTRUCT ps;
            BeginPaint(hWnd, &ps);
            EndPaint(hWnd, &ps);
            return 0;
        }
    case WM_CLOSE:
        PostQuitMessage(EXIT_SUCCESS);
        return 0;
    case WM_ERASEBKGND:
        return FALSE; // BeginPaint で PAINTSTRUCT の fErase を TRUE にする。
    case WM_KEYDOWN:
        if (wParam == VK_ESCAPE)
        {
            PostQuitMessage(EXIT_SUCCESS);
            return 0;
        }
        break;
    default:
        break;
    }

    return DefWindowProcA(hWnd, uMsg, wParam, lParam);
}

} // anonymous namespace

#endif // _WIN32

void InitDisplay(const InitDisplayArg& arg)
{
#ifdef _WIN32
    NW_G3D_ASSERT(s_hWndTV == NULL);
    NW_G3D_ASSERT(s_hWndDRC == NULL);

    nw::g3d::fnd::detail::SystemContext& system = nw::g3d::fnd::detail::GetSystemContext();

    if (arg.modeTV != GX2_TV_RENDER_NONE)
    {
        nw::g3d::fnd::detail::CreateWndArg createWndArg;
        createWndArg.wndName = "g3d demo TV";
        createWndArg.wndProc = WndProc;
        createWndArg.width = s_TblModeToSizeTV[arg.modeTV * 2 + 0];
        createWndArg.height = s_TblModeToSizeTV[arg.modeTV * 2 + 1];
        s_ModeTV = arg.modeTV;
        s_FormatTV = arg.formatTV;

        s_hWndTV = CreateWnd(createWndArg);
        HDC hDC = GetDC(s_hWndTV);
        wglMakeCurrent(hDC, static_cast<HGLRC>(system.ctx.hRC));
        nw::g3d::SetSwapInterval(arg.swapInterval);
        ReleaseDC(s_hWndTV, hDC);
    }

    if (arg.modeDRC != GX2_DRC_NONE)
    {
        nw::g3d::fnd::detail::CreateWndArg createWndArg;
        createWndArg.wndName = "g3d demo DRC";
        createWndArg.wndProc = WndProc;
        createWndArg.width = s_TblModeToSizeDRC[arg.modeDRC * 2 + 0];
        createWndArg.height = s_TblModeToSizeDRC[arg.modeDRC * 2 + 1];
        s_ModeDRC = arg.modeDRC;
        s_FormatDRC = arg.formatDRC;

        s_hWndDRC = CreateWnd(createWndArg);
        HDC hDC = GetDC(s_hWndDRC);
        wglMakeCurrent(hDC, static_cast<HGLRC>(system.ctx.hRC));
        nw::g3d::SetSwapInterval(arg.swapInterval);
        ReleaseDC(s_hWndDRC, hDC);
    }

    nw::g3d::GfxContext::Invalidate();
#else
    NW_G3D_ASSERT(s_ScanBufferTV == NULL);
    NW_G3D_ASSERT(s_ScanBufferDRC == NULL);

    if (arg.modeTV != GX2_TV_RENDER_NONE)
    {
        u32 size = 0;
        GX2Boolean scaleNeeded = GX2_FALSE;
        GX2CalcTVSize(arg.modeTV, arg.formatTV, GX2_BUFFERING_DOUBLE, &size, &scaleNeeded);
        nw::g3d::DebugPrint("(ScanBufferTV) size: %d, alignment: %d\n",
            size, GX2_SCAN_BUFFER_ALIGNMENT);
        s_ScanBufferTV = AllocMem2(size, GX2_SCAN_BUFFER_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(s_ScanBufferTV);
        DCInvalidateRange(s_ScanBufferTV, size);

        GX2SetTVBuffer(s_ScanBufferTV, size, arg.modeTV, arg.formatTV, GX2_BUFFERING_DOUBLE);
        GX2SetTVScale(
            s_TblModeToSizeTV[arg.modeTV * 2 + 0], s_TblModeToSizeTV[arg.modeTV * 2 + 1]);
        GX2SetTVEnable(GX2_TRUE);
    }

    if (arg.modeDRC != GX2_DRC_NONE)
    {
        u32 size = 0;
        GX2Boolean scaleNeeded = GX2_FALSE;
        GX2CalcDRCSize(arg.modeDRC, arg.formatDRC, GX2_BUFFERING_DOUBLE, &size, &scaleNeeded);
        nw::g3d::DebugPrint("(ScanBufferDRC) size: %d, alignment: %d\n",
            size, GX2_SCAN_BUFFER_ALIGNMENT);
        s_ScanBufferDRC = AllocMem2(size, GX2_SCAN_BUFFER_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(s_ScanBufferDRC);
        DCInvalidateRange(s_ScanBufferDRC, size);

        GX2SetDRCBuffer(s_ScanBufferDRC, size, arg.modeDRC, arg.formatDRC, GX2_BUFFERING_DOUBLE);
        GX2SetDRCScale(
            s_TblModeToSizeDRC[arg.modeDRC * 2 + 0], s_TblModeToSizeDRC[arg.modeDRC * 2 + 1]);
        GX2SetDRCEnable(GX2_TRUE);
    }

    GX2SetSwapInterval(arg.swapInterval);

#endif
}

void ShutdownDisplay()
{
#ifdef _WIN32
    if (s_hWndTV)
    {
        nw::g3d::fnd::detail::DestroyWnd(s_hWndTV);
        s_hWndTV = NULL;
    }
    if (s_hWndDRC)
    {
        nw::g3d::fnd::detail::DestroyWnd(s_hWndDRC);
        s_hWndDRC = NULL;
    }
#else
    if (s_ScanBufferTV)
    {
        FreeMem2(s_ScanBufferTV);
        s_ScanBufferTV = NULL;
        GX2SetTVEnable(GX2_FALSE);
    }
    if (s_ScanBufferDRC)
    {
        FreeMem2(s_ScanBufferDRC);
        s_ScanBufferDRC = NULL;
        GX2SetDRCEnable(GX2_FALSE);
    }
#endif
}

void CopyOut(const nw::g3d::GfxColorBuffer* pColorBuffer, GX2ScanTarget target)
{
    NW_G3D_ASSERT_NOT_NULL(pColorBuffer);
#ifdef _WIN32
    nw::g3d::fnd::detail::SystemContext& system = nw::g3d::fnd::detail::GetSystemContext();
    if (ActivateDisplay(target, system.ctx))
    {
        const nw::g3d::fnd::detail::SystemShader* shader = &system.swapCopyShader;
        if (pColorBuffer->GetGX2ColorBuffer()->surface.aa != GX2_AA_MODE_1X)
        {
            shader = &system.swapResolveShader;
        }
        glUseProgram(shader->hShader);

        glBindFramebuffer(GL_FRAMEBUFFER, 0);
        if (target < GX2_SCAN_TARGET_DRC_FIRST)
        {
            glViewport(0, 0, s_TblModeToSizeTV[s_ModeTV * 2], s_TblModeToSizeTV[s_ModeTV * 2 + 1]);
            glScissor(0, 0, s_TblModeToSizeTV[s_ModeTV * 2], s_TblModeToSizeTV[s_ModeTV * 2 + 1]);
            if (s_FormatTV == GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB)
            {
                glEnable(GL_FRAMEBUFFER_SRGB);
            }
        }
        else
        {
            glViewport(0, 0, s_TblModeToSizeDRC[s_ModeDRC * 2], s_TblModeToSizeDRC[s_ModeDRC * 2 + 1]);
            glScissor(0, 0, s_TblModeToSizeDRC[s_ModeDRC * 2], s_TblModeToSizeDRC[s_ModeDRC * 2 + 1]);
            if (s_FormatDRC == GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB)
            {
                glEnable(GL_FRAMEBUFFER_SRGB);
            }
        }

        system.BindTexture(pColorBuffer->handle, shader->texUnit, 0);
        glUniform1f(shader->locations[shader->LOCATION_GAMMA], 1.0f);
        system.DrawScreenQuad();

        NW_G3D_GL_ASSERT();
    }
    wglMakeCurrent(NULL, NULL); // GX2 の挙動に合わせて無効化する。
#else
    GX2CopyColorBufferToScanBuffer(pColorBuffer->GetGX2ColorBuffer(), target);
#endif
}

void SwapScanBuffers()
{
#ifdef _WIN32
    GfxContext::Prepare();
    if (s_hWndTV)
    {
        HDC hDC = GetDC(s_hWndTV);
        SwapBuffers(hDC);
        ReleaseDC(s_hWndTV, hDC);
    }
    if (s_hWndDRC)
    {
        HDC hDC = GetDC(s_hWndDRC);
        SwapBuffers(hDC);
        ReleaseDC(s_hWndDRC, hDC);
    }
    GfxContext::Invalidate(); // GX2 の挙動に合わせて無効化する。
#else
    GX2SwapScanBuffers();
    GfxManage::FlushCommands();

    while (true)
    {
        u32 swapCount, flipCount;
        OSTime lastFlip, lastVsyncTime;
        GX2GetSwapStatus (&swapCount, &flipCount, &lastFlip, &lastVsyncTime);
        if (flipCount >= swapCount)
        {
            break;
        }
        GX2WaitForVsync();
    }
#endif
}

bool ActivateDisplay(GX2ScanTarget target, nw::g3d::GfxContext& ctx)
{
#ifdef _WIN32
    if (target == GX2_SCAN_TARGET_TV)
    {
        HDC hDC = GetDC(s_hWndTV);
        wglMakeCurrent(hDC, static_cast<HGLRC>(ctx.hRC));
        ReleaseDC(s_hWndTV, hDC);
        return true;
    }
    else if (target == GX2_SCAN_TARGET_DRC_FIRST)
    {
        HDC hDC = GetDC(s_hWndDRC);
        wglMakeCurrent(hDC, static_cast<HGLRC>(ctx.hRC));
        ReleaseDC(s_hWndDRC, hDC);
        return true;
    }
    return false;
#else
    (void)ctx;
    (void)target;
    return false;
#endif
}

//--------------------------------------------------------------------------------------------------
// Menu

void CmdMenu::PrintMenu() const
{
    nw::g3d::DebugPrint("[menu]\n");
    nw::g3d::DebugPrint("   0: %s\n", m_pItems[m_NumItem].title);
    for (int idxItem = 0; idxItem < m_NumItem; ++idxItem)
    {
        nw::g3d::DebugPrint("  %2d: %s\n", idxItem + 1, m_pItems[idxItem].title);
    }
}

int CmdMenu::Run(int item, int argc, const char* argv[]) const
{
    nw::g3d::DebugPrint("\n");
    if (item <= 0 || m_NumItem < item)
    {
        return 0;
    }
    int retCode = EXIT_SUCCESS;
    do
    {
        NW_G3D_PERF_RESET();
        nw::g3d::DebugPrint("[%s]\n", m_pItems[item - 1].title);
#ifdef _WIN32
        _CrtMemState memState;
        (void)memState;
        _CrtMemCheckpoint(&memState);
#endif
        retCode = m_pItems[item - 1].pFunc(argc, argv);
#ifdef _WIN32
        _CrtMemDumpAllObjectsSince(&memState);
#endif
        NW_G3D_PERF_PRINT();
        NW_G3D_PERF_PRINT_TABLE();
        if (retCode > EXIT_SUCCESS)
        {
            nw::g3d::DebugPrint("EXIT_FAILURE\n");
        }
        nw::g3d::DebugPrint("\n");
    } while (retCode == -1);
    return retCode;
}

int CmdMenu::Select() const
{
    int item = 0;
    nw::g3d::DebugPrint("\n");
#ifdef _WIN32
    nw::g3d::DebugPrint("> ");
    int c = getchar();
    while (('\0' < c && c < '0') || '9' < c)
    {
        c = getchar();
    }
    while ('0' <= c && c <= '9')
    {
        item *= 10;
        item += c - '0';
        c = getchar();
    }
#else
    Pad pad;
    if (!pad.Reset())
    {
        return 0;
    }

    static const int lenOffset = strlen("> 00: ");
    nw::g3d::DebugPrint(">  0: %s", m_pItems[m_NumItem].title);
    int lastLen = m_pItems[m_NumItem].len + lenOffset;
    if (OSIsDebuggerPresent())
    {
        nw::g3d::DebugPrint("\n");
    }

    while (1)
    {
        if (!pad.Read())
        {
            nw::g3d::DebugPrint("Pad::Read Failed.\n");
            return 0;
        }

        bool updated = false;
        if (pad.IsTriggered(Pad::BUTTON_STICK_L_UP))
        {
            ++item;
            updated = true;
        }
        else if (pad.IsTriggered(Pad::BUTTON_STICK_L_DOWN))
        {
            --item;
            updated = true;
        }
        else if (pad.IsTriggered(Pad::BUTTON_STICK_L_RIGHT))
        {
            item += 10;
            updated = true;
        }
        else if (pad.IsTriggered(Pad::BUTTON_STICK_L_LEFT))
        {
            item -= 10;
            updated = true;
        }

        item = std::min(std::max(item, 0), m_NumItem);

        if (updated)
        {
            if (!OSIsDebuggerPresent())
            {
                for (int i = 0; i < lastLen; ++i)
                {
                    nw::g3d::DebugPrint("\b");
                }
            }
            if (item > 0)
            {
                nw::g3d::DebugPrint("> %2d: %s", item, m_pItems[item - 1].title);
                lastLen = m_pItems[item - 1].len + lenOffset;
            }
            else
            {
                nw::g3d::DebugPrint(">  0: %s", m_pItems[m_NumItem].title);
                lastLen = m_pItems[m_NumItem].len + lenOffset;
            }
            if (OSIsDebuggerPresent())
            {
                nw::g3d::DebugPrint("\n");
            }
        }

        if (pad.IsTriggered(Pad::BUTTON_A))
        {
            break;
        }
        OSSleepMilliseconds(16);
    }
#endif
    nw::g3d::DebugPrint("\n");
    return item;
}

int CmdMenu::Loop(int argc, const char* argv[]) const
{
    int retCode = 0;
    int item = 0;
    if (argc > 0)
    {
        item = atoi(argv[0]);
        --argc;
        ++argv;
    }
    if (item == 0)
    {
        // メニューから選択。
        while (NW_G3D_STATIC_CONDITION(1))
        {
            PrintMenu();
            item = Select();
            if (item == 0)
            {
                break;
            }
            retCode = Run(item, argc, argv);
        }
    }
    else
    {
        // 引数で指定された項目を実行。
        retCode = Run(item, argc, argv);
    }

    return retCode;
}

}}} // namespace nw::g3d::demo
