﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_DEMO_MATERIALPICK_H_
#define NW_G3D_DEMO_MATERIALPICK_H_

#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/g3d_ShaderObj.h>
#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_UserModel.h>

namespace nw { namespace g3d { namespace demo {

//! @brief マテリアルピックを行うクラスです。
class MaterialPicker
{
public:
    //! @brief ピック情報です。
    struct PickInfo
    {
        UserModel* userModelPtr;    //!< マテリアルを含む UserModel へのポインタです。
        u32 matId;                  //!< マテリアルの ID です。
    };

    //! @brief Setup に渡す引数です。
    struct SetupArg
    {
        //! @brief コンストラクタです。
        SetupArg(nw::g3d::ResShadingModel* pShadingModel)
            : pShadingModel(pShadingModel)
            , maxModel(1024)
            , width(1280)
            , height(720)
        {}

        nw::g3d::ResShadingModel* pShadingModel;    //!< マテリアルピック用シェーディングモデルです。
        u32 maxModel;                               //!< 登録できるモデル数の最大値です。
        u32 width;                                  //!< 画面の幅です。
        u32 height;                                 //!< 画面の高さです。
    };

    //! @brief コンストラクタです。
    MaterialPicker()
        : m_ScreenWidth(1280)
        , m_ScreenHeight(720)
        , m_pResShadingModel(NULL)
    {}

    //! @brief デストラクタです。
    ~MaterialPicker() {}

    //! @brief ピック用 ResShadingModel を取得します。
    nw::g3d::ResShadingModel* GetResShadingModel() { return m_pResShadingModel; }

    //! @brief マテリアルピッカーをセットアップします。
    void Setup(SetupArg& arg);

    //! @brief マテリアルピッカーをクリーンアップします。
    void Cleanup();

    //! @brief モデルを登録します。
    void Entry(UserModel* pUserModel);

    //! @brief シェイプブロックに書き込むマテリアル情報を更新します。
    void UpdateBlock();

    //! @brief モデルを削除します。
    void Erase(UserModel* pUserModel);

    //! @brief 登録されているモデルをすべて削除します。
    void Clear();

    //! @brief ピックします。
    //!
    //! @return ピックしたマテリアルの数を返します。
    int Pick(
        PickInfo* pickInfo,
        int numMax,
        const nw::g3d::math::Vec2& pos1,
        const nw::g3d::math::Vec2& pos2,
        nw::g3d::GfxContext* pCtx
    );
    //! @brief ピックします。
    //!
    //! @return ピックしたマテリアルの数を返します。
    int Pick(
        PickInfo* pickInfo,
        int numMax,
        const nw::g3d::math::Vec2& pos,
        nw::g3d::GfxContext* pCtx
    )
    {
        return Pick(pickInfo, numMax, pos, pos, pCtx);
    }

private:
    u32 RoundUp(u32 x, int align) { return (x + align - 1) & ~(align - 1); }

    u32 m_ScreenWidth;
    u32 m_ScreenHeight;
    nw::g3d::ResShadingModel* m_pResShadingModel;
    Vector<UserModel*> m_ModelList;
} ;

}}} // namespace nw::g3d::demo

#endif // NW_G3D_DEMO_MATERIALPICK_H_
