﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NAudio.CoreAudioApi;
using NAudio.Wave;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.Environment;
using System;
using System.Runtime.InteropServices;
using System.Threading;

namespace NintendoWare.Spy
{
    internal sealed class WasapiWavePlayer : ObservableObject
    {
        private IWaveProvider _sourceWaveProvider;

        private MMDevice _device;
        private AudioClient _audioClient;
        private AudioRenderClient _renderClient;
        private AudioClientShareMode _shareMode;
        private int _latency;

        private int _bytesPerFrame;
        private byte[] _readBuffer;
        private WaveFormat _outputWaveFormat;
        private bool _shouldResampling;

        private EventWaitHandle _updateBufferEvent;
        private Thread _playbackThread;
        private SynchronizationContext _syncContext;

        private volatile PlaybackState _state = PlaybackState.NotInitialized;

        //-----------------------------------------------------------------

        public enum PlaybackState
        {
            NotInitialized,
            Stopped,
            Playing,
            Paused,
        }

        public class ExceptionEventArgs : EventArgs
        {
            public ExceptionEventArgs(Exception e)
            {
                this.Exception = e;
            }

            public Exception Exception { get; private set; }
        }

        //-----------------------------------------------------------------

        static WasapiWavePlayer()
        {
            if (Environment.OSVersion.Version.Major < WindowsOSVersion.Windows7MajorVersion)
            {
                throw new NotSupportedException("this os is not supported.");
            }
        }

        //-----------------------------------------------------------------

        public event EventHandler<ExceptionEventArgs> ErrorOccurred;

        //-----------------------------------------------------------------

        /// <summary>
        /// 状態を取得します。
        /// </summary>
        public PlaybackState State
        {
            get { return _state; }

            private set
            {
                if (_state == value)
                {
                    return;
                }

                _state = value;
                this.NotifyPropertyChanged();
            }
        }

        public AudioClientShareMode ShareMode
        {
            get { return _shareMode; }
        }

        public bool IsAvailable
        {
            get
            {
                if (_device == null)
                {
                    return false;
                }

                return !_device.State.HasFlag(DeviceState.Unplugged);
            }
        }

        //-----------------------------------------------------------------

        private static MMDevice GetDefaultAudioEndpoint()
        {
            MMDeviceEnumerator enumerator = new MMDeviceEnumerator();
            return enumerator.GetDefaultAudioEndpoint(DataFlow.Render, Role.Console);
        }

        public void Initialize(IWaveProvider waveProvider, MMDevice device, AudioClientShareMode shareMode, int latency)
        {
            Ensure.Argument.NotNull(waveProvider);
            Ensure.Argument.NotNull(device);

            Ensure.Operation.True(_state == PlaybackState.NotInitialized, "WasapiWavePlayer is already initialized.");

            // NAudio は、AudioClient プロパティ内でインスタンス生成するので、
            // ここでオート変数に格納する必要がある。
            var newAudioClient = device.AudioClient;

            var validWaveFormat = this.ValidateWaveFormat(newAudioClient, shareMode, waveProvider.WaveFormat);

            // サンプリングレート変換が必要な場合は、ここでサポートの有無をチェックする
            if (validWaveFormat.SampleRate != waveProvider.WaveFormat.SampleRate)
            {
                using (new ResamplerDmoStream(waveProvider, validWaveFormat))
                {
                }

                _outputWaveFormat = validWaveFormat;
                _shouldResampling = true;
            }
            else
            {
                _outputWaveFormat = waveProvider.WaveFormat;
                _shouldResampling = false;
            }

            _sourceWaveProvider = waveProvider;

            long latencyRefTimes = Math.Max(
                newAudioClient.DefaultDevicePeriod,
                MilliSecondsToRefTimes(latency));

            newAudioClient.Initialize(
                shareMode,
                AudioClientStreamFlags.EventCallback,
                latencyRefTimes,
                latencyRefTimes,
                _outputWaveFormat,
                Guid.Empty);

            _latency = (int)RefTimesToMilliSeconds(latencyRefTimes);

            _updateBufferEvent = new EventWaitHandle(false, EventResetMode.AutoReset);
            newAudioClient.SetEventHandle(_updateBufferEvent);

            _device = device;
            _audioClient = newAudioClient;
            _renderClient = newAudioClient.AudioRenderClient;
            _shareMode = shareMode;

            this.State = PlaybackState.Stopped;
        }

        public void Uninitialize()
        {
            this.Stop();

            if (_audioClient != null)
            {
                _audioClient.Dispose();
                _audioClient = null;
            }

            _device = null;
            _renderClient = null;
            _outputWaveFormat = null;
            _updateBufferEvent = null;
            _latency = 0;

            this.State = PlaybackState.NotInitialized;
        }

        public void Play(SynchronizationContext syncContext = null)
        {
            if (this.State == PlaybackState.Playing)
            {
                return;
            }

            if (this.State == PlaybackState.Stopped)
            {
                _syncContext = syncContext;

                _playbackThread = new Thread(PlaybackThread);
                _playbackThread.Name = typeof(WasapiWavePlayer).FullName + "." + nameof(PlaybackThread);
                _playbackThread.Start();
            }

            this.State = PlaybackState.Playing;
        }

        public void Stop()
        {
            if (this.State <= PlaybackState.Stopped)
            {
                return;
            }

            this.State = PlaybackState.Stopped;
            _playbackThread = null;
        }

        public void Pause()
        {
            if (_state == PlaybackState.Playing)
            {
                _state = PlaybackState.Paused;
            }
        }

        protected override void OnDisposed(EventArgs e)
        {
            this.Uninitialize();
            base.OnDisposed(e);
        }

        private static long RefTimesToMilliSeconds(long value)
        {
            // nsec -> msec
            return value / 10000;
        }

        private static long MilliSecondsToRefTimes(long value)
        {
            // msec -> nsec
            return value * 10000;
        }

        private WaveFormat ValidateWaveFormat(
            AudioClient audioClient,
            AudioClientShareMode shareMode,
            WaveFormat desiredWaveFormat)
        {
            Ensure.Argument.NotNull(audioClient);
            Ensure.Argument.NotNull(desiredWaveFormat);

            WaveFormatExtensible validWaveFormat;

            // 要求フォーマットがサポートされている場合は、そのまま利用する
            if (audioClient.IsFormatSupported(shareMode, desiredWaveFormat, out validWaveFormat))
            {
                return desiredWaveFormat;
            }

            // 要求に近いフォーマットが見つかった場合は、それを利用する
            if (validWaveFormat != null)
            {
                return validWaveFormat;
            }

            // AudioClient の MIX フォーマットが有効なら、それを利用する
            if (audioClient.IsFormatSupported(shareMode, audioClient.MixFormat))
            {
                return audioClient.MixFormat;
            }

            // 指定サンプリングレート, 16bit, 2ch が有効なら、それを利用する
            var defaultWaveFormat = new WaveFormat(desiredWaveFormat.SampleRate, 16, 2);
            if (audioClient.IsFormatSupported(shareMode, defaultWaveFormat))
            {
                return defaultWaveFormat;
            }

            throw new NotSupportedException("desired wave format is not supported.");
        }

        private void PlaybackThread()
        {
            ResamplerDmoStream resamplerDmoStream = null;

            try
            {
                var waveProvider = _sourceWaveProvider;

                if (_shouldResampling)
                {
                    resamplerDmoStream = new ResamplerDmoStream(_sourceWaveProvider, _outputWaveFormat);
                    waveProvider = resamplerDmoStream;
                }

                this.InitializeBuffer();

                if (this.FillBuffer(waveProvider, _audioClient.BufferSize) == 0)
                {
                    return;
                }

                _updateBufferEvent.Reset();
                WaitHandle[] waitHandles = new WaitHandle[] { _updateBufferEvent };

                // バッファ充填の閾値 thresholdForFillBuffer
                //
                // ResamplerDmoStream によるサンプルレート変換によっては、
                // アップサンプル後の AudioClient バッファを完全に満たせない場合があり、
                // その際、再生が停止してしまう問題に対処するために、バッファ充填の閾値を設けている。
                //
                // 具体的には 32k -> 44.1k にアップサンプルする際に再現しており、
                // アップサンプル後の 1 サンプル / 44.1kHz は、ソースの 32kHz 1 サンプルに満たないために、
                // FillBuffer が 0 を返し、波形の終端とご認識されてしまっていた。
                var thresholdForFillBuffer = _audioClient.BufferSize / 4;

                _audioClient.Start();

                try
                {
                    while (true)
                    {
                        if (_state == PlaybackState.Stopped)
                        {
                            break;
                        }

                        if (_state == PlaybackState.Paused)
                        {
                            Thread.Sleep(_latency);
                            continue;
                        }

                        int signalIndex = WaitHandle.WaitAny(waitHandles, 3 * _latency, false);

                        if (signalIndex == WaitHandle.WaitTimeout)
                        {
                            throw new TimeoutException();
                        }

                        var avairableFramesCount = this.GetAvailableFramesCount();
                        if (avairableFramesCount > thresholdForFillBuffer)
                        {
                            if (FillBuffer(waveProvider, avairableFramesCount) == 0)
                            {
                                break;
                            }
                        }
                    }

                    // 最後に ReleaseBuffer() した波形が出力されきるまで待つ
                    // http://msdn.microsoft.com/en-us/library/windows/desktop/dd370844%28v=vs.85%29.aspx
                    Thread.Sleep(_latency);
                }
                finally
                {
                    _audioClient.Stop();
                    _audioClient.Reset();

                    this.FinalizeBuffer();
                }
            }
            catch (Exception e)
            {
                this.NotifyErrorOccurred(e);
            }
            finally
            {
                if (resamplerDmoStream != null)
                {
                    resamplerDmoStream.Dispose();
                }

                this.PostSetState(PlaybackState.Stopped);
                this.NotifyErrorOccurred(null);
            }
        }

        private int GetAvailableFramesCount()
        {
            // NOTE: バッファ長 - パディング長（ともにフレーム数）
            return _audioClient.BufferSize -
                ((_shareMode == AudioClientShareMode.Shared) ? _audioClient.CurrentPadding : 0);
        }

        private void PostSetState(PlaybackState newState)
        {
            if (_syncContext == null)
            {
                this.State = newState;
            }
            else
            {
                _syncContext.Post(state => this.State = (PlaybackState)state, newState);
            }
        }

        private void NotifyErrorOccurred(Exception e)
        {
            if (this.ErrorOccurred == null)
            {
                return;
            }

            if (_syncContext == null)
            {
                this.ErrorOccurred(this, new ExceptionEventArgs(e));
            }
            else
            {
                _syncContext.Post(state => this.ErrorOccurred?.Invoke(this, new ExceptionEventArgs(e)), null);
            }
        }

        private void InitializeBuffer()
        {
            Assertion.Operation.NotNull(_audioClient);
            Assertion.Operation.NotNull(_outputWaveFormat);

            _bytesPerFrame = _outputWaveFormat.Channels * _outputWaveFormat.BitsPerSample / 8;
            _readBuffer = new byte[_audioClient.BufferSize * _bytesPerFrame];
        }

        private void FinalizeBuffer()
        {
            _bytesPerFrame = 0;
            _readBuffer = null;
        }

        private int FillBuffer(IWaveProvider waveProvider, int frameCount)
        {
            IntPtr buffer = _renderClient.GetBuffer(frameCount);
            int readFrameCount = 0;

            try
            {
                int readLength = waveProvider.Read(_readBuffer, 0, frameCount * _bytesPerFrame);
                if (readLength == 0)
                {
                    // NOTE: 読み込めなかった場合は ReleaseBuffer() しなくてもよい、と書いてあるけど、
                    // 0 フレームをReleaseBufferしないとうまく動かなかったので、呼んでおく。
                    // http://msdn.microsoft.com/en-us/library/windows/desktop/dd368243%28v=vs.85%29.aspx
                    _renderClient.ReleaseBuffer(0, AudioClientBufferFlags.None);
                    return 0;
                }

                Marshal.Copy(_readBuffer, 0, buffer, readLength);
                readFrameCount = readLength / _bytesPerFrame;
            }
            catch
            {
                // NOTE: 読み込めなかった場合は ReleaseBuffer() しなくてもよい、と書いてあるけど、
                // 0 フレームをReleaseBufferしないとうまく動かなかったので、呼んでおく。
                // http://msdn.microsoft.com/en-us/library/windows/desktop/dd368243%28v=vs.85%29.aspx
                _renderClient.ReleaseBuffer(0, AudioClientBufferFlags.None);
                throw;
            }

            _renderClient.ReleaseBuffer(readFrameCount, AudioClientBufferFlags.None);

            return readFrameCount;
        }
    }
}
