﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NAudio.Wave;
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Linq;

namespace NintendoWare.Spy
{
    public class FinalOutWaveSource : ObservableObject, IWaveSource
    {
        private static readonly WaveSourceOutputInfo TvInfo = new WaveSourceOutputInfo(FinalOutSpyModel.TvOutName, FinalOutSpyModel.TvOutChannelCount);
        private static readonly WaveSourceOutputInfo DrcInfo = new WaveSourceOutputInfo(FinalOutSpyModel.DrcOutName, FinalOutSpyModel.DrcOutChannelCount);

        private readonly WeakReference<FinalOutSpyModel> _finalOut = new WeakReference<FinalOutSpyModel>(null);

        private Frame _audioFrameBegin = Frame.InvalidValue;
        private Frame _audioFrameEnd = Frame.InvalidValue;
        private Frame _currentAudioFrame = Frame.InvalidValue;
        private int _currentPlaybackSampleOffset = 0;
        private bool _isEndOfStream = false;

        //-----------------------------------------------------------------

        public int SamplingRate
        {
            get { return FinalOutSpyModel.SampleRate; }
        }

        public int BitsPerSample
        {
            get { return FinalOutSpyModel.BitsPerSample; }
        }

        public Frame AudioFrameBegin
        {
            get { return _audioFrameBegin; }
            set
            {
                if (this.SetPropertyValue(ref _audioFrameBegin, value))
                {
                    this.CurrentAudioFrame = this.ValidateAudioFrame(this.CurrentAudioFrame);
                }
            }
        }

        public Frame AudioFrameEnd
        {
            get { return _audioFrameEnd; }
            set
            {
                if (this.SetPropertyValue(ref _audioFrameEnd, value))
                {
                    this.CurrentAudioFrame = this.ValidateAudioFrame(this.CurrentAudioFrame);
                }
            }
        }

        /// <summary>
        /// 最終波形情報のないフレームを無音で再生します。
        /// </summary>
        public bool PlayMissingFrame { get; set; }

        /// <summary>
        /// 現在再生中のオーディオフレームを取得します。
        /// </summary>
        public Frame CurrentAudioFrame
        {
            get { return _currentAudioFrame; }
            private set { this.SetPropertyValue(ref _currentAudioFrame, this.ValidateAudioFrame(value)); }
        }

        public FinalOutSpyModel FinalOut
        {
            get { return _finalOut.GetTarget(); }
            set
            {
                if (this.FinalOut == value)
                {
                    return;
                }

                _finalOut.SetTarget(value);
                this.NotifyPropertyChanged();
            }
        }

        //-----------------------------------------------------------------

        public FinalOutWaveSource()
        {
            this.PlayMissingFrame = false;
        }

        public void Seek(Frame audioFrame)
        {
            this.CurrentAudioFrame = audioFrame;
            _currentPlaybackSampleOffset = 0;
            _isEndOfStream = false;
        }

        public IEnumerable<WaveSourceOutputInfo> EnumerateWaveSourceOutputInfos()
        {
            yield return TvInfo;
            yield return DrcInfo;
        }

        public int Mix(byte[] buffer, int offset, int count, string targetName, int channelCount)
        {
            Ensure.Argument.True(offset == 0); // NAudio.WaveBuffer は offset を扱えない。

            if (count <= 0)
            {
                return 0;
            }

            if (_currentAudioFrame == Frame.InvalidValue)
            {
                return 0;
            }

            // 再生範囲外なら、何も返さない
            if (_isEndOfStream ||
                this.CurrentAudioFrame < this.AudioFrameBegin ||
                this.AudioFrameEnd < this.CurrentAudioFrame)
            {
                return 0;
            }

            WaveBuffer waveBuffer = new WaveBuffer(buffer);
            var requiredSampleCount = count / sizeof(short) / channelCount; // 16 bit 固定

            var finalOutModel = this.FinalOut;

            // FinalOut データが存在しない場合、無音区間とみなし 0 を埋めて返す
            if (finalOutModel == null)
            {
                this.ClearSamples(waveBuffer, requiredSampleCount * channelCount);
                this.ForwardCurrentFrame(requiredSampleCount);
                return requiredSampleCount * channelCount * sizeof(short);
            }

            if (this.CurrentAudioFrame < finalOutModel.AudioFrameBegin)
            {
                if (this.PlayMissingFrame)
                {
                    // FinalOut データより前のフレームは無音区間とみなし 0 を埋めて返す
                    var silentFrameCount = (int)(finalOutModel.AudioFrameBegin - this.CurrentAudioFrame);
                    var silentSampleCount = Math.Min(silentFrameCount * FinalOutSpyModel.SamplePerFrame - _currentPlaybackSampleOffset, requiredSampleCount);
                    this.ClearSamples(waveBuffer, silentSampleCount * channelCount);
                    this.ForwardCurrentFrame(silentSampleCount);

                    if (silentSampleCount == requiredSampleCount)
                    {
                        return requiredSampleCount * channelCount * sizeof(short);
                    }

                    requiredSampleCount -= silentSampleCount;

                    Assertion.Operation.True(this.CurrentAudioFrame == finalOutModel.AudioFrameBegin);
                }
                else
                {
                    // FinalOut データより前の無音区間をスキップします。
                    this.CurrentAudioFrame = finalOutModel.AudioFrameBegin;
                }
            }

            // CurrentAudioFrame は AudioFrameBegin <= CurrentAudioFrame <= AudioFrameEnd になるように
            // セッターで補正されるため、末尾に達すると末尾のフレームにアクセスし続けてしまいます。
            // そのため、ループ中は CurrentAudioFrame を使わないようにします。
            Frame audioFrame = this.CurrentAudioFrame;

            // FinalOut データ内のフレームをコピーする
            int restSamplesPerChannel = requiredSampleCount;
            int destIndex = 0;

            while (restSamplesPerChannel > 0)
            {
                var copyCount = Math.Min(FinalOutSpyModel.SamplePerFrame - _currentPlaybackSampleOffset, restSamplesPerChannel);

                var finalOut = finalOutModel.GetFinalOut(audioFrame);
                if (finalOut == null)
                {
                    if (this.PlayMissingFrame)
                    {
                        // 無音区間とみなし０で埋めます。
                        for (int i = 0; i < copyCount; ++i)
                        {
                            for (int j = 0; j < channelCount; ++j)
                            {
                                waveBuffer.ShortBuffer[destIndex++] = 0;
                            }
                        }

                        restSamplesPerChannel -= copyCount;
                    }
                }
                else
                {
                    var sampleData = finalOut.SampleData;

                    // ミックス処理
                    switch (targetName)
                    {
                        case FinalOutSpyModel.TvOutName:
                        case null:
                            if (channelCount == 6)
                            {
                                foreach (int sampleIndex in Enumerable.Range(_currentPlaybackSampleOffset, copyCount))
                                {
                                    var l = sampleData.Samples[0][sampleIndex];
                                    var r = sampleData.Samples[1][sampleIndex];
                                    var rl = sampleData.Samples[2][sampleIndex];
                                    var rr = sampleData.Samples[3][sampleIndex];
                                    var fc = sampleData.Samples[4][sampleIndex];
                                    var lfe = sampleData.Samples[5][sampleIndex];

                                    waveBuffer.ShortBuffer[destIndex++] = l;
                                    waveBuffer.ShortBuffer[destIndex++] = r;
                                    waveBuffer.ShortBuffer[destIndex++] = fc;
                                    waveBuffer.ShortBuffer[destIndex++] = lfe;
                                    waveBuffer.ShortBuffer[destIndex++] = rl;
                                    waveBuffer.ShortBuffer[destIndex++] = rr;
                                }
                            }
                            else
                            {
                                foreach (int sampleIndex in Enumerable.Range(_currentPlaybackSampleOffset, copyCount))
                                {
                                    Enumerable.Range((int)FinalOutSpyModel.ChannelIndex.MainFrontLeft, channelCount)
                                        .ForEach(channelIndex => waveBuffer.ShortBuffer[destIndex++] = sampleData.Samples[channelIndex][sampleIndex]);
                                }
                            }
                            break;

                        case FinalOutSpyModel.DrcOutName:
                            foreach (int sampleIndex in Enumerable.Range(_currentPlaybackSampleOffset, copyCount))
                            {
                                Enumerable.Range((int)FinalOutSpyModel.ChannelIndex.DrcLeft, channelCount)
                                    .ForEach(channelIndex => waveBuffer.ShortBuffer[destIndex++] = sampleData.Samples[channelIndex][sampleIndex]);
                            }
                            break;
                    }

                    restSamplesPerChannel -= copyCount;
                }

                // 現在の SampleData 内のサンプルオフセットを更新する
                _currentPlaybackSampleOffset += copyCount;
                if (_currentPlaybackSampleOffset == FinalOutSpyModel.SamplePerFrame)
                {
                    _currentPlaybackSampleOffset = 0;
                    audioFrame += 1;

                    if (!this.PlayMissingFrame && finalOutModel.AudioFrameEnd < audioFrame)
                    {
                        // これ以上データが無いので、再生区間の末尾にジャンプします。
                        audioFrame = this.AudioFrameEnd + 1;
                    }

                    if (this.AudioFrameEnd < audioFrame)
                    {
                        _isEndOfStream = true;
                        break;
                    }
                }
            }

            this.CurrentAudioFrame = audioFrame;
            return destIndex * sizeof(short);
        }

        private void ClearSamples(WaveBuffer buffer, int sampleCount)
        {
            foreach (var sampleIndex in Enumerable.Range(0, sampleCount))
            {
                buffer.ShortBuffer[sampleIndex] = 0;
            }
        }

        private void ForwardCurrentFrame(int sampleCount)
        {
            var newPlaybackSampleOffset = _currentPlaybackSampleOffset + sampleCount;
            var newAudioFrameIndexDelta = newPlaybackSampleOffset / FinalOutSpyModel.SamplePerFrame;
            newPlaybackSampleOffset -= newAudioFrameIndexDelta * FinalOutSpyModel.SamplePerFrame;

            _currentPlaybackSampleOffset = newPlaybackSampleOffset;
            this.CurrentAudioFrame += newAudioFrameIndexDelta;
        }

        private Frame ValidateAudioFrame(Frame audioFrame)
        {
            if (audioFrame == Frame.InvalidValue)
            {
                return this.AudioFrameBegin;
            }

            if (audioFrame < this.AudioFrameBegin)
            {
                return this.AudioFrameBegin;
            }

            if (this.AudioFrameEnd < audioFrame)
            {
                return this.AudioFrameEnd;
            }

            return audioFrame;
        }
    }
}
