﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Linq;

namespace NintendoWare.Spy
{
    /// <summary>
    /// <see cref="Spy.PlotSpyModel"/> の変化を監視します。
    /// </summary>
    public class PlotSpyModelWatcher
    {
        private class WatchInfo
        {
            public object ObserverOwner { get; set; }
        }

        private readonly ObservableCollection<string> _floatFullNames = new ObservableCollection<string>();

        private readonly ObservableCollection<string> _stateFullNames = new ObservableCollection<string>();

        private readonly ObservableCollection<string> _nodeFullNames = new ObservableCollection<string>();

        private readonly Dictionary<string, WatchInfo> _floatWatchInfos = new Dictionary<string, WatchInfo>();

        private readonly Dictionary<string, WatchInfo> _stateWatchInfos = new Dictionary<string, WatchInfo>();

        private readonly Dictionary<string, WatchInfo> _nodeWatchInfos = new Dictionary<string, WatchInfo>();

        private PlotSpyModel _model;

        private object _observerOwner;

        /// <summary>
        /// 監視したい <see cref="PlotSpyModel.PlotFloat"/> のフルネームのコレクションです。
        /// <para>
        /// フルネームを登録しておくと、要素が追加されたとき、または値が変化したときに
        /// <see cref="Changed"/> イベントが発行されます。
        /// イベントハンドラの <c>sender</c> 引数には、変化した要素が渡されます。
        /// </para>
        /// </summary>
        public IList<string> Floats => _floatFullNames;

        /// <summary>
        /// 監視したい <see cref="PlotSpyModel.PlotState"/> のフルネームのコレクションです。
        /// <para>
        /// フルネームを登録しておくと、要素が追加されたとき、または値が変化したときに
        /// <see cref="Changed"/> イベントが発行されます。
        /// イベントハンドラの <c>sender</c> 引数には、変化した要素が渡されます。
        /// </para>
        /// </summary>
        public IList<string> States => _stateFullNames;

        /// <summary>
        /// 監視したい <see cref="PlotSpyModel.PlotNode"/> のフルネームのコレクションです。
        /// <para>
        /// フルネームを登録しておくと、要素が追加されたとき、または値が変化したときに
        /// <see cref="Changed"/> イベントが発行されます。
        /// イベントハンドラの <c>sender</c> 引数には、変化した要素が渡されます。
        /// </para>
        /// </summary>
        public IList<string> Nodes => _nodeFullNames;

        /// <summary>
        /// <see cref="Spy.PlotSpyModel"/> を取得または設定します。
        /// 設定した場合、監視対象がすでに存在するときは、ただちにイベントが発行されます。
        /// </summary>
        public PlotSpyModel PlotSpyModel
        {
            get { return _model; }
            set { this.SetPlotSpyModel(value); }
        }

        /// <summary>
        /// 監視中であるかを取得します。
        /// </summary>
        public bool IsWatching { get; private set; }

        /// <summary>
        /// <see cref="Spy.PlotSpyModel"/> の監視対象に変化があったときに通知します。
        /// </summary>
        public event EventHandler Changed;

        public PlotSpyModelWatcher()
        {
            _floatFullNames.CollectionChanged += (sender, args) =>
            {
                this.HandleCollectionChanged(args, AddFloatFullName, RemoveFloatFullName, ResetFloatFullNames);
            };

            _stateFullNames.CollectionChanged += (sender, args) =>
            {
                this.HandleCollectionChanged(args, AddStateFullName, RemoveStateFullName, ResetStateFullNames);
            };

            _nodeFullNames.CollectionChanged += (sender, args) =>
            {
                this.HandleCollectionChanged(args, AddNodeFullName, RemoveNodeFullName, ResetFullNames);
            };
        }

        /// <summary>
        /// 監視を開始します。
        /// <para>
        /// 監視対象がすでに存在するときは、ただちにイベントが発行されます。
        /// 終了するには <see cref="Stop"/> します。
        /// </para>
        /// </summary>
        public void Start()
        {
            if (this.IsWatching)
            {
                return;
            }

            this.IsWatching = true;

            if (_model != null)
            {
                this.InitializeObservation(_model);

                this.CheckExsistedWatchTarget(_model);
            }
        }

        /// <summary>
        /// 監視を終了します。
        /// </summary>
        public void Stop()
        {
            if (this.IsWatching == false)
            {
                return;
            }

            if (_model != null)
            {
                this.FinalizeObservation(_model);
            }

            this.IsWatching = false;
        }

        private void SetPlotSpyModel(PlotSpyModel model)
        {
            if (_model == model)
            {
                return;
            }

            if (this.IsWatching)
            {
                if (_model != null)
                {
                    this.FinalizeObservation(_model);
                }

                if (model != null)
                {
                    this.InitializeObservation(model);

                    this.CheckExsistedWatchTarget(model);
                }
            }

            _model = model;
        }

        private void HandleCollectionChanged(
            NotifyCollectionChangedEventArgs args,
            Action<string> addFunc,
            Action<string> removeFunc,
            Action resetFunc)
        {
            if (args.Action == NotifyCollectionChangedAction.Reset)
            {
                resetFunc();
            }
            else
            {
                if (args.Action == NotifyCollectionChangedAction.Remove ||
                    args.Action == NotifyCollectionChangedAction.Replace)
                {
                    args.OldItems
                        .Cast<string>()
                        .ForEach(it => removeFunc(it));
                }

                if (args.Action == NotifyCollectionChangedAction.Add ||
                    args.Action == NotifyCollectionChangedAction.Replace)
                {
                    args.NewItems
                        .Cast<string>()
                        .ForEach(it => addFunc(it));
                }
            }
        }

        private void InitializeObservation(PlotSpyModel model)
        {
            _observerOwner = new object();

            CollectionChangedObservation.GetObserver(_observerOwner, model.Floats).AddHandlerForAddItems(
                (items, info) =>
                {
                    items
                        .Cast<PlotSpyModel.PlotFloat>()
                        .Where(it => _floatWatchInfos.ContainsKey(it.FullName))
                        .ForEach(it => this.OnPlotFloatAdded(it, it.FullName));
                });

            CollectionChangedObservation.GetObserver(_observerOwner, model.States).AddHandlerForAddItems(
                (items, info) =>
                {
                    items
                        .Cast<PlotSpyModel.PlotState>()
                        .Where(it => _stateWatchInfos.ContainsKey(it.FullName))
                        .ForEach(it => this.OnPlotStateAdded(it, it.FullName));
                });

            CollectionChangedObservation.GetObserver(_observerOwner, model.Nodes).AddHandlerForAddItems(
                (items, info) =>
                {
                    items
                        .Cast<PlotSpyModel.PlotNode>()
                        .Where(it => _nodeWatchInfos.ContainsKey(it.FullName))
                        .ForEach(it => this.OnPlotNodeAdded(it, it.FullName));
                });
        }

        private void FinalizeObservation(PlotSpyModel model)
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);
            _observerOwner = null;

            _floatWatchInfos.Values.ForEach(it => FinalizeObservation(it));

            _stateWatchInfos.Values.ForEach(it => FinalizeObservation(it));

            _nodeWatchInfos.Values.ForEach(it => FinalizeObservation(it));
        }

        private void InitializeObservation(PlotSpyModel.PlotFloat plotFloat, WatchInfo watchInfo)
        {
            if (watchInfo.ObserverOwner == null)
            {
                watchInfo.ObserverOwner = new object();

                CollectionChangedObservation.GetObserver(watchInfo.ObserverOwner, plotFloat.Values).AddHandlerForAddItems(
                    (items, info) => this.OnPlotFloatChanged(plotFloat));
            }
        }

        private void InitializeObservation(PlotSpyModel.PlotState plotState, WatchInfo watchInfo)
        {
            if (watchInfo.ObserverOwner == null)
            {
                watchInfo.ObserverOwner = new object();

                CollectionChangedObservation.GetObserver(watchInfo.ObserverOwner, plotState.Values).AddHandlerForAddItems(
                    (items, info) => this.OnPlotStateChanged(plotState));
            }
        }

        private void InitializeObservation(PlotSpyModel.PlotNode plotNode, WatchInfo watchInfo)
        {
            if (watchInfo.ObserverOwner == null)
            {
                watchInfo.ObserverOwner = new object();

                CollectionChangedObservation.GetObserver(watchInfo.ObserverOwner, plotNode.Floats).AddHandlerForAddItems(
                    (items, info) => this.OnPlotNodeChanged(plotNode));

                CollectionChangedObservation.GetObserver(watchInfo.ObserverOwner, plotNode.States).AddHandlerForAddItems(
                    (items, info) => this.OnPlotNodeChanged(plotNode));
            }
        }

        private void FinalizeObservation(WatchInfo watchInfo)
        {
            if (watchInfo.ObserverOwner != null)
            {
                CollectionChangedObservation.RemoveObservers(watchInfo.ObserverOwner);
                watchInfo.ObserverOwner = null;
            }
        }

        private void CheckExsistedWatchTarget(PlotSpyModel model)
        {
            foreach (var key in _floatWatchInfos.Keys)
            {
                var item = model.GetFloat(key);
                if (item != null)
                {
                    this.OnPlotFloatAdded(item, key);
                }
            }

            foreach (var key in _stateWatchInfos.Keys)
            {
                var item = model.GetState(key);
                if (item != null)
                {
                    this.OnPlotStateAdded(item, key);
                }
            }

            foreach (var key in _nodeWatchInfos.Keys)
            {
                var item = model.GetNode(key);
                if (item != null)
                {
                    this.OnPlotNodeAdded(item, key);
                }
            }
        }

        private void AddFloatFullName(string key)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(key);

            _floatWatchInfos.Add(key, new WatchInfo());

            if (this.IsWatching)
            {
                // すでに存在する要素についてイベントを発行します。
                var plotFloat = _model.GetFloat(key);
                if (plotFloat != null)
                {
                    this.OnPlotFloatAdded(plotFloat, key);
                }
            }
        }

        private void RemoveFloatFullName(string key)
        {
            _floatWatchInfos.Remove(key);
        }

        private void ResetFloatFullNames()
        {
            _floatWatchInfos.Clear();
            _floatFullNames.ForEach(it => this.AddFloatFullName(it));
        }

        private void AddStateFullName(string key)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(key);

            _stateWatchInfos.Add(key, new WatchInfo());

            if (this.IsWatching)
            {
                // すでに存在する要素についてイベントを発行します。
                var plotState = _model.GetState(key);
                if (plotState != null)
                {
                    this.OnPlotStateAdded(plotState, key);
                }
            }
        }

        private void RemoveStateFullName(string key)
        {
            _stateWatchInfos.Remove(key);
        }

        private void ResetStateFullNames()
        {
            _stateWatchInfos.Clear();
            _stateFullNames.ForEach(it => this.AddStateFullName(it));
        }

        private void AddNodeFullName(string key)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(key);

            _nodeWatchInfos.Add(key, new WatchInfo());

            if (this.IsWatching)
            {
                // すでに存在する要素についてイベントを発行します。
                var plotNode = _model.GetNode(key);
                if (plotNode != null)
                {
                    this.OnPlotNodeAdded(plotNode, key);
                }
            }
        }

        private void RemoveNodeFullName(string key)
        {
            _nodeWatchInfos.Remove(key);
        }

        private void ResetFullNames()
        {
            _nodeWatchInfos.Clear();
            _nodeFullNames.ForEach(it => this.AddNodeFullName(it));
        }
        private void OnPlotFloatAdded(PlotSpyModel.PlotFloat plotFloat, string fullName)
        {
            WatchInfo watchInfo;
            if (_floatWatchInfos.TryGetValue(fullName, out watchInfo))
            {
                this.InitializeObservation(plotFloat, watchInfo);
                this.OnPlotFloatChanged(plotFloat);
            }
        }

        private void OnPlotFloatChanged(PlotSpyModel.PlotFloat plotFloat)
        {
            this.Changed?.Invoke(plotFloat, EventArgs.Empty);
        }

        private void OnPlotStateAdded(PlotSpyModel.PlotState plotState, string fullName)
        {
            WatchInfo watchInfo;
            if (_stateWatchInfos.TryGetValue(fullName, out watchInfo))
            {
                this.InitializeObservation(plotState, watchInfo);
                this.OnPlotStateChanged(plotState);
            }
        }

        private void OnPlotStateChanged(PlotSpyModel.PlotState plotState)
        {
            this.Changed?.Invoke(plotState, EventArgs.Empty);
        }

        private void OnPlotNodeAdded(PlotSpyModel.PlotNode plotNode, string fullName)
        {
            WatchInfo watchInfo;
            if (_nodeWatchInfos.TryGetValue(fullName, out watchInfo))
            {
                this.InitializeObservation(plotNode, watchInfo);
                this.OnPlotNodeChanged(plotNode);
            }
        }

        private void OnPlotNodeChanged(PlotSpyModel.PlotNode plotNode)
        {
            this.Changed?.Invoke(plotNode, EventArgs.Empty);
        }
    }
}
