﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Diagnostics.CodeAnalysis;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Media;

namespace NintendoWare.Spy
{
    /// <summary>
    /// Plot Spy モデルです。
    /// </summary>
    public sealed class PlotSpyModel : SpyModel
    {
        /// <summary>
        /// フルネームで要素の名前の間に挿入される文字です。
        /// </summary>
        public const char NameSeparator = '/';

        /// <summary>
        /// Spy が内部で使用するアイテムのフルネームに付く接頭辞です。
        /// </summary>
        public const string SpyInternalPrefix = "@";

        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_1_1_0_0 = new Version(1, 1, 0, 0);
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_1_1_1_0 = new Version(1, 1, 1, 0);
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_1_2_0_0 = new Version(1, 2, 0, 0);

        /// <summary>
        /// バージョン 1.3.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// パケット共通データ
        /// struct PacketCommonData {
        ///     u8 dataType;
        ///     u8 padding[3];
        /// };
        ///
        /// // 共通メタデータ
        /// struct ItemMetaData {
        ///     u64 id;
        ///     u64 parentId;
        ///     u8 color[4];
        ///     StringData(u8, 255) name;
        /// };
        ///
        /// struct FloatMetadataPacket {
        ///     PacketCommonData common;
        ///     double minValue;
        ///     double maxValue;
        ///     u8 interpolationMode;
        ///     u8 padding[3];
        ///     ItemMetaData itemMetaData;
        /// };
        ///
        /// struct FloatValuePacket {
        ///     PacketCommonData common;
        ///     u64 id;
        ///     double value;
        /// };
        ///
        /// struct ResetPacket {
        ///     PacketCommonData common;
        /// };
        ///
        /// struct NodeMetadataPacket {
        ///     PacketCommonData common;
        ///     ItemMetaData itemMetaData;
        /// };
        ///
        /// struct NodeDetachPacket {
        ///     PacketCommonData common;
        ///     u64 id;
        /// };
        ///
        /// struct StateMetadataPacket {
        ///     PacketCommonData common;
        ///     ItemMetaData itemMetaData;
        /// };
        ///
        /// struct StateValuePacket {
        ///     PacketCommonData common;
        ///     u64 id;
        ///     u8 color[4];
        ///     StringData(u8, 255) state;
        /// };
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_1_3_0_0 = new Version(1, 3, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(1, 4, 0, 0);

        private const int MaxItemNameLength = 255;
        private const int MaxNodeNameLength = 255;
        private const int MaxStateNameLength = 255;
        private const int MaxStateValueLength = 255;
        private const ulong InvalidId = 0;

        private readonly ItemCollection<PlotFloat> _floats = new ItemCollection<PlotFloat>();
        private readonly ItemCollection<PlotState> _states = new ItemCollection<PlotState>();
        private readonly ItemCollection<PlotNode> _nodes = new ItemCollection<PlotNode>();
        private bool _errorUnexpectedDataVersion = false;

        /// <summary>
        /// <see cref="PlotFloat"/> のリストです。
        /// コレクション末尾へのアイテムの追加のみ行われます。
        /// </summary>
        public IPlotObservableList<PlotFloat> Floats => _floats.Items;

        /// <summary>
        /// <see cref="PlotState"/> のリストです。
        /// コレクション末尾へのアイテムの追加のみ行われます。
        /// </summary>
        public IPlotObservableList<PlotState> States => _states.Items;

        /// <summary>
        /// <see cref="PlotNode"/> のリストです。
        /// コレクション末尾へのアイテムの追加のみ行われます。
        /// </summary>
        public IPlotObservableList<PlotNode> Nodes => _nodes.Items;

        /// <summary>
        /// サポートしないバージョンのデータを受信すると true に設定されます。
        /// </summary>
        public bool ErrorUnexpectedDataVersion
        {
            get { return _errorUnexpectedDataVersion; }
            private set { this.SetPropertyValue(ref _errorUnexpectedDataVersion, value); }
        }

        /// <summary>
        /// 要素の名前を連結してフルネームを作ります。
        /// 名前の間にはセパレータとして <see cref="NameSeparator"/> が挟まれます。
        /// </summary>
        /// <param name="parentsFullName"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        public static string CreateFullName(string parentsFullName, string name)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(name);

            if (string.IsNullOrEmpty(parentsFullName))
            {
                return name;
            }
            else
            {
                return parentsFullName + NameSeparator + name;
            }
        }

        /// <summary>
        /// フルネームを指定して <see cref="PlotFloat"/> を得ます。
        /// </summary>
        /// <param name="fullName">親階層の名前を含めたフルネームです。</param>
        /// <returns><see cref="PlotFloat"/> または null を返します。</returns>
        public PlotFloat GetFloat(string fullName)
        {
            return _floats.GetItem(fullName);
        }

        /// <summary>
        /// フルネームを指定して <see cref="PlotState"/> を得ます。
        /// </summary>
        /// <param name="fullName">親階層の名前を含めたフルネームです。</param>
        /// <returns><see cref="PlotState"/> または null を返します。</returns>
        public PlotState GetState(string fullName)
        {
            return _states.GetItem(fullName);
        }

        /// <summary>
        /// フルネームを指定して <see cref="PlotNode"/> を得ます。
        /// </summary>
        /// <param name="fullName">親階層の名前を含めたフルネームです。</param>
        /// <returns><see cref="PlotNode"/> または null を返します。</returns>
        public PlotNode GetNode(string fullName)
        {
            return _nodes.GetItem(fullName);
        }

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                this.ErrorUnexpectedDataVersion = true;
                return;
            }

            var reader = CreateDataReader(dataBlock);

            ulong id = InvalidId;
            if (this.DataVersion < Version_1_1_0_0)
            {
                id = reader.ReadUInt32();
            }

            var dataType = (PlotItemDataType)reader.ReadByte();

            // padding
            reader.ReadBytes(3);

            switch (dataType)
            {
                case PlotItemDataType.FloatValue:
                    this.ReadPlotFloatValue(id, reader, dataBlock);
                    break;

                case PlotItemDataType.FloatMetadata:
                    this.ReadPlotFloatMetadata(id, reader, dataBlock);
                    break;

                case PlotItemDataType.Reset:
                    this.ReadResetPacket(reader, dataBlock);
                    break;

                case PlotItemDataType.NodeMetadata:
                    this.ReadNodeMetadataPacket(reader, dataBlock);
                    break;

                case PlotItemDataType.NodeDetach:
                    this.ReadNodeDetachPacket(reader, dataBlock);
                    break;

                case PlotItemDataType.StateMetadata:
                    this.ReadStateMetadataPacket(reader, dataBlock);
                    break;

                case PlotItemDataType.StateValue:
                    this.ReadStateValuePacket(reader, dataBlock);
                    break;
            }
        }

        private void ReadPlotFloatMetadata(ulong id, BinaryReader reader, SpyDataBlock dataBlock)
        {
            ulong parentId = InvalidId;
            string name;
            Color color;
            double minimum;
            double maximum;
            byte interpolationMode;

            if (this.DataVersion >= Version_1_3_0_0)
            {
                minimum = reader.ReadDouble();
                maximum = reader.ReadDouble();

                interpolationMode = reader.ReadByte();
                reader.ReadBytes(3); // padding

                ReadItemMetaData(reader, dataBlock, out id, out parentId, out color, out name);
            }
            else
            {
                if (this.DataVersion >= Version_1_2_0_0)
                {
                    id = reader.ReadUInt64();
                    parentId = reader.ReadUInt64();
                }
                else
                {
                    if (this.DataVersion >= Version_1_1_0_0)
                    {
                        id = reader.ReadUInt32();
                    }
                }

                minimum = reader.ReadDouble();
                maximum = reader.ReadDouble();

                color = ReadColor(reader, dataBlock);

                interpolationMode = reader.ReadByte();
                reader.ReadBytes(3);    // padding

                var nameLength = reader.ReadByte();

                name = Encoding.UTF8.GetString(reader.ReadBytes(MaxItemNameLength), 0, nameLength);

                if (this.DataVersion < Version_1_2_0_0)
                {
                    if (this.DataVersion >= Version_1_1_0_0)
                    {
                        parentId = reader.ReadUInt32();
                    }
                }
            }

            // 無効な ID なら無視する
            if (id == InvalidId)
            {
                return;
            }

            var belongingFrame = this.GetBelongingFrame(dataBlock.Timestamp);
            var attachTime = new SpyTime(dataBlock.Timestamp, belongingFrame);

            var parent = _nodes.GetItem(parentId);

            name = SanitizeName(name);
            string fullName = CreateFullName(parent?.FullName, name);

            var item = _floats.GetItem(fullName);
            if (item != null)
            {
                _floats.AddConflicted(id, item);
            }
            else
            {
                item = new PlotFloat(
                    name,
                    fullName,
                    minimum,
                    maximum,
                    color,
                    this.ByteToPlotFloatInterpolationMode(interpolationMode),
                    parent,
                    attachTime);

                var notify = new List<Action>();

                if (parent != null)
                {
                    parent.AddFloat(item, notify);
                }

                _floats.Add(fullName, id, item, notify);

                // ツリー構造の変更が確定してから通知します。
                notify.ForEach(it => it());
            }
        }

        private PlotFloatInterpolationMode ByteToPlotFloatInterpolationMode(byte value)
        {
            switch (value)
            {
                case (byte)PlotFloatInterpolationMode.None:
                case (byte)PlotFloatInterpolationMode.Linear:
                    return (PlotFloatInterpolationMode)value;
            }

            return PlotFloatInterpolationMode.None;
        }

        private void ReadPlotFloatValue(ulong id, BinaryReader reader, SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= Version_1_2_0_0)
            {
                id = reader.ReadUInt64();
            }
            else
            {
                if (this.DataVersion >= Version_1_1_0_0)
                {
                    id = reader.ReadUInt32();
                }
            }

            // 無効な ID なら無視する
            if (id == InvalidId)
            {
                return;
            }

            var value = reader.ReadDouble();

            var item = _floats.GetItem(id);

            // 未検出の ID は無視する
            if (item == null)
            {
                return;
            }

            // 古いデータが後から来たときは無視する
            if (!item.Values.IsEmpty() && dataBlock.Timestamp < item.Values.Last().Timestamp)
            {
                return;
            }

            var belongingFrame = this.GetBelongingFrame(dataBlock.Timestamp);
            var time = new SpyTime(dataBlock.Timestamp, belongingFrame);

            item.AddValue(
                new PlotFloatValue(value, time));
        }

        private void ReadResetPacket(BinaryReader reader, SpyDataBlock dataBlock)
        {
        }

        private void ReadNodeMetadataPacket(BinaryReader reader, SpyDataBlock dataBlock)
        {
            ulong id;
            ulong parentId = InvalidId;
            string name;
            Color color = Colors.Gray;

            if (this.DataVersion >= Version_1_3_0_0)
            {
                ReadItemMetaData(reader, dataBlock, out id, out parentId, out color, out name);
            }
            else
            {
                if (this.DataVersion >= Version_1_2_0_0)
                {
                    id = reader.ReadUInt64();
                }
                else
                {
                    id = reader.ReadUInt32();
                }

                // 無効な ID なら無視する
                if (id == InvalidId)
                {
                    return;
                }

                var nameLength = reader.ReadByte();
                name = Encoding.UTF8.GetString(reader.ReadBytes(MaxNodeNameLength), 0, nameLength);
            }

            // 無効な ID なら無視する
            if (id == InvalidId)
            {
                return;
            }

            var belongingFrame = GetBelongingFrame(dataBlock.Timestamp);
            var time = new SpyTime(dataBlock.Timestamp, belongingFrame);

            var parent = _nodes.GetItem(parentId);

            name = SanitizeName(name);
            string fullName = CreateFullName(parent?.FullName, name);

            PlotNode node = _nodes.GetItem(fullName);
            if (node != null)
            {
                _nodes.AddConflicted(id, node);
            }
            else
            {
                node = new PlotNode(
                    name,
                    fullName,
                    color,
                    parent,
                    time);

                // ツリー構造の変更が確定してから通知します。
                var notify = new List<Action>();

                if (parent != null)
                {
                    parent.AddChildNode(node, notify);
                }

                _nodes.Add(fullName, id, node, notify);

                notify.ForEach(it => it());
            }
        }

        private void ReadNodeDetachPacket(BinaryReader reader, SpyDataBlock dataBlock)
        {
            ulong id;
            if (this.DataVersion >= Version_1_2_0_0)
            {
                id = reader.ReadUInt64();
            }
            else
            {
                id = reader.ReadUInt32();
            }

            // 無効な ID なら無視する
            if (id == InvalidId)
            {
                return;
            }

            _nodes.Remove(id);
        }

        private void ReadStateMetadataPacket(BinaryReader reader, SpyDataBlock dataBlock)
        {
            if (this.DataVersion < Version_1_1_1_0)
            {
                return;
            }

            ulong id;
            ulong parentId = InvalidId;
            Color color = Colors.Gray;
            string name;

            if (this.DataVersion >= Version_1_3_0_0)
            {
                ReadItemMetaData(reader, dataBlock, out id, out parentId, out color, out name);
            }
            else
            {
                if (this.DataVersion >= Version_1_2_0_0)
                {
                    id = reader.ReadUInt64();
                    parentId = reader.ReadUInt64();
                }
                else
                {
                    id = reader.ReadUInt32();
                }

                // 無効な ID なら無視する
                if (id == InvalidId)
                {
                    return;
                }

                var nameLength = reader.ReadByte();

                name = Encoding.UTF8.GetString(reader.ReadBytes(MaxStateNameLength), 0, nameLength);

                if (this.DataVersion < Version_1_2_0_0)
                {
                    parentId = reader.ReadUInt32();
                }
            }

            // 無効な ID なら無視する
            if (id == InvalidId)
            {
                return;
            }

            var belongingFrame = this.GetBelongingFrame(dataBlock.Timestamp);
            var attachTime = new SpyTime(dataBlock.Timestamp, belongingFrame);

            var parent = _nodes.GetItem(parentId);

            name = SanitizeName(name);
            string fullName = CreateFullName(parent?.FullName, name);

            var state = _states.GetItem(fullName);
            if (state != null)
            {
                _states.AddConflicted(id, state);
            }
            else
            {
                state = new PlotState(
                    name,
                    fullName,
                    color,
                    parent,
                    attachTime);

                var notify = new List<Action>();

                if (parent != null)
                {
                    parent.AddState(state, notify);
                }

                _states.Add(fullName, id, state, notify);

                // ツリー構造の変更が確定してから通知します。
                notify.ForEach(it => it());
            }
        }

        private void ReadStateValuePacket(BinaryReader reader, SpyDataBlock dataBlock)
        {
            if (this.DataVersion < Version_1_1_1_0)
            {
                return;
            }

            ulong id;
            if (this.DataVersion >= Version_1_2_0_0)
            {
                id = reader.ReadUInt64();
            }
            else
            {
                id = reader.ReadUInt32();
            }

            // 無効な ID なら無視する
            if (id == InvalidId)
            {
                return;
            }

            var color = ReadColor(reader, dataBlock);

            var valueLength = reader.ReadByte();

            string value = Encoding.UTF8.GetString(reader.ReadBytes(MaxStateValueLength), 0, valueLength);

            var state = _states.GetItem(id);

            // 未検出の ID は無視する
            if (state == null)
            {
                return;
            }

            // 古いデータが後から来たときは無視する
            if (!state.Values.IsEmpty() && dataBlock.Timestamp < state.Values.Last().Time.Timestamp)
            {
                return;
            }

            var belongingFrame = this.GetBelongingFrame(dataBlock.Timestamp);
            var time = new SpyTime(dataBlock.Timestamp, belongingFrame);

            state.AddValue(
                new PlotStateValue(
                    value,
                    color,
                    time));
        }

        private void ReadItemMetaData(
            BinaryReader reader,
            SpyDataBlock dataBlock,
            out ulong id,
            out ulong parentId,
            out Color color,
            out string name)
        {
            id = reader.ReadUInt64();

            parentId = reader.ReadUInt64();

            color = ReadColor(reader, dataBlock);

            var nameLength = reader.ReadByte();
            name = Encoding.UTF8.GetString(reader.ReadBytes(MaxItemNameLength), 0, nameLength);
        }

        private Color ReadColor(BinaryReader reader, SpyDataBlock dataBlock)
        {
            var r = reader.ReadByte();
            var g = reader.ReadByte();
            var b = reader.ReadByte();
            var a = reader.ReadByte();

            return Color.FromArgb(a, r, g, b);
        }

        private static string SanitizeName(string name)
        {
            if (string.IsNullOrEmpty(name))
            {
                return "()";
            }
            else
            {
                return name;
            }
        }

        public enum PlotFloatInterpolationMode
        {
            None = 0,
            Linear,
        }

        public class PlotFloat
        {
            private readonly ValueCollection<PlotFloatValue> _values = new ValueCollection<PlotFloatValue>();

            public PlotFloat(
                string name,
                string fullName,
                double minimum,
                double maximum,
                Color color,
                PlotFloatInterpolationMode interpolation,
                PlotNode parent,
                SpyTime attachTime)
            {
                this.Name = name;
                this.FullName = fullName;
                this.Minimum = minimum;
                this.Maximum = maximum;
                this.Color = color;
                this.InterpolationMode = interpolation;
                this.Parent = parent;
                this.AttachTime = attachTime;
            }

            public string Name { get; }
            public string FullName { get; }
            public double Minimum { get; }
            public double Maximum { get; }
            public Color Color { get; }
            public PlotFloatInterpolationMode InterpolationMode { get; }
            public PlotNode Parent { get; }
            public SpyTime AttachTime { get; }

            /// <summary>
            /// 値のリストです。
            /// コレクション末尾へのアイテムの追加のみ行われます。
            /// </summary>
            public IPlotObservableList<PlotFloatValue> Values => _values;

            /// <summary>
            /// 指定した時刻の値を取得します。
            /// </summary>
            /// <param name="time">時刻です。</param>
            /// <returns>
            /// 指定した時刻の値を返します。
            /// <para>
            /// 指定した時刻の値が無いときは、直前の値を返します。
            /// 指定した時刻に複数の値があるときは、その中の最後の値を返します。
            /// 指定した時刻よりも前の値が無いときは null を返します。
            /// </para>
            /// </returns>
            public PlotFloatValue FindValue(SpyGlobalTime time)
            {
                if (this.Values.Count == 0)
                {
                    return null;
                }

                var index = BinarySearchUtility.BinarySearch(this.Values, time, value => value.Timestamp, BinarySearchUtility.Options.BiggestIndex);

                if (index < 0)
                {
                    index = ~index - 1;

                    if (index < 0)
                    {
                        return null;
                    }
                }

                return this.Values[index];
            }

            internal void AddValue(PlotFloatValue value)
            {
                _values.Add(value);
            }
        }

        public class PlotFloatValue
        {
            public PlotFloatValue(double value, SpyTime time)
            {
                this.Value = value;
                this.Time = time;
            }

            public double Value { get; }
            public SpyTime Time { get; }
            public SpyGlobalTime Timestamp => this.Time.Timestamp;

            public override string ToString()
            {
                return string.Format("Timestamp={0}, Value={1}", this.Timestamp, this.Value);
            }
        }

        public class PlotState
        {
            private readonly ValueCollection<PlotStateValue> _values = new ValueCollection<PlotStateValue>();

            public PlotState(
                string name,
                string fullName,
                Color color,
                PlotNode parent,
                SpyTime attachTime)
            {
                this.Name = name;
                this.FullName = fullName;
                this.Color = color;
                this.Parent = parent;
                this.AttachTime = attachTime;
            }

            public string Name { get; }
            public string FullName { get; }
            public Color Color { get; }
            public PlotNode Parent { get; }
            public SpyTime AttachTime { get; }

            /// <summary>
            /// 値のリストです。
            /// コレクション末尾へのアイテムの追加のみ行われます。
            /// </summary>
            public IPlotObservableList<PlotStateValue> Values => _values;

            /// <summary>
            /// 指定した時刻の値を取得します。
            /// </summary>
            /// <param name="time">時刻です。</param>
            /// <returns>
            /// 指定した時刻の値を返します。
            /// <para>
            /// 指定した時刻の値が無いときは、直前の値を返します。
            /// 指定した時刻に複数の値があるときは、その中の最後の値を返します。
            /// 指定した時刻よりも前の値が無いときは null を返します。
            /// </para>
            /// </returns>
            public PlotStateValue FindValue(SpyGlobalTime time)
            {
                if (this.Values.Count == 0)
                {
                    return null;
                }

                var index = BinarySearchUtility.BinarySearch(this.Values, time, value => value.Timestamp, BinarySearchUtility.Options.BiggestIndex);

                if (index < 0)
                {
                    index = ~index - 1;

                    if (index < 0)
                    {
                        return null;
                    }
                }

                return this.Values[index];
            }

            internal void AddValue(PlotStateValue value)
            {
                _values.Add(value);
            }
        }

        public class PlotStateValue
        {
            public PlotStateValue(
                string value,
                Color color,
                SpyTime time)
            {
                this.Value = value;
                this.Color = color;
                this.Time = time;
            }

            public string Value { get; private set; }
            public Color Color { get; private set; }
            public SpyTime Time { get; private set; }
            public SpyGlobalTime Timestamp => this.Time.Timestamp;
        }

        public class PlotNode
        {
            private readonly ChildCollection<PlotFloat> _floats = new ChildCollection<PlotFloat>();
            private readonly ChildCollection<PlotState> _states = new ChildCollection<PlotState>();
            private readonly ChildCollection<PlotNode> _nodes = new ChildCollection<PlotNode>();

            public PlotNode(
                string name,
                string fullName,
                Color color,
                PlotNode parent,
                SpyTime attachTime)
            {
                this.Name = name;
                this.FullName = fullName;
                this.Color = color;
                this.Parent = parent;
                this.AttachTime = attachTime;
            }

            public string Name { get; }

            public string FullName { get; }

            public Color Color { get; }

            public PlotNode Parent { get; }

            public SpyTime AttachTime { get; }

            /// <summary>
            /// ノードの子の <see cref="PlotFloat"/> のリストです。
            /// コレクション末尾へのアイテムの追加のみ行われます。
            /// </summary>
            public IPlotObservableList<PlotFloat> Floats => _floats.Items;

            /// <summary>
            /// ノードの子の <see cref="PlotState"/> のリストです。
            /// コレクション末尾へのアイテムの追加のみ行われます。
            /// </summary>
            public IPlotObservableList<PlotState> States => _states.Items;

            /// <summary>
            /// ノードの子の <see cref="PlotNode"/> のリストです。
            /// コレクション末尾へのアイテムの追加のみ行われます。
            /// </summary>
            public IPlotObservableList<PlotNode> ChildNodes => _nodes.Items;

            /// <summary>
            /// 名前を指定して、ノードの子の <see cref="PlotFloat"/> を得ます。
            /// </summary>
            /// <param name="name"></param>
            /// <returns><see cref="PlotFloat"/> または null を返します。</returns>
            public PlotFloat GetFloat(string name)
            {
                return _floats.GetItem(name);
            }

            /// <summary>
            /// 名前を指定して、ノードの子の <see cref="PlotState"/> を得ます。
            /// </summary>
            /// <param name="name"></param>
            /// <returns><see cref="PlotState"/> または null を返します。</returns>
            public PlotState GetState(string name)
            {
                return _states.GetItem(name);
            }

            /// <summary>
            /// 名前を指定して、ノードの子の <see cref="PlotNode"/> を得ます。
            /// </summary>
            /// <param name="name"></param>
            /// <returns><see cref="PlotNode"/> または null を返します。</returns>
            public PlotNode GetChildNode(string name)
            {
                return _nodes.GetItem(name);
            }

            internal void AddFloat(PlotFloat item, ICollection<Action> notify)
            {
                if (_floats.ContainsKey(item.Name) == false)
                {
                    _floats.Add(item.Name, item, notify);
                }
            }

            internal void AddState(PlotState state, ICollection<Action> notify)
            {
                if (_states.ContainsKey(state.Name) == false)
                {
                    _states.Add(state.Name, state, notify);
                }
            }

            internal void AddChildNode(PlotNode node, ICollection<Action> notify)
            {
                if (_nodes.ContainsKey(node.Name) == false)
                {
                    _nodes.Add(node.Name, node, notify);
                }
            }
        }

        private enum PlotItemDataType : byte
        {
            FloatValue = 0,
            FloatMetadata = 1,
            Reset = 2,
            NodeMetadata = 3,
            NodeDetach = 4,
            StateMetadata = 5,
            StateValue = 6,
        }

        /// <summary>
        /// PlotSpyModel クラスの提供するコレクションのインターフェースです。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        public interface IPlotObservableList<T> : IList<T>, INotifyCollectionChanged, INotifyPropertyChanged
        {
        }

        /// <summary>
        /// アイテムの追加と変更通知のタイミングを分離したオブザーバブルな <see cref="Collection{T}"/> です。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        internal sealed class ExplicitNotifyObservableCollection<T> : Collection<T>, IPlotObservableList<T>
        {
            public event NotifyCollectionChangedEventHandler CollectionChanged;
            public event PropertyChangedEventHandler PropertyChanged;

            /// <summary>
            /// アイテムを追加します。
            /// 変更通知イベントの発行を notify に追加します。
            /// </summary>
            /// <param name="value"></param>
            /// <param name="notify"></param>
            public void Add(T value, ICollection<Action> notify)
            {
                var index = this.Count;
                Add(value);

                if (this.CollectionChanged != null)
                {
                    notify.Add(() => this.CollectionChanged?.Invoke(this, new NotifyCollectionChangedEventArgs(NotifyCollectionChangedAction.Add, value, index)));
                }

                if (this.PropertyChanged != null)
                {
                    notify.Add(() => this.PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(nameof(Count))));
                }
            }
        }

        /// <summary>
        /// <see cref="PlotSpyModel"/> が持つアイテムのコレクションです。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        internal sealed class ItemCollection<T>
        {
            /// <summary>
            /// key からアイテムへの辞書です。一意です。
            /// </summary>
            private readonly Dictionary<string, T> _keyDictionary =
                new Dictionary<string, T>();

            /// <summary>
            /// id からアイテムへの辞書です。一意ではありません。
            /// </summary>
            private readonly Dictionary<ulong, T> _idDictionary =
                new Dictionary<ulong, T>();

            private readonly ExplicitNotifyObservableCollection<T> _items =
                new ExplicitNotifyObservableCollection<T>();

            public IPlotObservableList<T> Items => _items;

            /// <summary>
            /// 新しいアイテムを登録します。
            /// 必要に応じてイベント発行を notify に登録します。
            /// </summary>
            /// <param name="key"></param>
            /// <param name="id"></param>
            /// <param name="item"></param>
            /// <param name="notify"></param>
            public void Add(string key, ulong id, T item, ICollection<Action> notify)
            {
                _keyDictionary.Add(key, item);
                _idDictionary[id] = item;
                _items.Add(item, notify);
            }

            /// <summary>
            /// 同じ key がすでに登録されているときに id だけ登録します。
            /// 同じ id の登録がすでにあるときは上書きします。
            /// </summary>
            /// <param name="id"></param>
            /// <param name="item"></param>
            public void AddConflicted(ulong id, T item)
            {
                _idDictionary[id] = item;
            }

            /// <summary>
            /// 指定の key のアイテムを得ます。
            /// </summary>
            /// <param name="key"></param>
            /// <returns></returns>
            public T GetItem(string key)
            {
                T item;
                _keyDictionary.TryGetValue(key, out item);
                return item;
            }

            /// <summary>
            /// 指定の id のアイテムを得ます。
            /// </summary>
            /// <param name="id"></param>
            /// <returns></returns>
            public T GetItem(ulong id)
            {
                T item;
                _idDictionary.TryGetValue(id, out item);
                return item;
            }

            /// <summary>
            /// id の登録のみ抹消します。
            /// </summary>
            /// <param name="id"></param>
            public void Remove(ulong id)
            {
                _idDictionary.Remove(id);
            }
        }

        /// <summary>
        /// <see cref="PlotNode"/> が持つ子アイテムのコレクションです。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        internal sealed class ChildCollection<T>
        {
            /// <summary>
            /// key からアイテムへの辞書です。一意です。
            /// </summary>
            private readonly Dictionary<string, T> _keyDictionary =
                new Dictionary<string, T>();

            private readonly ExplicitNotifyObservableCollection<T> _items =
                new ExplicitNotifyObservableCollection<T>();

            public IPlotObservableList<T> Items => _items;

            /// <summary>
            /// 新しいアイテムを登録します。
            /// 必要に応じてイベント発行を notify に登録します。
            /// </summary>
            /// <param name="key"></param>
            /// <param name="item"></param>
            /// <param name="notify"></param>
            public void Add(string key, T item, ICollection<Action> notify)
            {
                _keyDictionary.Add(key, item);
                _items.Add(item, notify);
            }

            /// <summary>
            /// 指定の key のアイテムを得ます。
            /// </summary>
            /// <param name="key"></param>
            /// <returns></returns>
            public T GetItem(string key)
            {
                T item;
                _keyDictionary.TryGetValue(key, out item);
                return item;
            }

            /// <summary>
            /// 指定の key をすでに持っているか調べます。
            /// </summary>
            /// <param name="key"></param>
            /// <returns></returns>
            public bool ContainsKey(string key)
            {
                return _keyDictionary.ContainsKey(key);
            }
        }

        /// <summary>
        /// <see cref="PlotFloat"/> や <see cref="PlotState"/> の値のコレクションです。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        internal sealed class ValueCollection<T> : ObservableCollection<T>, IPlotObservableList<T>
        { }
    }
}
