﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NAudio.CoreAudioApi;
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Settings;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;

namespace NintendoWare.Spy
{
    /// <summary>
    /// Spy の再生機能を提供します。
    /// </summary>
    public class SpyPlaybackService : ObservableObject
    {
        /// <summary>
        /// 再生デバイスが見つからなかった場合に発行される例外です。
        /// </summary>
        public class PlaybackDeviceNotFoundException : Exception { }

        /// <summary>
        /// 再生デバイスが Unplugged 状態だった場合に発行される例外です。
        /// </summary>
        public class PlaybackDeviceNotPluggedException : Exception { }

        /// <summary>
        /// プレイバックが予期せず中断されたことを通知するイベントの引数です。
        /// </summary>
        public class PlaybackAbortedEventArgs : EventArgs
        {
            public Exception Exception { get; }

            public PlaybackAbortedEventArgs(Exception ex)
            {
                this.Exception = ex;
            }
        }

        private const int DefaultLatency = 25;

        private readonly WeakReference<FrameSyncSpyModel> _frameSyncModel = new WeakReference<FrameSyncSpyModel>(null);
        private readonly WeakReference<FinalOutSpyModel> _finalOut = new WeakReference<FinalOutSpyModel>(null);
        private readonly WeakReference<WaveformSpyModel> _waveformModel = new WeakReference<WaveformSpyModel>(null);
        private readonly FinalOutWaveSource _finalOutWaveSource = new FinalOutWaveSource();
        private readonly WaveformWaveSource _waveformWaveSource = new WaveformWaveSource();
        private IWaveSource _currentWaveSource;
        private readonly Lazy<WasapiWavePlayer> _wavePlayer = new Lazy<WasapiWavePlayer>();

        private SpyService _spyService;
        private SettingsService _settingsService;

        private readonly object _observerOwner = new object();
        private readonly object _frameSyncObserverOwner = new object();
        private readonly object _finalOutObserverOwner = new object();
        private readonly object _waveformObserverOwner = new object();
        private readonly object _playerObserverOwner = new object();

        private WasapiWaveProvider _waveProvider = null;

        private PlaybackState _state = PlaybackState.Stopped;

        private SpyTime _begin = SpyTime.InvalidValue;
        private SpyTime _end = SpyTime.InvalidValue;
        private SpyTime _current = SpyTime.InvalidValue;

        private SpyTime _firstDataTime = SpyTime.InvalidValue;
        private SpyTime _latestDataTime = SpyTime.InvalidValue;

        private bool _shouldAutoScroll = false;
        private bool _isAutoScrollEnabled = true;
        private bool _isRangeUpdating = false;
        private bool _isCurrentUpdating = false;

        private string _currentTargetName = null;

        /// <summary>
        /// カレント時間の設定の権限を取得できなかったときのトークンの値です。
        /// </summary>
        public static readonly int TokenGrabInvalid = 0;

        /// <summary>
        /// カレント時間を連続的に変更する場合に、他からのカレント時間の設定を排除します。
        /// (スクロールバーをドラッグするような場合)
        /// </summary>
        private bool _isGrabbingCurrent = false;

        /// <summary>
        /// カレント時間の設定の権限を取得(グラブ)した人に与えられるトークンです。
        /// </summary>
        private int _tokenGrabCurrent = TokenGrabInvalid;

        //-----------------------------------------------------------------

        public enum PlaybackState
        {
            Stopped,
            Playing,
            Paused,
        }

        //-----------------------------------------------------------------

        public SpyPlaybackService()
        {
            // FinalOutWaveSource.CurrentAudioFrame と this.Current の同期
            PropertyChangedObservation.GetObserver(_observerOwner, _finalOutWaveSource)
                .AddHandler(
                    target => target.CurrentAudioFrame,
                    (target, args) => this.BeginUpdateCurrent());

            // WaveformWaveSource.CurrentTime と this.Current の同期
            PropertyChangedObservation.GetObserver(_observerOwner, _waveformWaveSource)
                .AddHandler(
                    target => target.CurrentTime,
                    (target, args) => this.BeginUpdateCurrent());
        }

        //-----------------------------------------------------------------

        public SpyTime Begin
        {
            get { return _begin; }

            private set
            {
                if (this.SetPropertyValue(ref _begin, value))
                {
                    _finalOutWaveSource.AudioFrameBegin = value.AudioFrame;
                    _waveformWaveSource.BeginTime = value.Timestamp;
                }
            }
        }

        public SpyTime End
        {
            get { return _end; }

            private set
            {
                if (this.SetPropertyValue(ref _end, value))
                {
                    _finalOutWaveSource.AudioFrameEnd = value.AudioFrame;
                    _waveformWaveSource.EndTime = value.Timestamp;
                }
            }
        }

        public SpyTime Current
        {
            get { return _current; }
        }

        public bool ShouldAutoScroll
        {
            get { return _shouldAutoScroll; }
            private set { this.SetPropertyValue(ref _shouldAutoScroll, value); }
        }

        public bool IsAutoScrollEnabled
        {
            get { return _isAutoScrollEnabled; }

            set
            {
                if (this.SetPropertyValue(ref _isAutoScrollEnabled, value))
                {
                    this.UpdateShouldAutoScroll();
                }
            }
        }

        public PlaybackState State
        {
            get { return _state; }
            private set { this.SetPropertyValue(ref _state, value); }
        }

        public FinalOutSpyModel FinalOutModel
        {
            get { return _finalOut.GetTarget(); }
            private set
            {
                var oldValue = this.FinalOutModel;
                if (oldValue == value)
                {
                    return;
                }

                this.FinalizeWavePlayer();

                if (oldValue != null)
                {
                    PropertyChangedObservation.RemoveObservers(_finalOutObserverOwner);
                }

                _finalOut.SetTarget(value);

                if (value != null)
                {
                    PropertyChangedObservation.GetObserver(_finalOutObserverOwner, value)
                        .AddHandler(
                            target => target.AudioFrameBegin,
                            (target, args) => this.BeginUpdateRange());

                    PropertyChangedObservation.GetObserver(_finalOutObserverOwner, value)
                        .AddHandler(
                            target => target.AudioFrameEnd,
                            (target, args) => this.BeginUpdateRange());
                }

                this.UpdateRange();
                this.UpdateWaveSources();
            }
        }

        public WaveformSpyModel WaveformModel
        {
            get { return _waveformModel.GetTarget(); }
            private set
            {
                var oldValue = this.WaveformModel;
                if (oldValue == value)
                {
                    return;
                }

                this.FinalizeWavePlayer();

                if (oldValue != null)
                {
                    PropertyChangedObservation.RemoveObservers(_waveformObserverOwner);
                }

                _waveformModel.SetTarget(value);

                if (value != null)
                {
                    PropertyChangedObservation.GetObserver(_waveformObserverOwner, value)
                        .AddHandler(
                            target => target.WaveformCount,
                            (target, args) => this.BeginUpdateRange());
                }

                this.UpdateRange();
                this.UpdateWaveSources();
            }
        }

        public string CurrentTargetName
        {
            get { return _currentTargetName; }
            set
            {
                if (this.SetPropertyValue(ref _currentTargetName, value))
                {
                    if (_waveProvider != null)
                    {
                        _waveProvider.TargetName = value;
                    }
                }
            }
        }

        public IEnumerable<WaveSourceOutputInfo> WaveSourceOutputInfos
        {
            get
            {
                if (this.FinalOutModel != null)
                {
                    return _finalOutWaveSource.EnumerateWaveSourceOutputInfos();
                }
                else if (this.WaveformModel != null)
                {
                    return _waveformWaveSource.EnumerateWaveSourceOutputInfos();
                }

                return new WaveSourceOutputInfo[0];
            }
        }

        private FrameSyncSpyModel FrameSyncModel
        {
            get { return _frameSyncModel.GetTarget(); }
            set
            {
                if (this.FrameSyncModel == value)
                {
                    return;
                }

                var oldValue = this.FrameSyncModel;

                if (oldValue != null)
                {
                    PropertyChangedObservation.RemoveObservers(_frameSyncObserverOwner);
                }

                _frameSyncModel.SetTarget(value);

                if (value != null)
                {
                    PropertyChangedObservation.GetObserver(_frameSyncObserverOwner, value)
                        .AddHandler(
                            target => target.MinimumFrame,
                            (target, args) => this.BeginUpdateRange());

                    PropertyChangedObservation.GetObserver(_frameSyncObserverOwner, value)
                        .AddHandler(
                            target => target.MaximumFrame,
                            (target, args) => this.BeginUpdateRange());
                }

                this.UpdateRange();
            }
        }

        /// <summary>
        /// 再生が予期せず中断した場合に通知します。
        /// </summary>
        public event EventHandler<PlaybackAbortedEventArgs> PlaybackAborted;

        private SynchronizationContext FrameworkSyncContext { get; set; }

        //-----------------------------------------------------------------

        public void Initialize(SpyService spyService, SettingsService settingsService)
        {
            Ensure.Argument.NotNull(spyService);
            Ensure.Argument.NotNull(settingsService);

            try
            {
                _spyService = spyService;
                _settingsService = settingsService;

                _spyService.EndLoadFiles += OnEndLoadFiles;

                CollectionChangedObservation.GetObserver(_observerOwner, _spyService.GetCurrentData().Models)
                    .AddHandler(
                        (sender, e) =>
                        {
                            this.FrameSyncModel = _spyService.GetCurrentData().Models.Values.OfType<FrameSyncSpyModel>().FirstOrDefault();
                            if (!_spyService.IsLoading)
                            {
                                this.UpdateModels();
                            }
                        });

                PropertyChangedObservation.GetObserver(_observerOwner, _spyService)
                    .AddHandler(
                        target => target.State,
                        (target, args) => this.UpdateShouldAutoScroll());

                PropertyChangedObservation.GetObserver(_observerOwner, _spyService)
                    .AddHandler(
                        target => target.LatestDataTimestamp,
                        (target, args) => this.UpdateLatestDataTimestamp());

                this.FrameworkSyncContext = SynchronizationContext.Current;
            }
            catch
            {
                this.Uninitialize();
                throw;
            }
        }

        public void Uninitialize()
        {
            this.FinalizeWavePlayer();

            PropertyChangedObservation.RemoveObservers(_observerOwner);
            CollectionChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_frameSyncObserverOwner);
            PropertyChangedObservation.RemoveObservers(_finalOutObserverOwner);
            PropertyChangedObservation.RemoveObservers(_waveformObserverOwner);

            _finalOutWaveSource.Dispose();
            _waveformWaveSource.Dispose();

            if (_spyService != null)
            {
                _spyService.EndLoadFiles -= OnEndLoadFiles;
                _spyService = null;
            }

            _settingsService = null;
            this.FrameworkSyncContext = null;
            this.FinalOutModel = null;
            this.WaveformModel = null;
            this.FrameSyncModel = null;
        }

        private void OnEndLoadFiles(object sender, EventArgs e)
        {
            this.UpdateModels();
        }

        private void UpdateModels()
        {
            this.FinalOutModel = _spyService.GetCurrentData().Models.Values.OfType<FinalOutSpyModel>().FirstOrDefault();
            this.WaveformModel = _spyService.GetCurrentData().Models.Values.OfType<WaveformSpyModel>().FirstOrDefault();
        }

        public void ResetDevice()
        {
            this.FinalizeWavePlayer();
        }

        /// <summary>
        /// カレント時間の設定権を取得します。
        /// この関数で得たトークンを使って SetCurrentTime() を呼び出します。
        /// 設定権を得られなかった場合は、カレント時間への設定は無視されます。
        /// </summary>
        /// <returns>設定権を表すトークンを返します。
        /// 設定権を得られなかったときはTokenGrabInvalidを返します。</returns>
        public int GetGrabForCurrent()
        {
            if (_isGrabbingCurrent)
            {
                // すでに他者がグラブしています。
                return TokenGrabInvalid;
            }
            else
            {
                // グラブします。新しく生成したトークンを返します。
                _isGrabbingCurrent = true;
                this.UpdateShouldAutoScroll();
                return ++_tokenGrabCurrent;
            }
        }

        /// <summary>
        /// カレント時間の設定権を解放します。
        /// </summary>
        /// <param name="tokenGrab"></param>
        public void ReleaseGrabForCurrent(int tokenGrab)
        {
            if (_isGrabbingCurrent && _tokenGrabCurrent == tokenGrab)
            {
                _isGrabbingCurrent = false;
                this.UpdateShouldAutoScroll();
            }
        }

        /// <summary>
        /// カレント時間の設定が可能か調べます。
        /// </summary>
        /// <param name="tokenGrab"></param>
        /// <returns></returns>
        protected bool IsWritableToCurrent(int tokenGrab)
        {
            return !_isGrabbingCurrent || _tokenGrabCurrent == tokenGrab;
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得(グラブ)している人がいないときだけ設定ができます。
        /// 時間の表示単位に応じて time のプロパティが選択されます。
        /// </summary>
        /// <param name="time"></param>
        /// <param name="timeUnit"></param>
        public void SetCurrentTime(SpyTime time, SpyTimeUnit timeUnit)
        {
            this.SetCurrentTime(time, timeUnit, TokenGrabInvalid);
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得しているか、ほかに取得している人がいないときに設定できます。
        /// 時間の表示単位に応じて time のプロパティが選択されます。
        /// </summary>
        /// <param name="time"></param>
        /// <param name="timeUnit"></param>
        /// <param name="tokenGrab"></param>
        public void SetCurrentTime(SpyTime time, SpyTimeUnit timeUnit, int tokenGrab)
        {
            switch (timeUnit)
            {
                case SpyTimeUnit.AppFrame:
                    this.SetCurrentAppFrame(time.AppFrame, tokenGrab);
                    break;

                case SpyTimeUnit.AudioFrame:
                    this.SetCurrentAudioFrame(time.AudioFrame, tokenGrab);
                    break;

                case SpyTimeUnit.Timestamp:
                    this.SetCurrentTime(time.Timestamp, tokenGrab);
                    break;
            }
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得(グラブ)している人がいないときだけ設定ができます。
        /// 時間の表示単位に応じて frameValue の値が解釈されます。
        /// </summary>
        /// <param name="frameValue"></param>
        /// <param name="timeUnit"></param>
        public void SetCurrentTime(long frameValue, SpyTimeUnit timeUnit)
        {
            this.SetCurrentTime(frameValue, timeUnit, TokenGrabInvalid);
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得しているか、ほかに取得している人がいないときに設定できます。
        /// 時間の表示単位に応じて frameValue の値が解釈されます。
        /// </summary>
        /// <param name="frameValue"></param>
        /// <param name="timeUnit"></param>
        /// <param name="tokenGrab"></param>
        public void SetCurrentTime(long frameValue, SpyTimeUnit timeUnit, int tokenGrab)
        {
            switch (timeUnit)
            {
                case SpyTimeUnit.AppFrame:
                    this.SetCurrentAppFrame(new Frame(frameValue), tokenGrab);
                    break;

                case SpyTimeUnit.AudioFrame:
                    this.SetCurrentAudioFrame(new Frame(frameValue), tokenGrab);
                    break;

                case SpyTimeUnit.Timestamp:
                case SpyTimeUnit.TimestampUsec:
                    this.SetCurrentTime(SpyGlobalTime.FromMicroSeconds(frameValue), tokenGrab);
                    break;
            }
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得(グラブ)している人がいないときだけ設定ができます。
        /// </summary>
        /// <param name="timestamp"></param>
        public void SetCurrentTime(SpyGlobalTime timestamp)
        {
            this.SetCurrentTime(timestamp, TokenGrabInvalid);
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得しているか、ほかに取得している人がいないときに設定できます。
        /// </summary>
        /// <param name="timestamp"></param>
        /// <param name="tokenGrab"></param>
        /// <seealso cref="GetGrabForCurrent"/>
        public void SetCurrentTime(SpyGlobalTime timestamp, int tokenGrab)
        {
            if (!IsWritableToCurrent(tokenGrab))
            {
                return;
            }

            if (this.Current.Timestamp == timestamp)
            {
                return;
            }

            var frameSyncModel = this.FrameSyncModel;
            SpyTime frame;
            if (frameSyncModel != null && frameSyncModel.AllFrames.TryFindValueOf(timestamp, out frame))
            {
                this.SetCurrentImpl(new SpyTime(timestamp, frame), true);
            }
            else
            {
                this.SetCurrentImpl(new SpyTime(timestamp, Frame.InvalidValue, Frame.InvalidValue), true);
            }
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得(グラブ)している人がいないときだけ設定ができます。
        /// </summary>
        /// <param name="audioFrame"></param>
        public void SetCurrentAudioFrame(Frame audioFrame)
        {
            this.SetCurrentAudioFrame(audioFrame, TokenGrabInvalid);
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得しているか、ほかに取得している人がいないときに設定できます。
        /// </summary>
        /// <param name="audioFrame"></param>
        /// <param name="tokenGrab"></param>
        public void SetCurrentAudioFrame(Frame audioFrame, int tokenGrab)
        {
            if (!IsWritableToCurrent(tokenGrab))
            {
                return;
            }

            if (this.Current.AudioFrame == audioFrame)
            {
                return;
            }

            var frameSyncModel = this.FrameSyncModel;
            if (frameSyncModel == null)
            {
                this.SetCurrentImpl(new SpyTime(SpyGlobalTime.InvalidValue, Frame.InvalidValue, audioFrame), true);
                return;
            }

            SpyTime frame;
            if (frameSyncModel.AudioFrames.TryFindValueOf(audioFrame, out frame))
            {
                this.SetCurrentImpl(frame, true);
            }
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得(グラブ)している人がいないときだけ設定ができます。
        /// </summary>
        /// <param name="appFrame"></param>
        public void SetCurrentAppFrame(Frame appFrame)
        {
            this.SetCurrentAudioFrame(appFrame, TokenGrabInvalid);
        }

        /// <summary>
        /// カレント時間を設定します。
        /// カレント時間の設定権を取得しているか、ほかに取得している人がいないときに設定できます。
        /// </summary>
        /// <param name="appFrame"></param>
        /// <param name="tokenGrab"></param>
        public void SetCurrentAppFrame(Frame appFrame, int tokenGrab)
        {
            if (!IsWritableToCurrent(tokenGrab))
            {
                return;
            }

            if (this.Current.AppFrame == appFrame)
            {
                return;
            }

            var frameSyncModel = this.FrameSyncModel;
            if (frameSyncModel == null)
            {
                this.SetCurrentImpl(new SpyTime(SpyGlobalTime.InvalidValue, appFrame, Frame.InvalidValue), true);
                return;
            }

            SpyTime frame;
            if (frameSyncModel.AppFrames.TryFindValueOf(appFrame, out frame))
            {
                this.SetCurrentImpl(frame, true);
            }
        }

        /// <summary>
        /// カレントフレームが末尾にあったら先頭に巻き戻します。
        /// </summary>
        public void RewindCurrentIfNecessary()
        {
            var endTime = this.End;

            // AppFrame 情報があるときはカレントフレームは最終フレームの手前で
            // 止まるようになっているので考慮します。
            var frameSyncModel = this.FrameSyncModel;
            int appFrameCount = frameSyncModel?.AppFrames.Count ?? 0;
            if (appFrameCount > 2)
            {
                endTime = frameSyncModel.AppFrames[appFrameCount - 2];
            }

            if (this.Current.Timestamp >= endTime.Timestamp)
            {
                this.SetCurrentImpl(this.Begin, true);
            }
        }

        /// <summary>
        /// 再生を開始・再開します。
        /// </summary>
        /// <exception cref="PlaybackDeviceNotFoundException"></exception>
        /// <exception cref="PlaybackDeviceNotPluggedException"></exception>
        public void Play()
        {
            try
            {
                if (_wavePlayer.Value.State == WasapiWavePlayer.PlaybackState.NotInitialized)
                {
                    InitializeWavePlayer();
                }

                if (!_wavePlayer.Value.IsAvailable)
                {
                    // 一度使われたデバイスが使えなくなったのは Unplugged になったと推定します。
                    this.FinalizeWavePlayer();
                    throw new PlaybackDeviceNotPluggedException();
                }

                this.State = PlaybackState.Playing;

                _wavePlayer.Value.Play(this.FrameworkSyncContext);
            }
            catch
            {
                this.State = PlaybackState.Stopped;
                throw;
            }
        }

        public void Stop()
        {
            _wavePlayer.Value.Stop();
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                this.Uninitialize();
            }

            base.Dispose(disposing);
        }

        private IWaveSource InitializeWaveSource()
        {
            var finalOutModel = this.FinalOutModel;
            if (finalOutModel != null)
            {
                _finalOutWaveSource.FinalOut = this.FinalOutModel;
                return _finalOutWaveSource;
            }

            var waveformModel = this.WaveformModel;
            if (waveformModel != null)
            {
                _waveformWaveSource.WaveformSpyModel = waveformModel;
                return _waveformWaveSource;
            }

            return null;
        }

        private void InitializeWavePlayer()
        {
            IWaveSource waveSource = this.InitializeWaveSource();
            Ensure.Operation.NotNull(waveSource);

            var enumerator = new MMDeviceEnumerator();
            var endPoints = enumerator.EnumerateAudioEndPoints(DataFlow.Render, DeviceState.Active | DeviceState.Unplugged);

            var appSettings = _settingsService.GetApplicationSettings();

            MMDevice device = null;

            if (string.IsNullOrEmpty(appSettings.PlaybackDeviceName))
            {
                device = endPoints.FirstOrDefault();
            }
            else
            {
                device = endPoints.FirstOrDefault(endPoint => endPoint.FriendlyName == appSettings.PlaybackDeviceName);
            }

            if (device == null)
            {
                throw new PlaybackDeviceNotFoundException();
            }

            if (device.State.HasFlag(DeviceState.Unplugged))
            {
                throw new PlaybackDeviceNotPluggedException();
            }

            _waveProvider = new WasapiWaveProvider(waveSource)
            {
                TargetName = _currentTargetName,
            };
            _wavePlayer.Value.Initialize(_waveProvider, device, AudioClientShareMode.Shared, DefaultLatency);

            PropertyChangedObservation.RemoveObservers(_playerObserverOwner);

            PropertyChangedObservation.GetObserver(_playerObserverOwner, _wavePlayer.Value)
                .AddHandler(
                    target => target.State,
                    (target, args) =>
                    {
                        switch (target.State)
                        {
                            case WasapiWavePlayer.PlaybackState.Playing:
                                this.State = PlaybackState.Playing;
                                break;

                            case WasapiWavePlayer.PlaybackState.Paused:
                                this.State = PlaybackState.Paused;
                                break;

                            default:
                                this.State = PlaybackState.Stopped;
                                break;
                        }
                    });

            _wavePlayer.Value.ErrorOccurred += this.OnWavePlayerErrorOccurred;

            _currentWaveSource = waveSource;
            this.SetCurrentToWaveSource();
        }

        private void FinalizeWavePlayer()
        {
            if (_wavePlayer == null)
            {
                return;
            }

            this.Stop();

            _wavePlayer.Value.ErrorOccurred -= this.OnWavePlayerErrorOccurred;
            _wavePlayer.Value.Uninitialize();
            _waveProvider = null;

            _currentWaveSource = null;

            PropertyChangedObservation.RemoveObservers(_playerObserverOwner);
        }

        private void OnWavePlayerErrorOccurred(object sender, WasapiWavePlayer.ExceptionEventArgs args)
        {
            if (args.Exception != null)
            {
                if (_wavePlayer != null)
                {
                    this.FinalizeWavePlayer();
                }

                this.PlaybackAborted?.Invoke(this, new PlaybackAbortedEventArgs(args.Exception));
            }
        }

        private void SetCurrentImpl(SpyTime value, bool isUpdatePlaybackCurrent)
        {
            Assertion.Argument.NotNull(value);

            if (_current == value)
            {
                return;
            }

            _current = value;

            if (this.State != PlaybackState.Playing)
            {
                // Current が変更されると、バインディングによってさらに変更されることがあります。
                // これがプレイバック中に起きると再生位置がおかしくなるので Current の変更を
                // WaveSource に反映しないようにします。

                if (isUpdatePlaybackCurrent)
                {
                    this.SetCurrentToWaveSource();
                }
            }

            this.NotifyPropertyChanged(() => this.Current);
        }

        private void SetCurrentToWaveSource()
        {
            if (_currentWaveSource == _finalOutWaveSource)
            {
                _finalOutWaveSource.Seek(this.Current.AudioFrame);
            }
            else if (_currentWaveSource == _waveformWaveSource)
            {
                _waveformWaveSource.CurrentTime = this.Current.Timestamp;
            }
        }

        private void BeginUpdateRange()
        {
            if (_isRangeUpdating)
            {
                return;
            }

            _isRangeUpdating = true;

            try
            {
                // 頻繁に更新すると負荷が上がりすぎるので、更新頻度を調整する
                ThreadPool.UnsafeQueueUserWorkItem(
                    state =>
                    {
                        Thread.Sleep(_settingsService.GetApplicationSettings().UpdateInterval);
                        this.FrameworkSyncContext.Post(postState => this.UpdateRange(), null);
                    },
                    null);
            }
            catch
            {
                _isRangeUpdating = false;
            }
        }

        private void UpdateRange()
        {
            _isRangeUpdating = false;
            var frameSyncModel = this.FrameSyncModel;
            var finalOutModel = this.FinalOutModel;
            var waveformModel = this.WaveformModel;

            if (frameSyncModel != null && frameSyncModel.MinimumFrame != null)
            {
                this.Begin = frameSyncModel.MinimumFrame;
                this.End = frameSyncModel.MaximumFrame ?? frameSyncModel.MinimumFrame;
            }
            else if (_spyService != null && _spyService.LatestDataTimestamp.IsValid)
            {
                this.Begin = _firstDataTime;
                this.End = _latestDataTime;
            }
            else if (finalOutModel != null && finalOutModel.AudioFrameBegin != Frame.InvalidValue)
            {
                this.Begin = new SpyTime(SpyGlobalTime.InvalidValue, Frame.InvalidValue, finalOutModel.AudioFrameBegin);
                this.End = new SpyTime(SpyGlobalTime.InvalidValue, Frame.InvalidValue, finalOutModel.AudioFrameEnd);
            }
            else if (waveformModel != null && waveformModel.BeginTime != SpyTime.InvalidValue)
            {
                this.Begin = waveformModel.BeginTime;
                this.End = waveformModel.EndTime;
            }
            else
            {
                this.Begin = SpyTime.InvalidValue;
                this.End = SpyTime.InvalidValue;
            }

            if (_isAutoScrollEnabled && !_isGrabbingCurrent)
            {
                // Current を this.End（最新フレーム）にすると、情報を送信済みのモデル、未送信のモデルにわれてしまうので、
                // 最新より1つ前のフレームに設定する
                if (frameSyncModel != null && frameSyncModel.AppFrames.Count > 2)
                {
                    this.SetCurrentImpl(frameSyncModel.AppFrames[frameSyncModel.AppFrames.Count - 2], true);
                }
                else if (_latestDataTime.Timestamp.IsValid)
                {
                    this.SetCurrentImpl(_latestDataTime, true);
                }
                else
                {
                    this.SetCurrentImpl(this.Begin, true);
                }
            }
        }

        private void BeginUpdateCurrent()
        {
            if (_isCurrentUpdating)
            {
                return;
            }

            _isCurrentUpdating = true;

            try
            {
                // 頻繁に更新すると負荷が上がりすぎるので、更新頻度を調整する
                ThreadPool.UnsafeQueueUserWorkItem(
                    state =>
                    {
                        Thread.Sleep(_settingsService.GetApplicationSettings().UpdateInterval);
                        this.FrameworkSyncContext.Post(postState => this.UpdateCurrent(), null);
                    },
                    null);
            }
            catch
            {
                _isCurrentUpdating = false;
            }
        }

        private void UpdateCurrent()
        {
            _isCurrentUpdating = false;
            var frameSyncModel = this.FrameSyncModel;
            SpyTime newCurrent = null;

            if (_currentWaveSource == _finalOutWaveSource)
            {
                var audioFrame = _finalOutWaveSource.CurrentAudioFrame;
                if (this.Current.AudioFrame == audioFrame)
                {
                    return;
                }

                if (frameSyncModel == null)
                {
                    newCurrent = new SpyTime(SpyGlobalTime.InvalidValue, Frame.InvalidValue, _finalOutWaveSource.CurrentAudioFrame);
                }
                else
                {
                    SpyTime frame;
                    if (frameSyncModel.AudioFrames.TryFindValueOf(audioFrame, out frame))
                    {
                        newCurrent = frame;
                    }
                }
            }
            else if (_currentWaveSource == _waveformWaveSource)
            {
                var currentTime = _waveformWaveSource.CurrentTime;
                if (this.Current.Timestamp == currentTime)
                {
                    return;
                }

                if (frameSyncModel == null)
                {
                    newCurrent = new SpyTime(currentTime, Frame.InvalidValue, Frame.InvalidValue);
                }
                else
                {
                    SpyTime frame;
                    if (frameSyncModel.AllFrames.TryFindValueOf(currentTime, out frame))
                    {
                        newCurrent = frame;
                    }
                }
            }

            if (newCurrent != null && !_isGrabbingCurrent)
            {
                this.SetCurrentImpl(newCurrent, false);
            }
        }

        private void UpdateShouldAutoScroll()
        {
            this.ShouldAutoScroll = this.IsAutoScrollEnabled && !_isGrabbingCurrent && _spyService.State != SpyService.ServiceState.NotStarted;
        }

        private void UpdateLatestDataTimestamp()
        {
            var timestamp = _spyService.LatestDataTimestamp;
            if (timestamp.IsValid)
            {
                var time = new SpyTime(timestamp, Frame.InvalidValue, Frame.InvalidValue);
                if (_firstDataTime.Timestamp.IsValid)
                {
                    _latestDataTime = time;
                }
                else
                {
                    _firstDataTime = time;
                    _latestDataTime = time;
                }
            }
            else
            {
                _firstDataTime = SpyTime.InvalidValue;
                _latestDataTime = SpyTime.InvalidValue;
            }

            this.BeginUpdateRange();
        }

        private void UpdateWaveSources()
        {
            NotifyPropertyChanged(nameof(WaveSourceOutputInfos));

            // CurrentTargetName が設定されていない or 無効値が残っている場合は、最初のターゲットに設定する
            if (string.IsNullOrEmpty(this.CurrentTargetName)
                || !this.WaveSourceOutputInfos.Any(info => info.Name == this.CurrentTargetName))
            {
                this.CurrentTargetName = this.WaveSourceOutputInfos.FirstOrDefault()?.Name;
            }
        }
    }
}
