﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Dynamic;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// XAMLでデザイン用のDataContextを記述するための辞書クラスです。
    /// <code>
    /// <d:UserControl.DataContext
    ///     xmlns:d="http://schemas.microsoft.com/expression/blend/2008"
    ///     xmlns:sys="clr-namespace:System;assembly=mscorlib"
    ///     xmlns:col="clr-namespace:System.Collections;assembly=mscorlib"
    ///     xmlns:spy="http://schemas.nintendo.com/nintendoware/spy"
    ///     >
    ///     <col:ArrayList>
    ///         <spy:MockupDictionary>
    ///             <sys:String x:Key="Name">Apple</sys:String>
    ///             <sys:String x:Key="Color">Red</sys:String>
    ///         </spy:MockupDictionary>
    ///         <spy:MockupDictionary>
    ///             <sys:String x:Key="Name">Banana</sys:String>
    ///             <sys:String x:Key="Color">Yellow</sys:String>
    ///             <sys:String x:Key="Taste">Good</sys:String>
    ///         </spy:MockupDictionary>
    ///     </col:ArrayList>
    /// </d:UserControl.DataContext>
    /// </code>
    /// </summary>
    public sealed class MockupDictionary : DynamicObject, IDictionary
    {
        private readonly Dictionary<string, object> _property = new Dictionary<string, object>();

        public override bool TryGetMember(GetMemberBinder binder, out object result)
        {
            return _property.TryGetValue(binder.Name, out result);
        }

        public override bool TrySetMember(SetMemberBinder binder, object value)
        {
            _property[binder.Name] = value;
            return true;
        }

        #region IDictionary メンバー

        void IDictionary.Add(object key, object value)
        {
            _property[(string)key] = value;
        }

        void IDictionary.Clear()
        {
            _property.Clear();
        }

        bool IDictionary.Contains(object key)
        {
            return _property.ContainsKey((string)key);
        }

        IDictionaryEnumerator IDictionary.GetEnumerator()
        {
            return (_property as IDictionary).GetEnumerator();
        }

        bool IDictionary.IsFixedSize
        {
            get { return (_property as IDictionary).IsFixedSize; }
        }

        bool IDictionary.IsReadOnly
        {
            get { return (_property as IDictionary).IsReadOnly; }
        }

        ICollection IDictionary.Keys
        {
            get { return (_property as IDictionary).Keys; }
        }

        void IDictionary.Remove(object key)
        {
            _property.Remove((string)key);
        }

        ICollection IDictionary.Values
        {
            get { return (_property as IDictionary).Values; }
        }

        object IDictionary.this[object key]
        {
            get
            {
                return _property[(string)key];
            }
            set
            {
                _property[(string)key] = value;
            }
        }

        #endregion

        #region ICollection メンバー

        void ICollection.CopyTo(Array array, int index)
        {
            (_property as ICollection).CopyTo(array, index);
        }

        int ICollection.Count
        {
            get { return _property.Count; }
        }

        bool ICollection.IsSynchronized
        {
            get { return (_property as ICollection).IsSynchronized; }
        }

        object ICollection.SyncRoot
        {
            get { return (_property as ICollection).SyncRoot; }
        }

        #endregion

        #region IEnumerable メンバー

        IEnumerator IEnumerable.GetEnumerator()
        {
            return (_property as IEnumerable).GetEnumerator();
        }

        #endregion
    }
}
