﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;

namespace NintendoWare.Spy.Windows
{
    [TemplatePart(Name = TemplatePartHeaderColumnDefinitionName, Type = typeof(Thumb))]
    [TemplatePart(Name = TemplatePartHorizontalThumbName, Type = typeof(Thumb))]
    public class HeaderedListBoxItem : ContentControl
    {
        private const string TemplatePartHeaderColumnDefinitionName = "HeaderColumnDefinition";
        private const string TemplatePartHorizontalThumbName = "HorizontalThumb";

        public static readonly DependencyProperty HeaderProperty = DependencyProperty.Register(
            nameof(Header),
            typeof(object),
            typeof(HeaderedListBoxItem),
            new FrameworkPropertyMetadata(
                null,
                (sender, e) => { },
                (d, value) => ((HeaderedListBoxItem)d).CoerceHeader(value)));

        public static readonly DependencyProperty HeaderTemplateProperty = DependencyProperty.Register(
            nameof(HeaderTemplate),
            typeof(DataTemplate),
            typeof(HeaderedListBoxItem),
            new FrameworkPropertyMetadata(null));

        public static readonly DependencyProperty HeaderWidthProperty = DependencyProperty.Register(
            nameof(HeaderWidth),
            typeof(double),
            typeof(HeaderedListBoxItem),
            new FrameworkPropertyMetadata(
                double.NaN,
                (d, e) => { (d as HeaderedListBoxItem).UpdateHeaderWidth(); }));

        private ColumnDefinition _headerColumnDefinition;
        private Thumb _horizontalThumb;

        static HeaderedListBoxItem()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(HeaderedListBoxItem),
                new FrameworkPropertyMetadata(typeof(HeaderedListBoxItem)));
        }

        public HeaderedListBoxItem()
        {
            // DataContext に依存する Header プロパティの再評価設定
            this.DataContextChanged += (sender, e) => this.CoerceValue(HeaderedListBoxItem.HeaderProperty);
        }

        public object Header
        {
            get { return (object)GetValue(HeaderProperty); }
            set { SetValue(HeaderProperty, value); }
        }

        public DataTemplate HeaderTemplate
        {
            get { return (DataTemplate)GetValue(HeaderTemplateProperty); }
            set { SetValue(HeaderTemplateProperty, value); }
        }

        public double HeaderWidth
        {
            get { return (double)GetValue(HeaderWidthProperty); }
            set { SetValue(HeaderWidthProperty, value); }
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            _headerColumnDefinition = this.Template.FindName(TemplatePartHeaderColumnDefinitionName, this) as ColumnDefinition;
            this.UpdateHeaderWidth();

            _horizontalThumb = this.Template.FindName(TemplatePartHorizontalThumbName, this) as Thumb;
            if (_horizontalThumb != null)
            {
                _horizontalThumb.DragDelta += (sender, e) =>
                {
                    this.Height = MathUtility.Clamp(this.ActualHeight + e.VerticalChange, this.MinHeight, this.MaxHeight);
                };
            }
        }

        private object CoerceHeader(object baseValue)
        {
            return baseValue ?? this.DataContext;
        }

        private void UpdateHeaderWidth()
        {
            if (_headerColumnDefinition != null)
            {
                _headerColumnDefinition.Width = new GridLength(this.HeaderWidth);
            }
        }
    }
}
