﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Windows.Documents;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Interactivity;
using System.Windows.Media;

namespace NintendoWare.Spy.Windows.Documents
{
    /// <summary>
    /// <see cref="ContentAdorner"/> 内で起きたイベントを、Adorner が設定されている <see cref="UIElement"/> に中継します。
    /// <para>
    /// この添付ビヘイビアは <see cref="ContentAdorner"/> のコンテントのルート要素に添付しなければなりません。
    /// </para>
    /// </summary>
    /// <example>
    /// <pre>
    /// &lt;Style
    ///     xmlns:ntf="http://schemas.nintendo.com/ntf"
    ///     xmlns:spy="http://schemas.nintendo.com/nintendoware/spy"&gt;
    ///     &lt;Setter Property="ntf:ContentAdorner.ContentTemplate"&gt;
    ///         &lt;Setter.Value&gt;
    ///             &lt;DataTemplate&gt;
    ///                 &lt;Border&gt;
    ///                     &lt;i:Interaction.Behaviors&gt;
    ///                         &lt;spy:ContentAdornerEventTunnelBehavior
    ///                             Event="ButtonBase.Click" /&gt;
    ///                     &lt;/i:Interaction.Behaviors&gt;
    ///
    ///                     &lt;Canvas&gt;
    ///                         &lt;Button Content="Click Me!" /&gt;
    ///                     &lt;/Canvas&gt;
    ///                 &lt;/Border&gt;
    ///             &lt;/DataTemplate&gt;
    ///         &lt;/Setter.Value&gt;
    ///     &lt;/Setter&gt;
    /// &lt;/Style&gt;
    /// </pre>
    /// </example>
    public class ContentAdornerEventTunnelBehavior : Behavior<UIElement>
    {
        /// <summary>
        /// 中継したいイベントを指定します。
        /// </summary>
        public RoutedEvent Event { get; set; }

        protected override void OnAttached()
        {
            base.OnAttached();

            this.AssociatedObject.AddHandler(this.Event, (RoutedEventHandler)OnEventRaised);
        }

        protected override void OnDetaching()
        {
            base.OnDetaching();

            this.AssociatedObject.RemoveHandler(this.Event, (RoutedEventHandler)OnEventRaised);
        }

        private void OnEventRaised(object sender, RoutedEventArgs e)
        {
            if (this.AssociatedObject == null)
            {
                return;
            }

            // ContentAdorner.Content 添付プロパティを使った場合。
            var parent = LogicalTreeHelper.GetParent(this.AssociatedObject);
            var contentControl = parent as ContentControl;
            if (contentControl == null)
            {
                // ContentAdorner.ContentTemplate 添付プロパティを使った場合。
                parent = VisualTreeHelper.GetParent(this.AssociatedObject);
                var contentPresenter = parent as ContentPresenter;
                if (contentPresenter == null)
                {
                    return;
                }

                parent = VisualTreeHelper.GetParent(contentPresenter);
                contentControl = parent as ContentControl;
                if (contentControl == null)
                {
                    return;
                }
            }

            parent = LogicalTreeHelper.GetParent(contentControl);
            var contentAdorner = parent as ContentAdorner;
            if (contentAdorner == null)
            {
                return;
            }

            var adornedElement = contentAdorner.AdornedElement;
            if (adornedElement == null)
            {
                return;
            }

            adornedElement.RaiseEvent(e);
        }
    }
}
