﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;

namespace NintendoWare.Spy.Foundation.Commands
{
    /// <summary>
    /// デリゲートによりコマンドハンドラを実装するためのクラスです。
    /// </summary>
    /// <typeparam name="TCommandArgs">コマンド引数の型を指定します。</typeparam>
    public sealed class DelegateCommandHandler<TCommandArgs> : CommandHandler
        where TCommandArgs : CommandArgs
    {
        private readonly Func<Command, TCommandArgs, CommandStatus> _queryStatus;
        private readonly Func<Command, TCommandArgs, CommandResult> _execute;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="targetCommand">
        /// 対象コマンドを取得します。
        /// </param>
        /// <param name="targetCommand">処理対象のコマンドを指定します。</param>
        /// <param name="queryStatus">QueryStatus メソッドを指定します。</param>
        /// <param name="execute">Execute メソッドを指定します。</param>
        public DelegateCommandHandler(
            Command targetCommand,
            Func<Command, TCommandArgs, CommandStatus> queryStatus,
            Func<Command, TCommandArgs, CommandResult> execute) :
            base(targetCommand)
        {
            Ensure.Argument.NotNull(queryStatus);
            Ensure.Argument.NotNull(execute);
            _queryStatus = queryStatus;
            _execute = execute;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの状態を返します。</returns>
        protected override CommandStatus QueryStatusImpl(Command command, CommandArgs commandArgs)
        {
            return _queryStatus(command, commandArgs as TCommandArgs);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの処理結果を返します。</returns>
        protected override CommandResult ExecuteImpl(Command command, CommandArgs commandArgs)
        {
            return _execute(command, commandArgs as TCommandArgs);
        }
    }
}
