﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;

namespace NintendoWare.Spy.Foundation.Commands
{
    /// <summary>
    /// コマンド機能を提供します。
    /// </summary>
    public sealed class CommandService : ObservableObject, ICommandTarget
    {
        private readonly CommandBindingList _commandBindings = new CommandBindingList();
        private readonly CommandBindingList _inheritableCommandBindings = new CommandBindingList();

        private readonly Dictionary<string, Func<string, object>> _commandArgBindings =
            new Dictionary<string, Func<string, object>>();

        private ICommandTarget _parentCommandTarget;
        private Func<ICommandTarget> _getInnerTarget;

        //-----------------------------------------------------------------

        private class StackItem
        {
            public StackItem(
                ICommandTarget commandTarget,
                IReadOnlyKeyedList<Command, CommandBinding> commandBindings)
            {
                this.CommandTarget = commandTarget;
                this.CommandBindings = commandBindings;
            }

            public ICommandTarget CommandTarget { get; }
            public IReadOnlyKeyedList<Command, CommandBinding> CommandBindings { get; }
        }

        //-----------------------------------------------------------------

        public event EventHandler<CommandEventArgs> CommandExecuted;

        //-----------------------------------------------------------------

        /// <summary>
        /// コマンドバインディングの一覧を取得します。
        /// </summary>
        public IKeyedList<Command, CommandBinding> CommandBindings
        {
            get { return _commandBindings; }
        }

        /// <summary>
        /// 継承可能なコマンドバインディングの一覧を取得します。
        /// </summary>
        public IKeyedList<Command, CommandBinding> InheritableCommandBindings
        {
            get { return _inheritableCommandBindings; }
        }

        /// <summary>
        /// コマンド引数バインディングの辞書を取得します。
        /// </summary>
        public IDictionary<string, Func<string, object>> CommandArgBindings
        {
            get { return _commandArgBindings; }
        }

        //-----------------------------------------------------------------

        public void Initialize(Func<ICommandTarget> getInnerTarget)
        {
            Ensure.Operation.Null(_getInnerTarget);
            _getInnerTarget = getInnerTarget;
        }

        public void Uninitialize()
        {
            _getInnerTarget = null;
        }

        /// <summary>
        /// コマンドターゲットを継承し、親のコマンドバインディングを取り扱えるようにします。
        /// </summary>
        /// <param name="parentCommandTarget">親コマンドターゲットを指定します。</param>
        public void Inherit(ICommandTarget parentCommandTarget)
        {
            Ensure.Argument.NotNull(parentCommandTarget);
            _parentCommandTarget = parentCommandTarget;
        }

        /// <summary>
        /// コマンドターゲットの継承を解除します。
        /// </summary>
        public void Uninherit()
        {
            _parentCommandTarget = null;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">コマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの状態を返します。</returns>
        public CommandStatus QueryStatus(Command command, CommandArgs commandArgs = null)
        {
            return this.QueryStatusImpl(command, commandArgs);
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの処理結果を返します。</returns>
        public CommandResult Execute(Command command, CommandArgs commandArgs = null)
        {
            return this.ExecuteImpl(command, commandArgs);
        }

        /// <summary>
        /// コマンド実行通知を発行します。
        /// </summary>
        /// <param name="e">イベントデータを指定します。</param>
        private void NotifyCommandExecuted(CommandEventArgs e)
        {
            Assertion.Argument.NotNull(e);

            if (this.CommandExecuted != null)
            {
                this.CommandExecuted(this, e);
            }
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの状態を返します。</returns>
        private CommandStatus QueryStatusImpl(Command command, CommandArgs commandArgs)
        {
            CommandArgs args = commandArgs != null ? commandArgs.DeepClone() : null;
            ICommandHandler handler = this.ResolveHandler(command, args);

            if (handler == null)
            {
                return CommandStatus.Unsupported;
            }

            return handler.QueryStatus(command, args);
        }

        /// <summary>
        /// 指定コマンドターゲットでコマンドを実行します。
        /// </summary>
        /// <param name="command">実行するコマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドの処理結果を返します。</returns>
        private CommandResult ExecuteImpl(Command command, CommandArgs commandArgs)
        {
            CommandArgs args = commandArgs != null ? commandArgs.DeepClone() : null;
            ICommandHandler handler = this.ResolveHandler(command, args);

            if (handler == null)
            {
                return new CommandResult(CommandResult.Code.CommandHandlerNotFound);
            }

            CommandResult result = null;

            try
            {
                result = handler.Execute(command, args);
            }
            catch
            {
                result = new CommandResult(CommandResult.Code.Failed);
            }

            this.NotifyCommandExecuted(
                new CommandEventArgs(this, command, args, result));

            return result;
        }

        /// <summary>
        /// コマンドハンドラを検索します。
        /// </summary>
        /// <param name="command">コマンドを指定します。</param>
        /// <param name="commandArgs">コマンド引数を指定します。</param>
        /// <returns>コマンドハンドラを返します。</returns>
        private ICommandHandler ResolveHandler(Command command, CommandArgs commandArgs)
        {
            Ensure.Argument.NotNull(command);

            // 対象となるコマンドターゲットを検索します。
            var commandTargetStack = this.CreateCommandTargetStack();

            while (commandTargetStack.Count > 0)
            {
                var stackItem = commandTargetStack.Pop();

                // 対象のコマンドハンドラを検索します。
                CommandBinding commandBinding = null;
                stackItem.CommandBindings.TryGetItem(command, out commandBinding);

                if (commandBinding != null)
                {
                    return commandBinding.CommandHandler;
                }
            }

            return null;
        }

        private Stack<StackItem> CreateCommandTargetStack()
        {
            var commandBindingsStack = new Stack<StackItem>();

            // 自身のコマンドバインディングを追加
            commandBindingsStack.Push(
                new StackItem(this, _commandBindings));

            commandBindingsStack.Push(
                new StackItem(this, _inheritableCommandBindings));

            // 祖先の継承コマンドバインディングを追加
            var parentCommandService = _parentCommandTarget as CommandService;

            while (parentCommandService != null)
            {
                commandBindingsStack.Push(
                    new StackItem(this, parentCommandService.InheritableCommandBindings));

                parentCommandService = parentCommandService._parentCommandTarget as CommandService;
            }

            return commandBindingsStack;
        }
    }
}
