﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;

namespace NintendoWare.Spy.Foundation.Commands
{
    /// <summary>
    /// コマンドのバインドをサポートします。
    /// </summary>
    public static class CommandBinder
    {
        /// <summary>
        /// 指定コマンドをバインドします。
        /// </summary>
        /// <param name="commandTarget">コマンドターゲットを指定します。</param>
        /// <param name="command">関連付けるコマンドを指定します。</param>
        /// <param name="commandHandler">関連付けるコマンドハンドラを指定します。</param>
        public static void BindCommand(this ICommandTarget commandTarget, Command command, ICommandHandler commandHandler)
        {
            Ensure.Argument.NotNull(commandTarget);
            Ensure.Argument.NotNull(command);
            Ensure.Argument.NotNull(commandHandler);

            commandTarget.CommandBindings.Add(
                new CommandBinding(command, commandHandler));
        }

        /// <summary>
        /// 指定コマンドをバインドします。
        /// </summary>
        /// <param name="commandTarget">コマンドターゲットを指定します。</param>
        /// <param name="command">関連付けるコマンドを指定します。</param>
        /// <param name="queryStatus">QueryStatus メソッドを指定します。</param>
        /// <param name="execute">Execute メソッドを指定します。</param>
        public static void BindCommand(
            this ICommandTarget commandTarget,
            Command command,
            Func<Command, CommandArgs, CommandStatus> queryStatus,
            Func<Command, CommandArgs, CommandResult> execute)
        {
            commandTarget.BindCommand<CommandArgs>(command, queryStatus, execute);
        }

        /// <summary>
        /// 指定コマンドをバインドします。
        /// </summary>
        /// <typeparam name="TCommandArgs">コマンド引数の型を指定します。</typeparam>
        /// <param name="commandTarget">コマンドターゲットを指定します。</param>
        /// <param name="command">関連付けるコマンドを指定します。</param>
        /// <param name="queryStatus">QueryStatus メソッドを指定します。</param>
        /// <param name="execute">Execute メソッドを指定します。</param>
        public static void BindCommand<TCommandArgs>(
            this ICommandTarget commandTarget,
            Command command,
            Func<Command, TCommandArgs, CommandStatus> queryStatus,
            Func<Command, TCommandArgs, CommandResult> execute)
            where TCommandArgs : CommandArgs
        {
            Ensure.Argument.NotNull(commandTarget);
            Ensure.Argument.NotNull(command);
            Ensure.Argument.NotNull(queryStatus);
            Ensure.Argument.NotNull(execute);

            commandTarget.CommandBindings.Add(
                new CommandBinding(command, new DelegateCommandHandler<TCommandArgs>(command, queryStatus, execute)));
        }

        /// <summary>
        /// 指定コマンドを継承を許可してバインドします。
        /// </summary>
        /// <param name="commandTarget">コマンドターゲットを指定します。</param>
        /// <param name="command">関連付けるコマンドを指定します。</param>
        /// <param name="commandHandler">関連付けるコマンドハンドラを指定します。</param>
        public static void BindInheritableCommand(
            this ICommandTarget commandTarget,
            Command command,
            ICommandHandler commandHandler)
        {
            Ensure.Argument.NotNull(commandTarget);
            Ensure.Argument.NotNull(command);
            Ensure.Argument.NotNull(commandHandler);

            commandTarget.InheritableCommandBindings.Add(
                new CommandBinding(command, commandHandler));
        }

        /// <summary>
        /// 指定コマンドを継承を許可してバインドします。
        /// </summary>
        /// <param name="commandTarget">コマンドターゲットを指定します。</param>
        /// <param name="command">関連付けるコマンドを指定します。</param>
        /// <param name="queryStatus">QueryStatus メソッドを指定します。</param>
        /// <param name="execute">Execute メソッドを指定します。</param>
        public static void BindInheritableCommand(
            this ICommandTarget commandTarget,
            Command command,
            Func<Command, CommandArgs, CommandStatus> queryStatus,
            Func<Command, CommandArgs, CommandResult> execute)
        {
            commandTarget.BindInheritableCommand<CommandArgs>(command, queryStatus, execute);
        }

        /// <summary>
        /// 指定コマンドを継承を許可してバインドします。
        /// </summary>
        /// <param name="commandTarget">コマンドターゲットを指定します。</param>
        /// <param name="command">関連付けるコマンドを指定します。</param>
        /// <param name="queryStatus">QueryStatus メソッドを指定します。</param>
        /// <param name="execute">Execute メソッドを指定します。</param>
        public static void BindInheritableCommand<TCommandArgs>(
            this ICommandTarget commandTarget,
            Command command,
            Func<Command, TCommandArgs, CommandStatus> queryStatus,
            Func<Command, TCommandArgs, CommandResult> execute)
            where TCommandArgs : CommandArgs
        {
            Ensure.Argument.NotNull(commandTarget);
            Ensure.Argument.NotNull(command);
            Ensure.Argument.NotNull(queryStatus);
            Ensure.Argument.NotNull(execute);

            commandTarget.InheritableCommandBindings.Add(
                new CommandBinding(command, new DelegateCommandHandler<TCommandArgs>(command, queryStatus, execute)));
        }
    }
}
