﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Reflection;
using System.Text;

namespace NintendoWare.Spy.Communication
{
    /// <summary>
    /// アプリに対応しているデータタイプの情報を要求するパケットです(Sync)。
    /// </summary>
    internal class QueryDataInfoPacket : Packet<NullPacketBody>
    {
        public QueryDataInfoPacket() :
            base(SpySession._Signature, (uint)PacketID.QueryDataInfo)
        { }
    }

    /// <summary>
    /// アプリが対応しているデータタイプの情報を返すパケットです(Sync)。
    /// </summary>
    internal class QueryDataInfoReplyPacket : Packet<QueryDataInfoReplyPacket.Content>
    {
        public QueryDataInfoReplyPacket() :
            base(SpySession._Signature, (uint)PacketID.QueryDataInfoReply)
        { }

        //-----------------------------------------------------------------

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
        public class Content : PacketBody
        {
            /// <summary>
            /// データタイプのバージョン
            /// </summary>
            private UInt32 DataVersionInternal { get; set; }

            /// <summary>
            /// データタイプのセッション中のみ有効なID番号。
            /// </summary>
            private UInt16 DataIdInternal { get; set; }

            /// <summary>
            /// dataName のバイト列の長さ。
            /// </summary>
            private UInt16 DataNameLengthInternal { get; set; }

            /// <summary>
            /// データタイプ名。ASCII文字列。
            /// </summary>
            private byte[] DataNameInternal { get; set; }

            /// <summary>
            /// DataNameInternal の読み出し時に ObjectBinaryReader にバイト列の長さを提供します。
            /// ObjectBinaryReader の規約により、プロパティ名は DataNameInternal + SizeInternal、
            /// 型は int でなければなりません。
            /// </summary>
            private int DataNameInternalSizeInternal { get { return this.DataNameLengthInternal; } }

            /// <summary>
            /// データタイプ名に続くパディングを読み飛ばします。
            /// </summary>
            private byte[] DataNamePaddingInternal { get; set; }

            /// <summary>
            /// DataNamePaddingInternal の読み出し時に ObjectBinaryReader にバイト列の長さを提供します。
            /// ObjectBinaryReader の規約により、プロパティ名は DataNamePaddingInternal + SizeInternal、
            /// 型は int でなければなりません。
            /// </summary>
            private int DataNamePaddingInternalSizeInternal { get { return (4 - (this.DataNameLengthInternal % 4)) % 4; } }

            public override uint Size
            {
                get
                {
                    return (uint)(
                        sizeof(UInt32) // DataVersionInternal
                        + sizeof(UInt16) // DataIdInternal
                        + sizeof(UInt16) // DataNameLengthInternal
                        + this.DataNameInternalSizeInternal // DataNameInternal
                        + this.DataNamePaddingInternalSizeInternal); // DataNamePaddingInternal
                }
            }

            public Version DataVersion
            {
                get
                {
                    return new Version(
                        (int)((this.DataVersionInternal >> 0) & 0xff), // major
                        (int)((this.DataVersionInternal >> 8) & 0xff), // minor
                        (int)((this.DataVersionInternal >> 16) & 0xff), // build
                        (int)((this.DataVersionInternal >> 24) & 0xff)); // revision
                }
            }

            public UInt32 DataID
            {
                get { return this.DataIdInternal; }
            }

            public string DataName
            {
                get { return (this.DataNameInternal == null) ? string.Empty : Encoding.ASCII.GetString(this.DataNameInternal); }
            }

            protected override IEnumerable<PropertyInfo> GetProperties()
            {
                yield return this.GetProperty(target => target.DataVersionInternal);
                yield return this.GetProperty(target => target.DataIdInternal);
                yield return this.GetProperty(target => target.DataNameLengthInternal);
                yield return this.GetProperty(target => target.DataNameInternal);
                yield return this.GetProperty(target => target.DataNamePaddingInternal);
            }
        }
    }
}
