﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
//#define DebugState
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.Communications;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using System.Threading;

namespace NintendoWare.Spy.Communication
{
    public class SpySession
    {
        public const string _Signature = "SSPY";

        /// <summary>
        /// ターゲットがプロトコルバージョンを送って来ない場合の代替バージョン番号。
        /// </summary>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        public static readonly Version ProtocolVersion_0_9_0_0 = new Version(0, 9, 0, 0);

        /// <summary>
        /// InitializeReply パケットにバージョン情報を追加。
        /// </summary>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        public static readonly Version ProtocolVersion_1_0_0_0 = new Version(1, 0, 0, 0);

        /// <summary>
        /// 別スレッドで <see cref="_immediateState"/> が変更されるのをチェックする間隔(ミリ秒)。
        /// </summary>
        private const int ImmediateStateCheckInterval = 100;

        //-----------------------------------------------------------------

        public enum SessionState
        {
            NotStarted,
            Starting,
            Stopping,
            Running,
        }

        public class StateChangedEventArgs : EventArgs
        {
            public StateChangedEventArgs(SessionState oldState, SessionState newState)
            {
                this.OldState = oldState;
                this.NewState = newState;
            }

            public SessionState OldState { get; private set; }
            public SessionState NewState { get; private set; }
        }

        public class DataInfo
        {
            public uint DataID { get; set; }
            public string DataName { get; set; }
            public Version DataVersion { get; set; }
        }

        //-----------------------------------------------------------------

        private readonly object _stateLock = new object();

        private readonly Lazy<SpySyncSession> _syncSession = new Lazy<SpySyncSession>();
        private readonly Lazy<SpyDataSession> _dataSession = new Lazy<SpyDataSession>();

        private SessionState _state = SessionState.NotStarted;
        private volatile SessionState _immediateState = SessionState.NotStarted;
        private SpySyncSession.SessionState _synchronizedSyncSessionState = SpySyncSession.SessionState.NotStarted;
        private SpyDataSession.SessionState _synchronizedDataSessionState = SpyDataSession.SessionState.NotStarted;

        private SynchronizationContext _syncContext;
        private int _threadID;
        private IComEndPoint _hostIO;

        private readonly Dictionary<uint, DataInfo> _dataInfos = new Dictionary<uint, DataInfo>();
        private readonly Dictionary<string, uint> _dataNameToId = new Dictionary<string, uint>();
        private HashSet<string> _requestedDataNames = new HashSet<string>();

        //-----------------------------------------------------------------

        /// <summary>
        /// 状態変化を通知します。
        /// 同期的に通知されます。
        /// </summary>
        public event EventHandler<StateChangedEventArgs> StateChanged;

        /// <summary>
        /// セッションの初期化が完了し、データ受信を開始することを通知します。
        /// 非同期に通知されます。
        /// </summary>
        public event EventHandler DataReceiveStarted;

        /// <summary>
        /// 受信したデータを通知します。
        /// 非同期に通知されます。
        /// </summary>
        public event EventHandler<SpyRawDataEventArgs> DataReceived;

        //-----------------------------------------------------------------

        public SessionState State
        {
            get { return _state; }

            private set
            {
                if (_state == value)
                {
                    return;
                }

                // Start() で指定されたスレッドで実行されることを保証する
                Assertion.Operation.True(Thread.CurrentThread.ManagedThreadId == _threadID);

                var oldState = _state;
                _state = value;

                Debug.WriteLine($"[StateChanged] {oldState} --> {value} ({DateTime.Now})");

                this.OnStateChanged(new StateChangedEventArgs(oldState, value));
            }
        }

        /// <summary>
        /// SyncSession、DataSessionの最新状態を考慮した接続状態を取得します。
        /// </summary>
        public SessionState StateWithEvaluation
        {
            get
            {
                if (_syncSession.Value.State == SpySyncSession.SessionState.Running &&
                    _dataSession.Value.State == SpyDataSession.SessionState.Running)
                {
                    return SessionState.Running;
                }
                return _state;
            }
        }

        /// <summary>
        /// リトルエンディアンなのかを取得します。
        /// </summary>
        public bool IsLittleEndian
        {
            get { return _hostIO.IsLittleEndian; }
        }

        /// <summary>
        /// アプリケーションが対応しているデータの情報です。
        /// </summary>
        public IEnumerable<DataInfo> DataInfosList
        {
            get
            {
                return _dataInfos.Values;
            }
        }

        /// <summary>
        /// アプリケーションが対応しているプロトコルバージョンです。
        /// </summary>
        public Version TargetProtocolVersion
        {
            get { return _syncSession.Value.TargetProtocolVersion; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// セッションを開始します。
        /// </summary>
        public void Start(IComEndPoint hostIO, SynchronizationContext syncContext, object syncPort, object dataPort, string workDirPath)
        {
            Ensure.Argument.NotNull(hostIO);
            Ensure.Argument.NotNull(syncContext);

            Ensure.Operation.Null(_hostIO);

            Assertion.Operation.AreEqual(this.State, SessionState.NotStarted);

            _hostIO = hostIO;
            _syncContext = syncContext;
            _threadID = Thread.CurrentThread.ManagedThreadId;
            _dataInfos.Clear();
            _dataNameToId.Clear();

            try
            {
                _syncSession.Value.StateChanged += this.UpdateState;
                _syncSession.Value.DataInfoReceived += this.UpdateDataInfo;
                _dataSession.Value.StateChanged += this.UpdateState;
                _dataSession.Value.DataReceived += this.NotifyDataReceived;
                _dataSession.Value.PushNotifyDataReadPacket = _syncSession.Value.TransferNotifyDataRead;

                _syncSession.Value.Start(hostIO, syncPort, syncContext, workDirPath);
                _dataSession.Value.Start(hostIO, dataPort, syncContext, workDirPath);

                // スレッド生成中の Start() 呼び出しを防ぐために、ここで先行して状態変更します。
                this.State = SessionState.Starting;
            }
            catch
            {
                this.StopAsync();
                throw;
            }
        }

        /// <summary>
        /// セッションを終了します。
        /// <para>
        /// セッションの終了処理が完了するまでブロックします。
        /// </para>
        /// </summary>
        public void Stop()
        {
            if (this.State == SessionState.NotStarted)
            {
                return;
            }

            this.StopAsync();

            // NOTE:
            // this.State の更新処理は SynchronizationContext.Post() でキューイングされ UI スレッドで実行されます。
            // このメソッドも UI スレッドから呼ばれるため
            // ビジーウェイトで this.State の変化を待っていると Post() でキューイングされた処理が行われず
            // デッドロックします。
            while (_immediateState != SessionState.NotStarted)
            {
                Thread.Sleep(ImmediateStateCheckInterval);
            }
        }

        /// <summary>
        /// セッションを終了します。
        /// <para>
        /// セッションの終了処理が完了すると <see cref="State"/> が非同期に <see cref="SessionState.NotStarted"/> に遷移します。
        /// </para>
        /// </summary>
        public void StopAsync()
        {
            _syncSession.Value.StopAsync();
            _dataSession.Value.StopAsync();
        }

        public void RequestSpyDatas(HashSet<string> dataNames)
        {
            if (!_requestedDataNames.SequenceEqual(dataNames))
            {
                _requestedDataNames = dataNames;
                this.SelectSpyDatas(dataNames);
            }
        }

        private void SelectSpyDatas(HashSet<string> datas)
        {
            var dataIds = datas.Where(item => _dataNameToId.ContainsKey(item)).Select(item => _dataNameToId[item]).ToArray();

            if (dataIds.Length > 0)
            {
                var max = dataIds.Max();
                var array = new uint[(max + 31) / 32];

                if (array.Length > 0)
                {
                    foreach (var item in dataIds)
                    {
                        array[item / 32] |= 1u << (int)(item % 32);
                    }
                }

                _syncSession.Value.SelectedSpyDataIDFlags = array;
            }
            else
            {
                _syncSession.Value.SelectedSpyDataIDFlags = new uint[0];
            }
        }

        private void UpdateState(object sender, EventArgs args)
        {
            lock (_stateLock)
            {
                Assertion.Operation.True(sender == _syncSession.Value || sender == _dataSession.Value, "unexpected sender");

                if (sender == _syncSession.Value)
                {
                    _synchronizedSyncSessionState = _syncSession.Value.State;
                }
                else if (sender == _dataSession.Value)
                {
                    _synchronizedDataSessionState = _dataSession.Value.State;
                }

                SessionState oldState = _immediateState;
                SessionState newState;

                // SyncSession が Running になったときの処理。
                if (sender == _syncSession.Value &&
                    _synchronizedSyncSessionState == SpySyncSession.SessionState.Running)
                {
                    // セッション初期化により DataID が確定するのをまって、データを要求します。
                    this.SelectSpyDatas(_requestedDataNames);

                    if (this.DataReceiveStarted != null)
                    {
                        this.DataReceiveStarted(this, EventArgs.Empty);
                    }

                    // DataSession のデータ読み出しを許可します。
                    _dataSession.Value.AllowRunning = true;
                }

                // どちらかの Session が停止したら全停止します。
                if ((sender == _syncSession.Value && _synchronizedSyncSessionState == SpySyncSession.SessionState.NotStarted) ||
                    (sender == _dataSession.Value && _synchronizedDataSessionState == SpyDataSession.SessionState.NotStarted))
                {
                    if (oldState == SessionState.Starting || oldState == SessionState.Running)
                    {
                        this.StopAsync();
                    }
                }

                // 状態遷移表
                //
                // |    Data     |                     Sync                                    |
                // |             |   NotStarted   |   Starting   |   Stopping   |   Running    |
                // |-------------|----------------|--------------|--------------|--------------|
                // | NoStarted   | (A) NotStarted | (C) Starting | (B) Stopping | (E) (*)      |
                // | Starting    | (C) Starting   | (C) Starting | (B) Stopping | (C) Starting |
                // | Stopping    | (B) Stopping   | (B) Stopping | (B) Stopping | (B) Stopping |
                // | WaitForSync | (C) Starting   | (C) Starting | (B) Stopping | (C) Starting |
                // | Running     | (E) (*)        | (C) Starting | (B) Stopping | (D) Running  |
                //
                // (*) Starting または Stopping

                if (_synchronizedSyncSessionState == SpySyncSession.SessionState.NotStarted &&
                    _synchronizedDataSessionState == SpyDataSession.SessionState.NotStarted)
                {
                    // (A)
                    newState = SessionState.NotStarted;
                }
                else if (_synchronizedSyncSessionState == SpySyncSession.SessionState.Stopping ||
                    _synchronizedDataSessionState == SpyDataSession.SessionState.Stopping)
                {
                    // (B)
                    newState = SessionState.Stopping;
                }
                else if (_synchronizedSyncSessionState == SpySyncSession.SessionState.Starting ||
                    _synchronizedDataSessionState == SpyDataSession.SessionState.Starting ||
                    _synchronizedDataSessionState == SpyDataSession.SessionState.WaitForSync)
                {
                    // (C)
                    newState = SessionState.Starting;
                }
                else if (_synchronizedSyncSessionState == SpySyncSession.SessionState.Running &&
                    _synchronizedDataSessionState == SpyDataSession.SessionState.Running)
                {
                    // (D)
                    newState = SessionState.Running;
                }
                else
                {
                    // (E)
                    Assertion.Operation.True(
                        (_synchronizedSyncSessionState == SpySyncSession.SessionState.Running && _synchronizedDataSessionState == SpyDataSession.SessionState.NotStarted) ||
                        (_synchronizedDataSessionState == SpyDataSession.SessionState.Running && _synchronizedSyncSessionState == SpySyncSession.SessionState.NotStarted));

                    if ((sender == _syncSession.Value && _synchronizedSyncSessionState == SpySyncSession.SessionState.Running) ||
                        (sender == _dataSession.Value && _synchronizedDataSessionState == SpyDataSession.SessionState.Running))
                    {
                        // 1. セッション開始後、一方のセッションが先行して Running になった場合。
                        //    NotStarted (NotStarted, NotStarted)
                        //    --> Starting (Starting, NotStarted)
                        //    --> Starting (Running, NotStarted)
                        //
                        // (実際には Data セッションが先行して Running になることは無い)
                        newState = SessionState.Starting;
                    }
                    else
                    {
                        // 2. Running 状態から、一方のセッションが先行して NotStarted になった場合。
                        //    Running (Running, Running)
                        //    --> Stopping (Running, Stopping)
                        //    --> Stopping (Running, NotStarted)
                        //
                        // 3. Running 状態で、一方のセッションが異常終了した場合。
                        //    Running (Running, Running)
                        //    --> Stopping (Running, NotStarted)
                        //
                        // 4. Starting 状態で、一方のセッションが異常終了した場合。
                        //    Starting (Running, Starting)
                        //    --> Stopping (Running, NotStarted)
                        newState = SessionState.Stopping;
                    }
                }

#if DebugState
                Debug.WriteLine($"[UpdateState] {oldState} --> {newState} ({DateTime.Now})");
#endif

                // 許される状態遷移
                Assertion.Operation.True(
                    (oldState == newState) ||
                    (oldState == SessionState.NotStarted && newState == SessionState.Starting) ||
                    (oldState == SessionState.Starting && newState == SessionState.Running) ||
                    (oldState == SessionState.Starting && newState == SessionState.Stopping) ||
                    (oldState == SessionState.Starting && newState == SessionState.NotStarted) ||
                    (oldState == SessionState.Running && newState == SessionState.Stopping) ||
                    (oldState == SessionState.Stopping && newState == SessionState.NotStarted));

                _immediateState = newState;

                if (_syncContext != null)
                {
                    _syncContext.Post(
                        state =>
                        {
                            if (_syncContext != null)
                            {
                                this.State = (SessionState)state;
                            }
                        },
                        newState);
                }
            }
        }

        private void UpdateDataInfo(object sender, SpySyncSession.DataInfoReceivedEventArgs e)
        {
            _dataInfos[e.DataID] = new DataInfo()
            {
                DataID = e.DataID,
                DataName = e.DataName,
                DataVersion = e.DataVersion,
            };
            _dataNameToId[e.DataName] = e.DataID;
        }

        private void OnStateChanged(StateChangedEventArgs args)
        {
            if (this.State == SessionState.NotStarted)
            {
                _syncSession.Value.StateChanged -= this.UpdateState;
                _dataSession.Value.StateChanged -= this.UpdateState;
                _dataSession.Value.DataReceived -= this.NotifyDataReceived;
                _syncSession.Value.DataInfoReceived -= this.UpdateDataInfo;

                _hostIO = null;
                _syncContext = null;
                _threadID = 0;
            }

            if (this.StateChanged != null)
            {
                this.StateChanged(this, args);
            }
        }

        private void NotifyDataReceived(object sender, SpyRawDataEventArgs args)
        {
            if (this.DataReceived != null)
            {
                DataInfo info;

                if (_dataInfos.TryGetValue(args.DataID, out info))
                {
                    args.SetDataInfo(info.DataName, info.DataVersion);
                    this.DataReceived(this, args);
                }
            }
        }
    }
}
