﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
//#define DebugState
using Nintendo.ToolFoundation.Contracts;
using Nintendo.ToolFoundation.IO;
using NintendoWare.Spy.Foundation.Communications;
using System;
using System.Diagnostics;
using System.IO;
using System.Net.Sockets;
using System.Text;
using System.Threading;

namespace NintendoWare.Spy.Communication
{
    internal class SpyDataSession
    {
        private const int ConnectionTimeout = 5000;
        private const int SendTimeout = 5000;
        private const int ReceiveTimeout = 5000;
        private const int PollingInterval = 3;

        //-----------------------------------------------------------------

        public enum SessionState
        {
            NotStarted,
            Starting,
            Stopping,
            WaitForSync,
            Running,
        }

        //-----------------------------------------------------------------

        private readonly object _hostIOLock = new object();

        private SessionState _state = SessionState.NotStarted;

        private SynchronizationContext _syncContext;
        private IComEndPoint _hostIO;
        private IComChannel _channel;

        private readonly AutoResetEvent _stateChangedEventWait = new AutoResetEvent(false);
        private volatile bool _isAvailable;
        private volatile bool _allowRunning;
        private Thread _thread;

        private BinaryReader _reader;
        private Stream _stream;
        private object _port;

        private string _workDirPath;
        private BinaryReader _dataFileReader0;
        private BinaryReader _dataFileReader1;
        private readonly ManualResetEvent _stopEvent = new ManualResetEvent(false);
        private Thread _dataListenThread;

        //-----------------------------------------------------------------

        public event EventHandler StateChanged;

        public event EventHandler<SpyRawDataEventArgs> DataReceived;

        public Action<uint> PushNotifyDataReadPacket { get; set; }

        //-----------------------------------------------------------------

        public SessionState State
        {
            get { return _state; }

            private set
            {
                if (_state == value)
                {
                    return;
                }

#if DebugState
                Debug.WriteLine($"[DataStateChanged] {_state} --> {value} ({DateTime.Now})");
#endif

                _state = value;

                if (this.StateChanged != null)
                {
                    this.StateChanged(this, EventArgs.Empty);
                }
            }
        }

        private bool IsAvailable
        {
            get
            {
                return _isAvailable;
            }

            set
            {
                lock (_stateChangedEventWait)
                {
                    if (_isAvailable != value)
                    {
                        _isAvailable = value;
                        _stateChangedEventWait.Set();
                    }
                }
            }
        }

        public bool AllowRunning
        {
            get
            {
                return _allowRunning;
            }

            set
            {
                lock (_stateChangedEventWait)
                {
                    if (_allowRunning != value)
                    {
                        _allowRunning = value;
                        _stateChangedEventWait.Set();
                    }
                }
            }
        }

        public bool IsLittleEndian
        {
            get { return _hostIO.IsLittleEndian; }
        }

        private bool IsConnected
        {
            get
            {
                lock (_hostIOLock)
                {
                    return _hostIO != null && _hostIO.IsConnected;
                }
            }
        }

        //-----------------------------------------------------------------

        public void Start(IComEndPoint hostIO, object port, SynchronizationContext syncContext, string workDirPath)
        {
            Ensure.Argument.NotNull(hostIO);
            Ensure.Argument.NotNull(syncContext);
            Ensure.Operation.NotNull(this.PushNotifyDataReadPacket);

            Assertion.Operation.AreEqual(this.State, SessionState.NotStarted);

            _syncContext = syncContext;
            _hostIO = hostIO;
            this.IsAvailable = true;
            this.AllowRunning = false;
            _workDirPath = workDirPath;

            try
            {
                _thread = new Thread(this.ThreadMain);
                _thread.Name = $"{nameof(SpyDataSession)}.{nameof(ThreadMain)}";
                _thread.Start(port);
            }
            catch
            {
                this.StopAsync();

                if (_thread != null)
                {
                    _thread.Join();
                    _thread = null;
                }

                _syncContext = null;
                _hostIO = null;
                this.IsAvailable = false;
                throw;
            }
        }

        public void StopAsync()
        {
            this.IsAvailable = false;
            this.StopDataListener();
        }

        private void WaitForStateChange()
        {
            _stateChangedEventWait.WaitOne(1000 /* milli-seconds */);
        }

        private bool Connect(object port)
        {
            Ensure.Operation.True(this.State == SessionState.Starting);

            try
            {
                // DataChannel はパケットを同期受信待ちするので、ReceiveTimeout は Infinite
                _channel = this.ConnectHostIO(port, Timeout.Infinite, Timeout.Infinite);
                if (_channel == null)
                {
                    return false;
                }
            }
            catch
            {
                this.DisconnectHostIO();
                throw;
            }

            // 通信パフォーマンスを向上させるために、BufferedStream を利用
            _stream = new BufferedStream(_channel.GetStream());
            Ensure.Operation.NotNull(_stream);

            _port = port;

            if (this.IsLittleEndian)
            {
                _reader = LittleEndianBinaryReader.Create(_stream, Encoding.ASCII);
            }
            else
            {
                _reader = BigEndianBinaryReader.Create(_stream, Encoding.ASCII);
            }

            return true;
        }

        private void Disconnect()
        {
            _reader = null;

            if (_stream != null)
            {
                _stream.Close();
                _stream = null;
            }

            this.DisconnectHostIO();
            _port = null;
        }

        private IComChannel ConnectHostIO(object channelID, int sendTimeout, int receiveTimeout)
        {
            lock (_hostIOLock)
            {
                _hostIO.Connect(_syncContext);
                if (!this.IsConnected)
                {
                    return null;
                }
            }

            var stopwatch = Stopwatch.StartNew();
            while (true)
            {
                if (!this.IsAvailable || stopwatch.ElapsedMilliseconds > ConnectionTimeout)
                {
                    return null;
                }

                IComChannel result = null;

                lock (_hostIOLock)
                {
                    if (!this.IsConnected)
                    {
                        return null;
                    }

                    result = _hostIO.OpenChannel(channelID);
                }

                if (result != null)
                {
                    result.SendTimeout = sendTimeout;
                    result.ReceiveTimeout = receiveTimeout;

                    return result;
                }

                Thread.Sleep(10);
            }
        }

        private void DisconnectHostIO()
        {
            if (_channel != null)
            {
                _channel.Close();
                _channel = null;
            }

            lock (_hostIOLock)
            {
                if (_hostIO != null)
                {
                    _hostIO.Disconnect();
                    _hostIO = null;
                }
            }

            _syncContext = null;
        }

        /// <summary>
        /// データ通信スレッドです。
        /// 状態変更はすべてこのスレッドで行います。
        /// </summary>
        /// <param name="port">ポートを指定します。</param>
        private void ThreadMain(object port)
        {
            this.State = SessionState.Starting;

            try
            {
                // インスタンスを使いまわす
                var packet = new DataPacket();

                while (true)
                {
                    if (!this.IsAvailable)
                    {
                        this.State = SessionState.Stopping;
                        break;
                    }

                    try
                    {
                        // 接続処理
                        if (this.State == SessionState.Running)
                        {
                            this.ProcessDataPacket(packet);
                        }
                        else if (this.State == SessionState.Starting)
                        {
                            if (!this.Connect(port))
                            {
                                return;
                            }

                            this.State = SessionState.WaitForSync;
                        }
                        else if (this.State == SessionState.WaitForSync)
                        {
                            if (this.AllowRunning)
                            {
                                this.State = SessionState.Running;
                            }
                            else
                            {
                                // IsAvailable, AllowRunning のいずれかの値が変化するまでブロックします。
                                this.WaitForStateChange();
                            }
                        }
                        else if (this.State == SessionState.NotStarted)
                        {
                            Assertion.Fail("invalid state.");
                            break;
                        }
                    }
                    catch
                    {
                        // 接続中に例外が発生した場合は、停止します。
                        this.StopAsync();
                        break;
                    }
                }
            }
            finally
            {
                this.Disconnect();
                _thread = null;

                this.State = SessionState.NotStarted;
            }
        }

        private void StartDataListener()
        {
            const string DATA_FILE_NAME0 = "SpyData0.spychan";
            const string DATA_FILE_NAME1 = "SpyData1.spychan";

            if (_dataListenThread == null)
            {
                try
                {
                    lock (_hostIOLock)
                    {
                        // 初期化中にタイムアウトなどで接続が切れた場合。
                        if (_hostIO == null)
                        {
                            return;
                        }

                        _dataFileReader0 = CreateDateReader(DATA_FILE_NAME0);
                        _dataFileReader1 = CreateDateReader(DATA_FILE_NAME1);

                        _stopEvent.Reset();
                        _dataListenThread = new Thread(this.DataListenThreadMain);
                        _dataListenThread.Start();
                    }
                }
                catch
                {
                    this.StopDataListener();
                }
            }
        }

        private void StopDataListener()
        {
            if (_dataListenThread != null)
            {
                _stopEvent.Set();

                _dataListenThread.Join();
                _dataListenThread = null;
            }

            if (_dataFileReader0 != null)
            {
                _dataFileReader0.Dispose();
                _dataFileReader0 = null;
            }

            if (_dataFileReader1 != null)
            {
                _dataFileReader1.Dispose();
                _dataFileReader1 = null;
            }
        }

        private BinaryReader CreateDateReader(string fileName)
        {
            var stream = new BufferedStream(new FileStream(Path.Combine(_workDirPath, fileName), FileMode.Open, FileAccess.Read, FileShare.ReadWrite));
            if (this.IsLittleEndian)
            {
                return LittleEndianBinaryReader.Create(stream);
            }
            else
            {
                return BigEndianBinaryReader.Create(stream);
            }
        }

        private void DataListenThreadMain()
        {
            const int HEADER_SIZE = 16;
            const int BODY_HEADER_SIZE = 16;

            var packet = new DataPacket(); // インスタンスを使いまわす
            var reader = _dataFileReader0;
            var stream = _dataFileReader0.BaseStream;

#if DebugState
            Debug.WriteLine($"[SpyDataSession.DataListenThreadMin] PushDataReadPacket(0) ({DateTime.Now})");
#endif
            this.PushNotifyDataReadPacket(0);

            // スレッドの終了イベント(stopEvent)が設定されるまで、パケットを読み続ける。
            while (!_stopEvent.WaitOne(0))
            {
                // 通信用ファイルを切り替える。
                bool needFileSwitch = false;

                // ヘッダが読めるようになるまでポーリング。
                if (!this.WaitForDataArrive(stream, stream.Position + HEADER_SIZE))
                {
                    return;
                }

                this.ReadHeader(reader, packet.Header);
                var dataPosition = stream.Position;

                Ensure.Operation.True(new string(packet.Header.Signature) == SpySession._Signature);

                switch ((PacketID)packet.Header.ID)
                {
                    default:
                        break; // 無視

                    case PacketID.DataEnd:
                        needFileSwitch = true;
                        break;

                    case PacketID.Data:
                        {
                            // 構造上はパケット内に複数のDataパケットを含めることができるので対応しておく。
                            while (stream.Position + BODY_HEADER_SIZE < dataPosition + packet.Header.Size)
                            {
                                // データパケット・ヘッダが読めるようになるまでポーリング。
                                if (!this.WaitForDataArrive(stream, stream.Position + BODY_HEADER_SIZE))
                                {
                                    return;
                                }

                                this.ReadDataBody(reader, packet.Body);

                                Ensure.Operation.True(stream.Position + packet.Body.PayloadLength + packet.Body.PaddingLength <= dataPosition + packet.Header.Size);

                                // データパケット・ボディが読めるようになるまでポーリング。
                                if (!this.WaitForDataArrive(stream, stream.Position + packet.Body.PayloadLength + packet.Body.PaddingLength))
                                {
                                    return;
                                }

                                this.ReadDataPayload(reader, packet.Body);

                                Ensure.Operation.True(packet.Body.Timestamp <= long.MaxValue);

                                this.RaiseDataEvent(packet.Body.DataID, packet.Body.Payload, (long)(packet.Body.Timestamp));
                            }
                        }
                        break;
                }

                // パケット末尾のパディングが読めるようになるまでポーリング。
                if (!this.WaitForDataArrive(stream, dataPosition + packet.Header.Size))
                {
                    return;
                }

                // パケット末尾のパディングを読み飛ばし。
                if (stream.Position < dataPosition + packet.Header.Size)
                {
                    stream.Seek(dataPosition + packet.Header.Size, SeekOrigin.Begin);
                }

                // データファイルが終端に達したら読み出すデータファイルを切り替える。
                if (needFileSwitch)
                {
                    reader = (reader == _dataFileReader0) ? _dataFileReader1 : _dataFileReader0;
                    stream = reader.BaseStream;
                    stream.Seek(0, SeekOrigin.Begin);

                    // アプリに次のデータファイルの読み出しに入ったことを通知する。
                    this.PushNotifyDataReadPacket((reader == _dataFileReader0) ? 0u : 1u);
                }
            }
        }

        /// <summary>
        /// Stream.Length が目標の長さになるまでスリープします。
        /// </summary>
        /// <param name="stream">ストリームです。</param>
        /// <param name="length">目標とする Stream.Length の値です。</param>
        /// <returns>
        /// Length が目標に達したら true を返します。
        /// その前にスレッドの終了イベントがセットされると false を返します。
        /// </returns>
        private bool WaitForDataArrive(Stream stream, long length)
        {
            while (stream.Length < length)
            {
                if (_stopEvent.WaitOne(PollingInterval))
                {
                    return false;
                }
            }

            return true;
        }

        private void ProcessDataPacket(DataPacket packet)
        {
            Assertion.Argument.NotNull(packet);

            try
            {
                // リフレクションを使って読むと遅いので、ベタに読み込みます。
                this.ReadHeader(_reader, packet.Header);

                Ensure.Operation.True(new string(packet.Header.Signature) == SpySession._Signature);

                switch ((PacketID)packet.Header.ID)
                {
                    case PacketID.Data:
                        this.ReadDataBody(_reader, packet.Body);
                        this.ReadDataPayload(_reader, packet.Body);
                        Ensure.Operation.True(packet.Body.Timestamp < long.MaxValue);
                        this.RaiseDataEvent(packet.Body.DataID, packet.Body.Payload, (long)(packet.Body.Timestamp));
                        break;

                    case PacketID.SetOutputDirReply:
                        switch (packet.Header.Result)
                        {
                            case 0:
                                this.StopDataListener();
                                break;

                            case 1:
                                this.StartDataListener();
                                break;
                        }
                        this.SkipBytes(_reader, (int)packet.Header.Size);
                        break;

                    default:
                        this.SkipBytes(_reader, (int)packet.Header.Size);
                        break;
                }

                Ensure.Operation.True(packet.Header.IsResultSuccess);
            }
            catch (EndOfStreamException)
            {
                throw;
            }
            // TODO : ★タイムアウトしないようにできないか要調査
            //        CTR は非同期通信できないので、非同期APIを利用するよう変更するのもきつそう
            // ReceiveTimeout を Inifinite に設定しても何故かタイムアウトするので、無視しておく
            catch (IOException ex)
            {
                if ((ex.InnerException as SocketException)?.SocketErrorCode != SocketError.TimedOut)
                {
                    throw;
                }
            }
        }

        private void ReadHeader(BinaryReader reader, PacketHeader header)
        {
            Assertion.Argument.NotNull(reader);
            Assertion.Argument.NotNull(header);

            // リフレクションを使って読むと遅いので、ベタに読み込みます。
            var signature = reader.ReadChars(PacketHeader.SignatureSize);
            Ensure.True(signature.Length == PacketHeader.SignatureSize, () => new EndOfStreamException());
            header.Signature = signature;
            header.ID = reader.ReadUInt32();
            header.Size = reader.ReadUInt32();
            header.Result = reader.ReadUInt32();
        }

        private void ReadDataBody(BinaryReader reader, DataPacket.Content body)
        {
            Assertion.Argument.NotNull(reader);
            Assertion.Argument.NotNull(body);

            // リフレクションを使って読むと遅いので、ベタに読み込みます。
            var timestampLow = reader.ReadUInt32();
            var timestampHigh = reader.ReadUInt32();
            body.Timestamp = (((ulong)timestampHigh) << 32) + timestampLow;
            body.DataID = reader.ReadUInt32();
            body.PayloadLength = reader.ReadUInt32();
            body.PaddingLength = reader.ReadUInt32();
        }

        private void ReadDataPayload(BinaryReader reader, DataPacket.Content body)
        {
            Assertion.Argument.NotNull(reader);
            Assertion.Argument.NotNull(body);

            // リフレクションを使って読むと遅いので、ベタに読み込みます。
            body.Payload = reader.ReadBytes((int)body.PayloadLength);
            Ensure.True(body.Payload.Length == body.PayloadLength, () => new EndOfStreamException());

            // パディングを読み捨てる
            this.SkipBytes(reader, (int)body.PaddingLength);
        }

        /// <summary>
        /// 指定した長さのバイト列を読み飛ばします。
        /// </summary>
        /// <param name="reader">バイナリリーダーです。</param>
        /// <param name="length">読み飛ばすバイト数です。</param>
        private void SkipBytes(BinaryReader reader, int length)
        {
            const int ONCE_LENGTH = 512;

            Assertion.Argument.NotNull(reader);
            Assertion.Argument.NotNull(reader.BaseStream);
            Assertion.Argument.True(length >= 0);

            if (length <= 0)
            {
                return;
            }

            if (length <= ONCE_LENGTH)
            {
                var bytes = reader.ReadBytes(length);
                Ensure.True(bytes.Length == length, () => new EndOfStreamException());
            }
            else
            {
                // reader.ReadBytes()は毎回メモリを確保するので
                // 代わりにreader.BaseStream.ReadBytes()を使う。
                var bytes = new byte[ONCE_LENGTH];
                while (length > 0)
                {
                    var readBytes = Math.Min(ONCE_LENGTH, length);
                    var actualReadBytes = reader.BaseStream.Read(bytes, 0, readBytes);
                    Ensure.True(actualReadBytes == readBytes, () => new EndOfStreamException());
                    length -= readBytes;
                }
            }
        }

        private void RaiseDataEvent(uint dataID, byte[] data, long timestamp)
        {
            Assertion.Argument.NotNull(data);

            if (this.DataReceived != null)
            {
                this.DataReceived(this, new SpyRawDataEventArgs(dataID, data, timestamp));
            }
        }
    }
}
