﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.Spy.Settings;
using System;
using System.Windows;

namespace NintendoWare.Spy.Windows
{
    public class WindowPositionViewModel : ObservableObject
    {
        private readonly object _observerOwner = new object();
        private readonly WeakReference<WindowPosition> _model = new WeakReference<WindowPosition>(null);

        private double _left = double.NaN;
        private double _top = double.NaN;
        private double _width = double.NaN;
        private double _height = double.NaN;
        private double _normalWidth = double.NaN;
        private double _normalHeight = double.NaN;
        private WindowState _windowState = WindowState.Normal;

        //-----------------------------------------------------------------

        public double Left
        {
            get { return _left; }
            set { this.SetPropertyValue(ref _left, value); }
        }

        public double Top
        {
            get { return _top; }
            set { this.SetPropertyValue(ref _top, value); }
        }

        public double Width
        {
            get { return _width; }
            set
            {
                if (this.SetPropertyValue(ref _width, value))
                {
                    if (this.WindowState == WindowState.Normal)
                    {
                        this.NormalWidth = value;
                    }
                }
            }
        }

        public double Height
        {
            get { return _height; }
            set
            {
                if (this.SetPropertyValue(ref _height, value))
                {
                    if (this.WindowState == WindowState.Normal)
                    {
                        this.NormalHeight = value;
                    }
                }
            }
        }

        public double NormalWidth
        {
            get { return _normalWidth; }
            private set { this.SetPropertyValue(ref _normalWidth, value); }
        }

        public double NormalHeight
        {
            get { return _normalHeight; }
            private set { this.SetPropertyValue(ref _normalHeight, value); }
        }

        public WindowState WindowState
        {
            get { return _windowState; }
            set { this.SetPropertyValue(ref _windowState, value); }
        }

        //-----------------------------------------------------------------

        protected override void DisposeManagedInstance()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);

            base.DisposeManagedInstance();
        }

        public void SetModel(WindowPosition model)
        {
            var currentModel = _model.GetTarget();

            if (currentModel == model)
            {
                return;
            }

            if (currentModel != null)
            {
                PropertyChangedObservation.RemoveObservers(_observerOwner);
            }

            _model.SetTarget(model);

            if (model != null)
            {
                this.Left = model.Left;
                this.Top = model.Top;
                this.Width = model.Width;
                this.Height = model.Height;
                this.NormalWidth = model.Width;
                this.NormalHeight = model.Height;
                this.WindowState = model.WindowState;

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.Left,
                        (target, args) => this.Left = target.Left);

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.Top,
                        (target, args) => this.Top = target.Top);

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.Width,
                        (target, args) =>
                        {
                            this.Width = target.Width;
                            this.NormalWidth = target.Width;
                        });

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.Height,
                        (target, args) =>
                        {
                            this.Height = target.Height;
                            this.NormalHeight = target.Height;
                        });

                PropertyChangedObservation.GetObserver(_observerOwner, model)
                    .AddHandler(
                        target => target.WindowState,
                        (target, args) => this.WindowState = target.WindowState);
            }
        }
    }
}
