﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Plugins;
using NintendoWare.Spy.Settings;
using System.Windows.Controls;

namespace NintendoWare.Spy.Windows
{
    internal class PlotPanelPresenter : SpyPanelPresenter
    {
        private readonly object _observerOwner = new object();
        private readonly object _settingsObserverOwner = new object();
        private PlotPanel _view;
        private PlotPanelViewModel _viewModel;

        protected override void OnInitialize()
        {
            base.OnInitialize();

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Current,
                    (target, args) => UpdateCurrentTime());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Begin,
                    (target, args) => UpdateBeginTime());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.End,
                    (target, args) => UpdateEndTime());

            this.UpdateBeginTime();
            this.UpdateEndTime();
            this.UpdateCurrentTime();
        }

        protected override void OnUninitialize()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            Disposer.SafeDispose(ref _viewModel);

            base.OnUninitialize();
        }

        protected override Control CreateContent()
        {
            _viewModel = new PlotPanelViewModel(this.GetSpyService(), this.GetPlaybackService());

            _view = new PlotPanel()
            {
                DataContext = _viewModel,
            };

            return _view;
        }

        protected override void UpdateSpyModel(string dataName, SpyModel model)
        {
            switch (dataName)
            {
                case PlotSpyModelPlugin.SpyDataName:
                    _viewModel.SetPlotModel((PlotSpyModel)model);
                    break;

                case MarkerSpyModelPlugin.SpyDataName:
                    _viewModel.SetMarkerModel((MarkerSpyModel)model);
                    break;

                case FrameSyncSpyModelPlugin.SpyDataName:
                    _viewModel.SetFrameSyncModel((FrameSyncSpyModel)model);
                    break;
            }
        }

        private void ApplySettings(ApplicationSettings settings)
        {
            if (_viewModel != null)
            {
                _viewModel.TimeUnit = settings.TimeUnit;
            }
        }

        protected override void OnSettingsApplied(SettingsService service)
        {
            base.OnSettingsApplied(service);

            this.ApplySettings(service.GetApplicationSettings());

            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            PropertyChangedObservation.GetObserver(_settingsObserverOwner, service.GetApplicationSettings())
                .AddHandlerForAnyProperties(
                    (target, args) => this.ApplySettings(target));
        }

        private void UpdateCurrentTime()
        {
            _viewModel.SetCurrentFrameFromPresenter(
                this.GetPlaybackService().Current.GetMicroSeconds());
        }

        private void UpdateBeginTime()
        {
            _viewModel.MinimumFrame =
                this.GetPlaybackService().Begin.GetMicroSeconds();
        }

        private void UpdateEndTime()
        {
            _viewModel.MaximumFrame =
                this.GetPlaybackService().End.GetMicroSeconds();
        }
    }
}
