﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.Commands;
using System;
using System.Diagnostics.CodeAnalysis;
using System.IO;

namespace NintendoWare.Spy.Commands
{
    /// <summary>
    /// 録音波形の保存 コマンドのハンドラです。
    /// </summary>
    internal static class AudioSaveRecordWaveDataHandler
    {
        public static CommandStatus CanExecute(AudioSaveRecordWaveDataCommandArgs commandArgs)
        {
            if (commandArgs == null)
            {
                return CommandStatus.Unsupported;
            }

            if (commandArgs.SpyService == null)
            {
                return CommandStatus.Unsupported;
            }

            if (commandArgs.SpyService.IsLoading)
            {
                return CommandStatus.Unsupported;
            }

            if (commandArgs.SpyPlaybackService == null)
            {
                return CommandStatus.Disabled;
            }

            if (commandArgs.SpyPlaybackService.FinalOutModel == null &&
                commandArgs.SpyPlaybackService.WaveformModel == null)
            {
                return CommandStatus.Disabled;
            }

            if (commandArgs.WaveSourceOutputInfo == null)
            {
                return CommandStatus.Disabled;
            }

            if (commandArgs.ConnectionSettings == null)
            {
                return CommandStatus.Disabled;
            }

            if (commandArgs.ConnectionSettings.IsConnecting)
            {
                return CommandStatus.Disabled;
            }

            if (string.IsNullOrEmpty(commandArgs.OutputFilePath))
            {
                return CommandStatus.Disabled;
            }

            return CommandStatus.Enabled;
        }

        public static CommandResult Execute(AudioSaveRecordWaveDataCommandArgs commandArgs)
        {
            Ensure.Operation.True(CanExecute(commandArgs).IsEnabled());

            try
            {
                IWaveSource waveSource = null;

                var finalOutSpyModel = commandArgs.SpyPlaybackService.FinalOutModel;
                if (finalOutSpyModel != null)
                {
                    waveSource = new FinalOutWaveSource()
                    {
                        AudioFrameBegin = commandArgs.SpyPlaybackService.Begin.AudioFrame,
                        AudioFrameEnd = commandArgs.SpyPlaybackService.End.AudioFrame,
                        FinalOut = finalOutSpyModel,
                    };
                }
                else
                {
                    var waveformSpyModel = commandArgs.SpyPlaybackService.WaveformModel;
                    if (waveformSpyModel != null)
                    {
                        waveSource = new WaveformWaveSource()
                        {
                            BeginTime = commandArgs.SpyPlaybackService.Begin.Timestamp,
                            EndTime = commandArgs.SpyPlaybackService.End.Timestamp,
                            WaveformSpyModel = waveformSpyModel,
                        };
                    }
                }

                if (waveSource != null)
                {
                    using (var fileStream = File.Open(commandArgs.OutputFilePath, FileMode.Create, FileAccess.Write, FileShare.Read))
                    {
                        WriteRawWaveData(fileStream, waveSource, commandArgs.WaveSourceOutputInfo);
                    }
                }
            }
            catch
            {
                return new CommandResult(CommandResult.Code.Failed);
            }

            return CommandResult.CreateSucceeded();
        }

        [SuppressMessage("StyleCop.CSharp.ReadabilityRules", "SA1121:UseBuiltInTypeAlias", Justification = "バイナリのサイズを明示するため")]
        private static void WriteRawWaveData(Stream stream, IWaveSource waveSource, WaveSourceOutputInfo waveSourceOutputInfo)
        {
            Assertion.Argument.NotNull(stream);
            Assertion.Argument.NotNull(waveSource);

            int channelCount = waveSourceOutputInfo.ChannelCount;
            var buffer = new byte[1 * 1024 * 1024];

            using (var binWriter = new BinaryWriter(stream))
            {
                UInt16 nBlockAlign = (UInt16)(sizeof(UInt16) * channelCount);
                UInt32 sampleRate = (UInt32)waveSource.SamplingRate;
                UInt32 nAvgBytesPerSec = sampleRate * nBlockAlign;

                binWriter.Write(new char[4] { 'R', 'I', 'F', 'F' });
                binWriter.Write((UInt32)0); // chunk size
                binWriter.Write(new char[4] { 'W', 'A', 'V', 'E' });
                binWriter.Write(new char[4] { 'f', 'm', 't', ' ' });
                binWriter.Write((UInt32)16); // chunk size
                binWriter.Write((UInt16)1); // PCM
                binWriter.Write((UInt16)channelCount); // channel
                binWriter.Write(sampleRate);
                binWriter.Write(nAvgBytesPerSec);
                binWriter.Write(nBlockAlign);
                binWriter.Write((UInt16)16); // bits
                binWriter.Write(new char[4] { 'd', 'a', 't', 'a' });
                binWriter.Write((UInt32)0); // chunk size

                UInt32 dataChunkSize = 0;

                while (true)
                {
                    var validLength = waveSource.Mix(buffer, 0, buffer.Length, waveSourceOutputInfo.Name, channelCount);

                    if (validLength <= 0)
                    {
                        break;
                    }

                    stream.Write(buffer, 0, validLength);
                    dataChunkSize += (UInt32)validLength;
                }

                binWriter.Seek(4, SeekOrigin.Begin);
                binWriter.Write(0x24 + dataChunkSize);
                binWriter.Seek(0x28, SeekOrigin.Begin);
                binWriter.Write(dataChunkSize);
            }
        }
    }
}
