﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Windows.Data;
using System.Windows.Media;

namespace NintendoWare.SpySample.Windows
{
    public class SampleBPanelViewModel : ObservableObject
    {
        private const int HistoryMax = 32;
        private const string ActorPositionXName = "@SpySampleB/ActorPositionX";
        private const string ActorPositionYName = "@SpySampleB/ActorPositionY";

        private readonly object observerOwner = new object();
        private readonly object plotObserverOwner = new object();

        private readonly SpyPlaybackService playbackService;

        private PlotSpyModel plotModel;

        private SpyTimeUnit timeUnit = SpyTimeUnit.Timestamp;

        private PlotSpyModel.PlotFloat positionX;
        private PlotSpyModel.PlotFloat positionY;

        private Brush strokeX;
        private Brush strokeY;

        private double maximumX;
        private double minimumX;
        private double maximumY;
        private double minimumY;

        private PositionViewModel currentPosition = PositionViewModel.NullObject;
        private ObservableCollection<PositionViewModel> positionHistory = new ObservableCollection<PositionViewModel>();

        private ObservableList<PlotFloatViewModel> plotFloats = new ObservableList<PlotFloatViewModel>();

        public class PositionViewModel : ObservableObject
        {
            private double x;
            private double y;
            private Brush fill;

            public static PositionViewModel NullObject { get; } = new PositionViewModel();

            public double X
            {
                get { return this.x; }
                set { this.SetPropertyValue(ref this.x, value); }
            }

            public double Y
            {
                get { return this.y; }
                set { this.SetPropertyValue(ref this.y, value); }
            }

            public Brush Fill
            {
                get { return this.fill; }
                set { this.SetPropertyValue(ref this.fill, value); }
            }
        }

        public SampleBPanelViewModel(SpyPlaybackService playbackService)
        {
            Ensure.Argument.NotNull(playbackService);

            this.playbackService = playbackService;

            PropertyChangedObservation.GetObserver(this.observerOwner, this.playbackService).AddHandler(
                target => target.Current,
                (sender, e) => this.UpdateCurrentTime());

            PropertyChangedObservation.GetObserver(this.observerOwner, this.playbackService).AddHandler(
                target => target.Begin,
                (sender, e) => this.NotifyPropertyChanged(nameof(MinimumFrame)));

            PropertyChangedObservation.GetObserver(this.observerOwner, this.playbackService).AddHandler(
                target => target.End,
                (sender, e) => this.NotifyPropertyChanged(nameof(MaximumFrame)));
        }

        protected override void DisposeManagedInstance()
        {
            PropertyChangedObservation.RemoveObservers(this.observerOwner);
            CollectionChangedObservation.RemoveObservers(this.plotObserverOwner);

            base.DisposeManagedInstance();
        }

        public SpyTimeUnit TimeUnit
        {
            get { return this.timeUnit; }
            set { this.SetPropertyValue(ref this.timeUnit, value); }
        }

        public double CurrentFrame
        {
            get { return this.playbackService.Current.GetMicroSeconds(); }
            set
            {
                if (this.CurrentFrame == value)
                {
                    return;
                }

                if (!double.IsNaN(value))
                {
                    this.playbackService?.SetCurrentTime(SpyGlobalTime.FromMicroSeconds((long)value));
                }

                this.NotifyPropertyChanged(nameof(CurrentFrame));
            }
        }
        public double MinimumFrame
        {
            get { return this.playbackService.Begin.GetMicroSeconds(); }
        }

        public double MaximumFrame
        {
            get { return this.playbackService.End.GetMicroSeconds(); }
        }

        public Brush StrokeX
        {
            get { return this.strokeX; }
            private set { this.SetPropertyValue(ref this.strokeX, value); }
        }

        public Brush StrokeY
        {
            get { return this.strokeY; }
            private set { this.SetPropertyValue(ref this.strokeY, value); }
        }

        public double MaximumX
        {
            get { return this.maximumX; }
            private set { this.SetPropertyValue(ref this.maximumX, value); }
        }

        public double MinimumX
        {
            get { return this.minimumX; }
            private set { this.SetPropertyValue(ref this.minimumX, value); }
        }

        public double MaximumY
        {
            get { return this.maximumY; }
            private set { this.SetPropertyValue(ref this.maximumY, value); }
        }

        public double MinimumY
        {
            get { return this.minimumY; }
            private set { this.SetPropertyValue(ref this.minimumY, value); }
        }

        public PositionViewModel CurrentPosition
        {
            get { return this.currentPosition; }
            private set { this.SetPropertyValue(ref this.currentPosition, value); }
        }

        public ICollectionView PositionHistory
        {
            get { return CollectionViewSource.GetDefaultView(this.positionHistory); }
        }

        public IList<PlotFloatViewModel> PlotFloats
        {
            get { return this.plotFloats; }
        }

        public void SetPlotModel(PlotSpyModel model)
        {
            if (this.plotModel == model)
            {
                return;
            }

            this.Clear();

            this.plotModel = model;
            this.SyncPlotItems();
            this.UpdateCurrentTime();
        }

        private void Clear()
        {
            this.positionX = null;
            this.positionY = null;
            this.plotFloats.Clear();
            this.positionHistory.Clear();

            this.CurrentPosition = PositionViewModel.NullObject;
            this.plotModel = null;
        }

        private void SyncPlotItems()
        {
            CollectionChangedObservation.RemoveObservers(this.plotObserverOwner);

            if (this.plotModel == null)
            {
                this.positionX = null;
                this.positionY = null;

                return;
            }

            if (!this.FindDataBufferPlotFloats(this.plotModel))
            {
                CollectionChangedObservation.GetObserver(this.plotObserverOwner, this.plotModel.Floats).AddHandlerForAddItems(
                    (sender, e) => this.FindDataBufferPlotFloats(this.plotModel));
            }
        }

        private bool FindDataBufferPlotFloats(PlotSpyModel model)
        {
            if (this.positionX == null)
            {
                this.positionX = model.GetFloat(ActorPositionXName);

                if (this.positionX != null)
                {
                    this.MaximumX = this.positionX.Maximum;
                    this.MinimumX = this.positionX.Minimum;
                    this.StrokeX = new SolidColorBrush(Color.FromRgb(this.positionX.Color.R, this.positionX.Color.G, this.positionX.Color.B));

                    this.plotFloats.Add(new PlotFloatViewModel(this.positionX));
                }
            }

            if (this.positionY == null)
            {
                this.positionY = model.GetFloat(ActorPositionYName);

                if (this.positionY != null)
                {
                    this.MaximumY = this.positionY.Maximum;
                    this.MinimumY = this.positionY.Minimum;
                    this.StrokeY = new SolidColorBrush(Color.FromRgb(this.positionY.Color.R, this.positionY.Color.G, this.positionY.Color.B));

                    this.plotFloats.Add(new PlotFloatViewModel(this.positionY));
                }
            }

            return this.positionX != null && this.positionY != null;
        }

        private void UpdateCurrentTime()
        {
            this.positionHistory.Clear();

            if (this.positionX == null || this.positionY == null)
            {
                this.CurrentPosition = PositionViewModel.NullObject;
                return;
            }

            var positionXIndex = this.FindNearestValueIndex(this.positionX, this.playbackService.Current.Timestamp);
            var positionYIndex = this.FindNearestValueIndex(this.positionY, this.playbackService.Current.Timestamp);

            if (!positionXIndex.HasValue || !positionYIndex.HasValue)
            {
                this.CurrentPosition = PositionViewModel.NullObject;
                return;
            }

            this.CurrentPosition = new PositionViewModel()
            {
                X = this.positionX.Values[positionXIndex.Value].Value,
                Y = this.positionY.Values[positionYIndex.Value].Value,
            };

            this.positionHistory.Add(this.CurrentPosition);
            positionXIndex--;
            positionYIndex--;

            while (this.positionHistory.Count < HistoryMax
                && positionXIndex >= 0
                && positionYIndex >= 0)
            {
                var newPosition = new PositionViewModel()
                {
                    X = this.positionX.Values[positionXIndex.Value].Value,
                    Y = this.positionY.Values[positionYIndex.Value].Value,
                };
                this.positionHistory.Insert(0, newPosition);

                positionXIndex--;
                positionYIndex--;
            }

            // プロット色の更新
            foreach (var index in Enumerable.Range(0, this.positionHistory.Count))
            {
                var alpha = 64 + (191 * (index + HistoryMax - this.positionHistory.Count) / HistoryMax);

                if (index > 0
                    && index < this.positionHistory.Count - 1
                    && this.positionHistory[index - 1].X == this.positionHistory[index].X
                    && this.positionHistory[index - 1].Y == this.positionHistory[index].Y)
                {
                    alpha = 0;
                }

                var brush = new SolidColorBrush(Color.FromArgb((byte)alpha, this.positionX.Color.R, this.positionX.Color.G, this.positionX.Color.B));
                brush.Freeze();

                this.positionHistory[index].Fill = brush;
            }

            this.NotifyPropertyChanged(nameof(CurrentFrame));
        }

        private int? FindNearestValueIndex(PlotSpyModel.PlotFloat model, SpyGlobalTime time)
        {
            if (model.Values.Count == 0)
            {
                return null;
            }

            if (model.Values.Count == 1)
            {
                return 0;
            }

            var index = BinarySearchUtility.BinarySearch(model.Values, time, value => value.Timestamp, BinarySearchUtility.Options.SmallestIndex);

            if (index < 0)
            {
                index = Math.Max(0, ~index - 1);
            }

            return index >= model.Values.Count ? null : (int?)index;
        }
    }
}
