﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Linq;
using System.Windows;
using System.Windows.Controls;

namespace NintendoWare.SpySample.Windows.Controls
{
    [TemplatePart(Name = PartCanvas, Type = typeof(Canvas))]
    public class Plot2DView : ItemsControl
    {
        private const string PartCanvas = "PartCanvas";
        private const int MaximumGridCount = 500;

        public static readonly DependencyProperty ScaleProperty =
            DependencyProperty.Register(
                nameof(Scale),
                typeof(double),
                typeof(Plot2DView),
                new FrameworkPropertyMetadata(1.0),
                v => IsValidFiniteDouble(v, d => d > 0));

        public static readonly DependencyProperty MaximumXProperty =
            DependencyProperty.Register(
                nameof(MaximumX),
                typeof(double),
                typeof(Plot2DView),
                new FrameworkPropertyMetadata(
                    100.0,
                    (d, e) => { },
                    (d, v) => Self(d).CoerceMaximumX((double)v)),
                IsValidFiniteDouble);

        public static readonly DependencyProperty MinimumXProperty =
            DependencyProperty.Register(
                nameof(MinimumX),
                typeof(double),
                typeof(Plot2DView),
                new FrameworkPropertyMetadata(
                    0.0,
                    (d, e) => Self(d).OnMinimumXChanged(e)),
                IsValidFiniteDouble);

        public static readonly DependencyProperty MaximumYProperty =
            DependencyProperty.Register(
                nameof(MaximumY),
                typeof(double),
                typeof(Plot2DView),
                new FrameworkPropertyMetadata(
                    100.0,
                    (d, e) => { },
                    (d, v) => Self(d).CoerceMaximumY((double)v)),
                IsValidFiniteDouble);

        public static readonly DependencyProperty MinimumYProperty =
            DependencyProperty.Register(
                nameof(MinimumY),
                typeof(double),
                typeof(Plot2DView),
                new FrameworkPropertyMetadata(
                    0.0,
                    (d, e) => Self(d).OnMinimumYChanged(e)),
                IsValidFiniteDouble);

        public double Scale
        {
            get { return (double)this.GetValue(ScaleProperty); }
            set { this.SetValue(ScaleProperty, value); }
        }

        public double MaximumX
        {
            get { return (double)this.GetValue(MaximumXProperty); }
            set { this.SetValue(MaximumXProperty, value); }
        }

        public double MinimumX
        {
            get { return (double)this.GetValue(MinimumXProperty); }
            set { this.SetValue(MinimumXProperty, value); }
        }

        public double MaximumY
        {
            get { return (double)this.GetValue(MaximumYProperty); }
            set { this.SetValue(MaximumYProperty, value); }
        }

        public double MinimumY
        {
            get { return (double)this.GetValue(MinimumYProperty); }
            set { this.SetValue(MinimumYProperty, value); }
        }

        static Plot2DView()
        {
            DefaultStyleKeyProperty.OverrideMetadata(
                typeof(Plot2DView),
                new FrameworkPropertyMetadata(typeof(Plot2DView)));
        }

        internal void UpdateItemCanvasLeft(Plot2DViewItem item, double left)
        {
            // 論理座標からスクリーン座標に変換
            var width = this.MaximumX - this.MinimumX;
            var value = (left - this.MinimumX) * this.ActualWidth * this.Scale / width;

            // センタリング
            value -= item.ActualWidth / 2;

            item.SetValue(Canvas.LeftProperty, value);
        }

        internal void UpdateItemCanvasTop(Plot2DViewItem item, double top)
        {
            // 論理座標からスクリーン座標に変換
            var height = this.MaximumX - this.MinimumX;
            var value = (top - this.MinimumX) * this.ActualHeight * this.Scale / height;

            // センタリング
            value -= item.ActualHeight / 2;

            item.SetValue(Canvas.TopProperty, value);
        }

        protected override void OnItemsSourceChanged(IEnumerable oldValue, IEnumerable newValue)
        {
            base.OnItemsSourceChanged(oldValue, newValue);
        }

        protected override bool IsItemItsOwnContainerOverride(object item)
        {
            return item is Plot2DViewItem;
        }

        protected override DependencyObject GetContainerForItemOverride()
        {
            return new Plot2DViewItem();
        }

        protected override void PrepareContainerForItemOverride(DependencyObject element, object item)
        {
            base.PrepareContainerForItemOverride(element, item);

            var target = (Plot2DViewItem)element;
            target.SetOwner(this);
        }

        protected override void ClearContainerForItemOverride(DependencyObject element, object item)
        {
            ((Plot2DViewItem)element).SetOwner(null);

            base.ClearContainerForItemOverride(element, item);
        }

        protected override void OnRenderSizeChanged(SizeChangedInfo sizeInfo)
        {
            base.OnRenderSizeChanged(sizeInfo);

            this.UpdateItemCanvasPositions();
        }

        private void UpdateItemCanvasPositions()
        {
            foreach (Plot2DViewItem itemContainer in
                Enumerable.Range(0, this.ItemContainerGenerator.Items.Count)
                .Select(index => this.ItemContainerGenerator.ContainerFromIndex(index)))
            {
                this.UpdateItemCanvasLeft(itemContainer, itemContainer.X);
                this.UpdateItemCanvasTop(itemContainer, itemContainer.Y);
            }
        }

        private object CoerceMaximumX(double v)
        {
            return Math.Max(this.MinimumX, v);
        }

        private object CoerceMaximumY(double v)
        {
            return Math.Max(this.MinimumY, v);
        }

        private void OnMinimumXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(MaximumXProperty);
        }

        private void OnMinimumYChanged(DependencyPropertyChangedEventArgs e)
        {
            this.CoerceValue(MaximumYProperty);
        }

        private static bool IsValidFiniteDouble(object v)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsInfinity(d);
            }
            else
            {
                return false;
            }
        }

        private static bool IsValidFiniteDouble(object v, Func<double, bool> predicate)
        {
            if (v is double)
            {
                var d = (double)v;
                return !double.IsNaN(d) && !double.IsInfinity(d) && predicate(d);
            }
            else
            {
                return false;
            }
        }

        private static Plot2DView Self(DependencyObject d)
        {
            return (Plot2DView)d;
        }
    }
}
