﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Threading;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public class ZoomRatioCalculator
    {
        // -4 => 1:6
        // -3 => 1:4
        // -2 => 1:3
        // -1 => 1:2
        //  0 => 1:1
        //  1 => 2:1
        //  2 => 3:1
        //  3 => 4:1
        //  4 => 6:1
        //  5 => 8:1
        //  6 => 12:1
        public static int CalcZoomValue(int zoomIndex)
        {
            if (zoomIndex == 0)
            {
                return 1;
            }

            int index;
            bool minusFlag;

            if (zoomIndex > 0)
            {
                index = zoomIndex;
                minusFlag = false;
            }
            else
            {
                index = -zoomIndex;
                minusFlag = true;
            }

            int value;
            int n;
            if ((index & 0x1) != 0)
            {
                // 奇数の時は、2 * 2^n
                n = (index - 1) / 2;
                value = 2;
            }
            else
            {
                // 偶数の時は、3 * 2^n
                n = (index - 2) / 2;
                value = 3;
            }
            value <<= n;

            if (minusFlag)
            {
                return -value;
            }
            else
            {
                return value;
            }
        }
        public static long CalcPointCountFromWidth(int width, int zoomValue)
        {
            long pointCount;

            if (zoomValue > 0)
            {
                pointCount = width / zoomValue;
            }
            else
            {
                pointCount = width * (-zoomValue);
            }

            return pointCount;
        }
        public static int CalcWidthFromPointCount(long pointCount, int zoomValue)
        {
            long width;

            if (zoomValue > 0)
            {
                width = pointCount * zoomValue;
            }
            else
            {
                width = pointCount / (-zoomValue);
            }

            return (int)width;
        }
    }

    public class ZoomRatio
    {
        private int _zoomIndex = 0;
        private int _zoomValue = 1;

        public ZoomRatio(int index)
        {
            Index = index;
        }

        public int Index
        {
            get { return _zoomIndex; }
            set
            {
                _zoomIndex = value;
                _zoomValue = ZoomRatioCalculator.CalcZoomValue(_zoomIndex);
            }
        }

        public long CalcPointCountFromWidth(int width)
        {
            return ZoomRatioCalculator.CalcPointCountFromWidth(width, _zoomValue);
        }
        public int CalcWidthFromPointCount(long pointCount)
        {
            return ZoomRatioCalculator.CalcWidthFromPointCount(pointCount, _zoomValue);
        }
    }

    public class LineGraphDrawer
    {
        public class BitmapStock
        {
            public Bitmap Bitmap { get; private set; } = null;
            public long LeftSampleIndex { get; private set; } = -1;
            public long RightSampleIndex { get; private set; } = -1;
            public int ZoomRatioIndex { get; private set; } = -1;
            public int BeginSamplePosX { get; private set; } = -1;
            public float YScale { get; private set; } = 0.0f;

            public bool Apply(
                Graphics gBmp,
                int height,
                ref long drawLeft,
                ref long drawRight,
                long leftSampleIndex,
                long rightSampleIndex,
                int beginSamplePosX,
                int zoomRatioIndex,
                float yScale)
            {
                if (this.Bitmap == null || this.ZoomRatioIndex != zoomRatioIndex || this.Bitmap.Height != height || this.YScale != yScale)
                {
                    return false;
                }

                int zoomValue = ZoomRatioCalculator.CalcZoomValue(zoomRatioIndex);

                // ビットマップの再利用
                long left = Math.Max(this.LeftSampleIndex, leftSampleIndex);
                long right = Math.Min(this.RightSampleIndex, rightSampleIndex);

                if (left < right)
                {
                    int xLeft = beginSamplePosX + ZoomRatioCalculator.CalcWidthFromPointCount(left, zoomValue);

                    Rectangle srcRect = new Rectangle();
                    srcRect.X = this.BeginSamplePosX + ZoomRatioCalculator.CalcWidthFromPointCount(left, zoomValue);
                    srcRect.Width = this.BeginSamplePosX + ZoomRatioCalculator.CalcWidthFromPointCount(right, zoomValue) - srcRect.X;
                    srcRect.Y = 0;
                    srcRect.Height = this.Bitmap.Height;

                    gBmp.DrawImage(this.Bitmap, xLeft, 0, srcRect, GraphicsUnit.Pixel);

                    if (this.RightSampleIndex <= rightSampleIndex)
                    {
                        drawLeft = right - 1;
                    }
                    else if (leftSampleIndex < this.LeftSampleIndex)
                    {
                        drawRight = left + 1;
                    }
                }

                return true;
            }

            public void Update(
                Bitmap bmp,
                long leftSampleIndex,
                long rightSampleIndex,
                int beginSamplePosX,
                int zoomRatioIndex,
                float yScale)
            {
                this.Bitmap = bmp;
                this.ZoomRatioIndex = zoomRatioIndex;
                this.LeftSampleIndex = leftSampleIndex;
                this.RightSampleIndex = rightSampleIndex;
                this.BeginSamplePosX = beginSamplePosX;
                this.YScale = yScale;
            }
        }

        public delegate float GetSampleValueFunc(long sampleIndex, object userData);
        public delegate float[] GetSampleMinMaxFunc(long sampleIndexBegin, long sampleIndexEnd, object userData);

        public static void DrawLineGraph(
            Graphics g, Pen pen,
            int x, int y, int width, int height,
            int beginSamplePosX, int centerY,
            long beginSampleIndex, long endSampleIndex,
            int zoomRatioIndex, float yScale,
            GetSampleValueFunc getSampleValueFunc,
            GetSampleMinMaxFunc getSampleMinMaxFunc,
            object userData)
        {
            int RangeHeight = height / 2;
            int zoomValue = ZoomRatioCalculator.CalcZoomValue(zoomRatioIndex);

            if (zoomRatioIndex < -8)
            {
                // 縦線描画
                int leftPosX = beginSamplePosX + ZoomRatioCalculator.CalcWidthFromPointCount(beginSampleIndex, zoomValue);
                int rightPosX = beginSamplePosX + ZoomRatioCalculator.CalcWidthFromPointCount(endSampleIndex, zoomValue);
                leftPosX = MathUtility.Clamp(leftPosX, 0, width - 1);
                rightPosX = MathUtility.Clamp(rightPosX, 0, width - 1);

                PointF[] point = new PointF[2];

                for (int posX = leftPosX; posX <= rightPosX; posX++)
                {
                    long sampleIndexLeft = ZoomRatioCalculator.CalcPointCountFromWidth(posX - beginSamplePosX, zoomValue);
                    long sampleIndexRight = ZoomRatioCalculator.CalcPointCountFromWidth(posX + 1 - beginSamplePosX, zoomValue) - 1;

                    if (sampleIndexRight >= beginSampleIndex && sampleIndexLeft <= endSampleIndex)
                    {
                        sampleIndexLeft = MathUtility.Clamp(sampleIndexLeft, beginSampleIndex, endSampleIndex);
                        sampleIndexRight = MathUtility.Clamp(sampleIndexRight, beginSampleIndex, endSampleIndex);

                        float minValue = float.MaxValue;
                        float maxValue = float.MinValue;

                        if (getSampleMinMaxFunc != null)
                        {
                            float[] minmax = getSampleMinMaxFunc(sampleIndexLeft, sampleIndexRight, userData);
                            minValue = minmax[0];
                            maxValue = minmax[1];
                        }
                        else
                        {
                            for (long sampleIndex = sampleIndexLeft; sampleIndex <= sampleIndexRight; sampleIndex++)
                            {
                                float sampleValue = getSampleValueFunc(sampleIndex, userData);

                                minValue = Math.Min(minValue, sampleValue);
                                maxValue = Math.Max(maxValue, sampleValue);
                            }
                        }

                        point[0].X = x + posX;
                        point[0].Y = Math.Min(y - (minValue * yScale) + centerY, height - 1);
                        point[1].X = x + posX;
                        point[1].Y = Math.Min(y - (maxValue * yScale) + centerY, height - 1);

                        // 一定の値のときに見えなくならないように。
                        if (point[1].Y == point[0].Y)
                        {
                            point[1].X += 1;
                        }

                        g.DrawLine(pen, point[0], point[1]);
                    }
                }
            }
            else
            {
                // 通常折れ線描画
                DrawLineGraphBasic(
                    g, pen,
                    x, y,
                    width, height,
                    beginSamplePosX, centerY,
                    beginSampleIndex, endSampleIndex,
                    zoomRatioIndex, yScale,
                    getSampleValueFunc, userData);
            }
        }

        private static void DrawLineGraphBasic(
            Graphics g, Pen pen,
            int x, int y,
            int width, int height,
            int beginSamplePosX, int centerY,
            long beginSampleIndex, long endSampleIndex,
            int zoomRatioIndex, float yScale,
            GetSampleValueFunc getSampleValueFunc,
            object userData)
        {
            int zoomValue = ZoomRatioCalculator.CalcZoomValue(zoomRatioIndex);

            long pointCount = endSampleIndex - beginSampleIndex + 1;
            if (pointCount >= 2)
            {
                PointF[] points = new PointF[pointCount];

                long sampleIndex = endSampleIndex;
                int pointIndex = 0;

                while (sampleIndex >= 0 && pointIndex < pointCount)
                {
                    int posX = beginSamplePosX + ZoomRatioCalculator.CalcWidthFromPointCount(sampleIndex, zoomValue);
                    float sampleValue = getSampleValueFunc(sampleIndex, userData);

                    points[pointIndex].X = x + posX;
                    points[pointIndex].Y = Math.Min(y - sampleValue * yScale + centerY, height);

                    pointIndex++;
                    sampleIndex--;
                }

                g.DrawLines(pen, points);
            }
        }

        public static void DrawScrollBar(Graphics g, int x, int y, int width, int height, long leftSampleIndex, long rightSampleIndex, long lastSampleIndex)
        {
            g.FillRectangle(Brushes.LightGray, x, y, width, height);

            if (lastSampleIndex >= 0)
            {
                int x1 = (int)(width * leftSampleIndex / (lastSampleIndex + 1));
                int x2 = (int)(width * rightSampleIndex / (lastSampleIndex + 1));
                g.FillRectangle(
                    Brushes.Gray,
                    x + x1,
                    y,
                    x + x2 - x1 + 1,
                    height);
            }
        }
    }

    public static class CancelableLineGraphDrawer
    {
        private const int MinMaxGraphThreshold = -8;

        public delegate float[] GetSampleValueFunc(CancellationToken ct, long sampleIndex);
        public delegate float[] GetSampleMinMaxFunc(CancellationToken ct, long sampleIndexBegin, long sampleIndexEnd);

        public class DataStore
        {
            public Dictionary<long, float[]> Datas { get; private set; }
            public int ZoomRatioIndex { get; private set; }
            public bool IsMinMaxGraph { get; private set; }

            public DataStore(int zoomRatioIndex)
            {
                this.Datas = new Dictionary<long, float[]>();
                this.ZoomRatioIndex = zoomRatioIndex;
                this.IsMinMaxGraph = zoomRatioIndex < MinMaxGraphThreshold;
            }
        }

        /// <summary>
        /// グラフの描画に必要なデータを収集します。
        /// </summary>
        public static void FetchData(
            CancellationToken ct,
            DataStore dataStore,
            ref long leftSampleIndex,
            ref long rightSampleIndex,
            GetSampleValueFunc getSampleValueFunc,
            GetSampleMinMaxFunc getSampleMinMaxFunc = null)
        {
            Ensure.Argument.True(leftSampleIndex <= rightSampleIndex);
            Ensure.Argument.NotNull(getSampleValueFunc);

            if (dataStore.ZoomRatioIndex < MinMaxGraphThreshold)
            {
                // 縦線描画用
                FetchDataMinMax(
                    ct,
                    dataStore,
                    ref leftSampleIndex,
                    ref rightSampleIndex,
                    getSampleValueFunc,
                    getSampleMinMaxFunc);
            }
            else
            {
                // 折れ線描画用
                FetchDataBasic(
                    ct,
                    dataStore,
                    ref leftSampleIndex,
                    ref rightSampleIndex,
                    getSampleValueFunc);
            }
        }

        /// <summary>
        /// グラフを描画します。
        /// </summary>
        public static void Draw(
            Graphics g,
            Pen pen,
            int x,
            int y,
            int width,
            int height,
            int centerY,
            float scaleY,
            DataStore dataStore,
            int beginSamplePosX,
            long leftSampleIndex,
            long rightSampleIndex,
            int channel)
        {
            if (dataStore.IsMinMaxGraph)
            {
                DrawMinMaxGraph(
                    g,
                    pen,
                    x,
                    y,
                    width,
                    height,
                    centerY,
                    scaleY,
                    dataStore,
                    beginSamplePosX,
                    leftSampleIndex,
                    rightSampleIndex,
                    channel);
            }
            else
            {
                DrawBasicGraph(
                    g,
                    pen,
                    x,
                    y,
                    width,
                    height,
                    centerY,
                    scaleY,
                    dataStore,
                    beginSamplePosX,
                    leftSampleIndex,
                    rightSampleIndex,
                    channel);
            }
        }

        /// <summary>
        /// 通常の折れ線グラフ用にサンプルを取得します。
        /// </summary>
        private static void FetchDataBasic(
            CancellationToken ct,
            DataStore dataStore,
            ref long beginSampleIndex,
            ref long endSampleIndex,
            GetSampleValueFunc getSampleValueFunc)
        {
            long sampleIndex;
            for (sampleIndex = beginSampleIndex; sampleIndex <= endSampleIndex; ++sampleIndex)
            {
                if (ct.IsCancellationRequested)
                {
                    break;
                }

                if (!dataStore.Datas.ContainsKey(sampleIndex))
                {
                    var sample = getSampleValueFunc(ct, sampleIndex);
                    if (sample == null)
                    {
                        break;
                    }

                    dataStore.Datas[sampleIndex] = sample;
                }
            }

            endSampleIndex = sampleIndex - 1;
        }

        /// <summary>
        /// 縦棒グラフ用にサンプルを取得します。
        /// </summary>
        private static void FetchDataMinMax(
            CancellationToken ct,
            DataStore dataStore,
            ref long leftSampleIndex,
            ref long rightSampleIndex,
            GetSampleValueFunc getSampleValueFunc,
            GetSampleMinMaxFunc getSampleMinMaxFunc)
        {
            int zoomValue = ZoomRatioCalculator.CalcZoomValue(dataStore.ZoomRatioIndex);

            int leftDataIndex = ZoomRatioCalculator.CalcWidthFromPointCount(leftSampleIndex, zoomValue);
            int rightDataIndex = ZoomRatioCalculator.CalcWidthFromPointCount(rightSampleIndex, zoomValue);

            int dataIndex;
            for (dataIndex = leftDataIndex; dataIndex <= rightDataIndex; ++dataIndex)
            {
                if (ct.IsCancellationRequested)
                {
                    break;
                }

                if (!dataStore.Datas.ContainsKey(dataIndex))
                {
                    long left = ZoomRatioCalculator.CalcPointCountFromWidth(dataIndex, zoomValue);
                    long right = ZoomRatioCalculator.CalcPointCountFromWidth(dataIndex + 1, zoomValue) - 1;
                    Ensure.Operation.True(leftSampleIndex <= right && left <= rightSampleIndex);

                    float[] minmax;

                    if (getSampleMinMaxFunc != null)
                    {
                        minmax = getSampleMinMaxFunc(ct, left, right);
                    }
                    else
                    {
                        minmax = MySampleMinMaxFunc(ct, left, right, getSampleValueFunc);
                    }

                    if (minmax == null)
                    {
                        break;
                    }

                    dataStore.Datas[dataIndex] = minmax;
                }
            }

            rightSampleIndex = ZoomRatioCalculator.CalcPointCountFromWidth(dataIndex, zoomValue) - 1;
        }

        /// <summary>
        /// ユーザがGetSampleMinMaxFuncを提供しなかった場合に代わりに使います。
        /// </summary>
        private static float[] MySampleMinMaxFunc(
            CancellationToken ct,
            long beginIndex,
            long endIndex,
            GetSampleValueFunc getSampleValueFunc)
        {
            if (ct.IsCancellationRequested)
            {
                return null;
            }

            var sample = getSampleValueFunc(ct, beginIndex);
            if (sample == null)
            {
                return null;
            }

            if (sample.Length == 0)
            {
                return sample;
            }

            // 最初のサンプルのチャンネル数で結果のチャンネル数を決定します。
            var nbChannels = sample.Length;

            var minmax = new float[nbChannels * 2];
            for (int i = 0; i < nbChannels; ++i)
            {
                minmax[i] = minmax[i + nbChannels] = sample[i];
            }

            for (long sampleIndex = beginIndex + 1; sampleIndex <= endIndex; sampleIndex++)
            {
                if (ct.IsCancellationRequested)
                {
                    return null;
                }

                sample = getSampleValueFunc(ct, sampleIndex);
                if (sample == null)
                {
                    return null;
                }

                var num = Math.Min(nbChannels, sample.Length);
                for (int i = 0; i < num; ++i)
                {
                    minmax[i] = Math.Min(minmax[i], sample[i]);
                    minmax[i + nbChannels] = Math.Max(minmax[i + nbChannels], sample[i]);
                }
            }

            return minmax;
        }

        /// <summary>
        /// 通常の折れ線グラフを描画します。
        /// </summary>
        private static void DrawBasicGraph(
            Graphics g,
            Pen pen,
            int x,
            int y,
            int width,
            int height,
            int centerY,
            float scaleY,
            DataStore dataStore,
            int beginSamplePosX,
            long leftSampleIndex,
            long rightSampleIndex,
            int channel)
        {
            int zoomValue = ZoomRatioCalculator.CalcZoomValue(dataStore.ZoomRatioIndex);

            int pointCount = (int)(rightSampleIndex - leftSampleIndex + 1);

            PointF[] points = new PointF[pointCount];

            int pointIndex = 0;
            for (var sampleIndex = leftSampleIndex; sampleIndex <= rightSampleIndex; ++sampleIndex)
            {
                int posX = beginSamplePosX + ZoomRatioCalculator.CalcWidthFromPointCount(sampleIndex, zoomValue);

                float value = 0;

                float[] data;
                if (dataStore.Datas.TryGetValue(sampleIndex, out data))
                {
                    if (channel < data.Length)
                    {
                        value = data[channel];
                    }
                }

                points[pointIndex].X = x + posX;
                points[pointIndex].Y = Math.Min(y - value * scaleY + centerY, height);
                pointIndex++;
            }

            if (points.Length == 1)
            {
                g.DrawLine(pen, points[0], new PointF(points[0].X, points[0].Y));
            }
            else
            {
                g.DrawLines(pen, points);
            }
        }

        /// <summary>
        /// 縦棒グラフを描画します。
        /// </summary>
        private static void DrawMinMaxGraph(
            Graphics g,
            Pen pen,
            int x,
            int y,
            int width,
            int height,
            int centerY,
            float scaleY,
            DataStore dataStore,
            int beginSamplePosX,
            long leftSampleIndex,
            long rightSampleIndex,
            int channel)
        {
            int RangeHeight = height / 2;

            int zoomValue = ZoomRatioCalculator.CalcZoomValue(dataStore.ZoomRatioIndex);
            long leftDataIndex = ZoomRatioCalculator.CalcWidthFromPointCount(leftSampleIndex, zoomValue);
            long rightDataIndex = ZoomRatioCalculator.CalcWidthFromPointCount(rightSampleIndex, zoomValue);

            PointF[] point = new PointF[2];

            for (long dataIndex = leftDataIndex; dataIndex <= rightDataIndex; ++dataIndex)
            {
                float minValue = 0;
                float maxValue = 0;

                float[] minmax;
                if (dataStore.Datas.TryGetValue(dataIndex, out minmax))
                {
                    if (channel < minmax.Length / 2)
                    {
                        minValue = minmax[channel];
                        maxValue = minmax[minmax.Length / 2 + channel];
                    }
                }

                var posX = dataIndex + beginSamplePosX;

                point[0].X = x + posX;
                point[0].Y = Math.Min(y - (minValue * scaleY) + centerY, height - 1);
                point[1].X = x + posX;
                point[1].Y = Math.Min(y - (maxValue * scaleY) + centerY, height - 1);

                ++posX;

                // 一定の値のときに見えなくならないように。
                if (point[1].Y == point[0].Y)
                {
                    point[1].X += 1;
                }

                g.DrawLine(pen, point[0], point[1]);
            }
        }
    }
}
