﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.NwSoundSpyPlugin.Models;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public partial class ChannelMeterControl : UserControl
    {
        public List<VoiceInfoSpyModel.VoiceArray> VoiceArrayList { get; set; }
        private int _currentFrameIndex = 0;
        private int _selectedChannelIndex = -1;

        private const int ChannelCountMax = 96;

        public bool PauseFlag
        {
            get;
            set;
        }

        public ChannelMeterControl()
        {
            // this.InitializeComponent();
            DoubleBuffered = true;
            BackColor = Color.Black;
        }

        public void Reset()
        {
            VoiceArrayList = null;
            Invalidate();
        }

        private int CalcChannelWidth()
        {
            return ClientSize.Width / ChannelCountMax;
        }

        private int CalcPoxXFromChannelIndex(int channelIndex)
        {
            int channelWidth = CalcChannelWidth();

            return channelIndex * channelWidth;
        }

        private int CalcChannelIndexFromPosX(int x)
        {
            int channelWidth = CalcChannelWidth();
            if (channelWidth <= 0)
            {
                return 0;
            }

            int channelIndex = x / channelWidth;
            return channelIndex;
        }

        private void UpdateSelectedChannel(int channelIndex)
        {
            if (channelIndex < 0)
            {
                channelIndex = 0;
            }
            else if (channelIndex > ChannelCountMax - 1)
            {
                channelIndex = ChannelCountMax - 1;
            }
            _selectedChannelIndex = channelIndex;
        }

        protected override void OnMouseDown(MouseEventArgs e)
        {
            base.OnMouseDown(e);

            if (e.Button == MouseButtons.Left)
            {
                int channelIndex = CalcChannelIndexFromPosX(e.X);
                UpdateSelectedChannel(channelIndex);
                Invalidate();
            }
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            if (e.Button == MouseButtons.Left)
            {
                int channelIndex = CalcChannelIndexFromPosX(e.X);
                UpdateSelectedChannel(channelIndex);
                Invalidate();
            }
        }

        protected override void OnPaint(PaintEventArgs e)
        {
            const int NoPaintSize = 30;

            if (ClientSize.Width <= NoPaintSize || ClientSize.Height <= NoPaintSize)
            {
                return;
            }

            const int MarginBottom = 30;

            Graphics g = e.Graphics;
            Brush brNormal = new SolidBrush(Color.Yellow);
            Brush brSelect = new SolidBrush(Color.Red);

            if (VoiceArrayList != null)
            {
                int channelWidth = CalcChannelWidth();
                int height = Size.Height - MarginBottom;
                if (_selectedChannelIndex >= 0)
                {
                    int x = _selectedChannelIndex * channelWidth + channelWidth / 2;
                    g.DrawLine(Pens.DarkRed, x, 0, x, height);
                }

                if (VoiceArrayList.Count > 0)
                {
                    if (!PauseFlag)
                    {
                        _currentFrameIndex = VoiceArrayList.Count - 1;
                    }

                    g.DrawLine(Pens.Gray,
                        0, ClientSize.Height - 1 - MarginBottom,
                        ClientSize.Width - 1, ClientSize.Height - 1 - MarginBottom);

                    VoiceInfoSpyModel.VoiceArray voiceArray = VoiceArrayList[_currentFrameIndex];

                    int channelCount = 0;
                    int channelIndex = 0;
                    foreach (VoiceInfoSpyModel.VoiceInfo info in voiceArray.Item)
                    {
                        int h = 0;
                        if (info != null && info.Status != VoiceInfoSpyModel.VoiceStatus.Stop)
                        {
                            h = (int)(info.Volume * height) + 1;
                            channelCount++;
                        }

                        Brush br = (channelIndex == _selectedChannelIndex) ? brSelect : brNormal;
                        g.FillRectangle(br, channelIndex * channelWidth, height - h, channelWidth, h);

                        channelIndex++;
                    }

                    {
                        string str = string.Format("{0,3}", channelCount);
                        g.DrawString(str, this.Font, Brushes.White, 0, ClientSize.Height - 1 - MarginBottom + 6);
                        g.FillRectangle(
                            Brushes.YellowGreen,
                            30, ClientSize.Height - 1 - MarginBottom + 10,
                            (ClientSize.Width - 30) * channelCount / ChannelCountMax, 10);
                    }

                    if (_selectedChannelIndex >= 0)
                    {
                        VoiceInfoSpyModel.VoiceInfo info = voiceArray.Item[_selectedChannelIndex];

                        int h = this.Font.Height;
                        int y = 0;
                        g.DrawString("index :", this.Font, Brushes.White, 0, y);
                        y += h;
                        g.DrawString("volume :", this.Font, Brushes.White, 0, y);
                        y += h;
                        g.DrawString("pitch :", this.Font, Brushes.White, 0, y);
                        y += h;
                        g.DrawString("main L :", this.Font, Brushes.White, 0, y);
                        y += h;
                        g.DrawString("main R :", this.Font, Brushes.White, 0, y);
                        y += h;
                        g.DrawString("drc L  :", this.Font, Brushes.White, 0, y);
                        y += h;
                        g.DrawString("drc R  :", this.Font, Brushes.White, 0, y);
                        y += h;

                        if (info != null && info.Status != VoiceInfoSpyModel.VoiceStatus.Stop)
                        {
                            int x = 50;
                            y = 0;
                            string str = string.Format("{0,6}", _selectedChannelIndex);
                            g.DrawString(str, this.Font, Brushes.White, x, y);
                            y += h;

                            str = string.Format("{0,6:F2}", info.Volume);
                            g.DrawString(str, this.Font, Brushes.White, x, y);
                            y += h;

                            str = string.Format("{0,6:F2}", info.Pitch);
                            g.DrawString(str, this.Font, Brushes.White, x, y);
                            y += h;

                            str = string.Format("{0,6:F2}", info.MainMixLeft);
                            g.DrawString(str, this.Font, Brushes.White, x, y);
                            y += h;

                            str = string.Format("{0,6:F2}", info.MainMixRight);
                            g.DrawString(str, this.Font, Brushes.White, x, y);
                            y += h;

                            str = string.Format("{0,6:F2}", info.DrcMixLeft);
                            g.DrawString(str, this.Font, Brushes.White, x, y);
                            y += h;

                            str = string.Format("{0,6:F2}", info.DrcMixRight);
                            g.DrawString(str, this.Font, Brushes.White, x, y);
                            y += h;
                        }
                    }
                }
            }
        }

        public void SetCurrentIndex(int profileIndex)
        {
            if (VoiceArrayList != null && VoiceArrayList.Count > 0)
            {
                _currentFrameIndex = profileIndex;
                if (_currentFrameIndex > VoiceArrayList.Count - 1)
                {
                    _currentFrameIndex = VoiceArrayList.Count - 1;
                }

                Invalidate();
            }
        }
    }
}
