﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System;
using System.Linq;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class WaveformPanelViewModelMock : WaveformPanelViewModel
    {
        private readonly WaveformSpyModelMock _modelMock = new WaveformSpyModelMock();
        private readonly WaveformCache _waveCache = new WaveformCache();

        public WaveformPanelViewModelMock()
        {
            this.Items.Add(new WaveformChannelViewModel() { Name = "F-L", ChannelNumber = 0, });
            this.Items.Add(new WaveformChannelViewModel() { Name = "F-R", ChannelNumber = 1, });
            this.Items.Add(new WaveformChannelViewModel() { Name = "R-L", ChannelNumber = 2, });
            this.Items.Add(new WaveformChannelViewModel() { Name = "R-R", ChannelNumber = 3, });
            this.Items.Add(new WaveformChannelViewModel() { Name = "F-C", ChannelNumber = 4, });
            this.Items.Add(new WaveformChannelViewModel() { Name = "LFE", ChannelNumber = 5, });

            this.Items.ForEach(it => it.EnumerateLineChartSamples = _waveCache.EnumerateSamples);

            _waveCache.WaveProvider = _modelMock;
            _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainFrontLeft);
            _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainFrontRight);
            _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainRearLeft);
            _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainRearRight);
            _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainFrontCenter);
            _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainLfe);

            _modelMock.UpdateEvent += (s, e) =>
            {
                _waveCache.SourceUpdated();
            };
        }
    }

    public class WaveformSpyModelMock : ObservableObject, WaveformCache.IWaveProvider
    {
        private const int MaxCacheCount = 500;
        private const int SampleRate = 48000;
        private const int SamplePerFrame = 144;
        private const double Duration = (double)SamplePerFrame / SampleRate;

        public const int BitsPerSample = 16;
        public const int ChannelCountMax = 8;

        private SpyGlobalTime _beginTime = SpyGlobalTime.InvalidValue;
        private SpyGlobalTime _endTime = SpyGlobalTime.InvalidValue;
        private readonly CacheDictionary<int, WaveformSpyModel.Waveform> _cache = new CacheDictionary<int, WaveformSpyModel.Waveform>(MaxCacheCount);

        public event EventHandler UpdateEvent;

        public SpyGlobalTime BeginTime
        {
            get { return _beginTime; }
            private set { this.SetPropertyValue(ref _beginTime, value); }
        }

        public SpyGlobalTime EndTime
        {
            get { return _endTime; }
            private set
            {
                if (this.SetPropertyValue(ref _endTime, value))
                {
                    this.UpdateEvent?.Invoke(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// 受信した波形データの総フレーム数です。
        /// </summary>
        public int WaveformCount => 5000;

        public WaveformSpyModelMock()
        {
        }

        /// <summary>
        /// 指定時間を区間に含む波形データのインデックスを取得します。
        /// </summary>
        /// <param name="time"></param>
        /// <returns>
        /// time を区間に含む波形データがあるときは、そのインデックスを返します。
        /// time を区間に含む波形データが無いときは、
        /// それ以後の最初の波形データのインデックスの１の補数(負の値)を返します。
        /// time 以後の波形データが無いときは、WaveformCount プロパティの１の補数(負の値)を返します。
        /// </returns>
        public int GetWaveformIndex(SpyGlobalTime time)
        {
            int waveformIndex = (int)Math.Floor(time.Seconds / Duration);

            if (waveformIndex < 0)
            {
                return ~0;
            }
            else if (waveformIndex >= this.WaveformCount)
            {
                return ~this.WaveformCount;
            }
            else
            {
                return waveformIndex;
            }
        }

        /// <summary>
        /// 指定したインデックスの波形データを返します。
        /// </summary>
        /// <param name="waveformIndex"></param>
        /// <returns></returns>
        public WaveformSpyModel.Waveform GetWaveform(int waveformIndex)
        {
            Ensure.Argument.Range(waveformIndex, 0, this.WaveformCount);

            WaveformSpyModel.Waveform value = null;
            if (_cache.TryGetValue(waveformIndex, out value))
            {
                return value;
            }
            else
            {
                value = MakeWaveform(waveformIndex);
                _cache.Add(waveformIndex, value);
                return value;
            }
        }

        private WaveformSpyModel.Waveform MakeWaveform(int waveformIndex)
        {
            double time = waveformIndex * Duration;

            var waveform = new WaveformSpyModel.Waveform(SampleRate, SamplePerFrame);

            var secPerSample = 1.0 / SampleRate;

            waveform.Time = SpyGlobalTime.FromSeconds(time);
            waveform.WaveformIndex = waveformIndex;
            waveform.SampleIndex = waveformIndex * SamplePerFrame;

            waveform.SampleData.Samples.ForEach((samples, channel) =>
            {
                for (int i = 0; i < waveform.SampleCount; i++)
                {
                    var t = 200 * (time + i * secPerSample);
                    var value = short.MaxValue * Math.Sin(t);
                    samples[i] = (short)value;
                }
            });

            return waveform;
        }
    }
}
