﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Windows.Controls;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Windows.Media;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class WaveformChannelViewModel : ObservableObject, ILineChartItemData
    {
        private const double MaxScaleX = 20; // これより上がると横スクロールできなくなる。

        private string _name = string.Empty;
        private double _currentX = 0.0;
        private double _minimumX = 0.0;
        private double _maximumX = 0.0;
        private double _originX = 0.0;
        private double _scaleX = 0.001;
        private double _scaleY = 1.0;
        private bool _visible = true;
        private double _height = 50.0;

        public delegate IEnumerable<LineChartSample> EnumerateLineChartSamplesFunc(double start, double end, double scale, int sampleIndex);

        public delegate double EvaluateLineChartValueFunc(double x, int channelIndex);

        public EnumerateLineChartSamplesFunc EnumerateLineChartSamples { get; set; }

        public EvaluateLineChartValueFunc EvaluateLineChartValue { get; set; }

        public event EventHandler SamplesChanged;

        public WaveformPanelViewModel Owner { get; set; }

        public string Name
        {
            get { return _name; }
            set { this.SetPropertyValue(ref _name, value); }
        }

        public double CurrentX
        {
            get { return _currentX; }
            set
            {
                this.SetCurrentXImpl(value, from: this);
            }
        }

        public double MaximumX
        {
            get { return _maximumX; }
            set
            {
                if (this.SetPropertyValue(ref _maximumX, value))
                {
                    this.NotifySamplesChanged();
                }
            }
        }

        public double MinimumX
        {
            get { return _minimumX; }
            set
            {
                if (this.SetPropertyValue(ref _minimumX, value))
                {
                    this.NotifySamplesChanged();
                }
            }
        }

        public double OriginX
        {
            get { return _originX; }
            set { this.SetOriginXImpl(value, from: this); }
        }

        public double ScaleX
        {
            get { return _scaleX; }
            set { this.SetScaleXImpl(value, from: this); }
        }

        public double ScaleY
        {
            get { return _scaleY; }
            set { this.SetScaleYImpl(value, from: this); }
        }

        public int ChannelNumber { get; set; }

        public ObservableCollection<ILineChartItemData> Items { get; } =
            new ObservableCollection<ILineChartItemData>();

        public bool IsVisible
        {
            get { return _visible; }
            set { this.SetPropertyValue(ref _visible, value); }
        }

        public LineChartDrawType DrawType
        {
            get { return LineChartDrawType.Line; }
        }

        public Color Color
        {
            get { return Colors.White; }
        }

        public LineChartInterpolation Interpolation
        {
            get { return LineChartInterpolation.None; }
        }

        public double MinimumY
        {
            get { return short.MinValue; }
        }

        public double MaximumY
        {
            get { return short.MaxValue; }
        }

        public double CenterY
        {
            get { return 0.0; }
        }

        public double Height
        {
            get { return _height; }
            set { SetPropertyValue(ref _height, value); }
        }

        public WaveformChannelViewModel()
        {
            this.EnumerateLineChartSamples = this.EnumerateLineChartSamplesDummy;
            this.EvaluateLineChartValue = this.EvaluateLineChartValueDummy;

            this.Items.Add(this);
        }

        public void SetCurrentXFromOwner(double value)
        {
            this.SetCurrentXImpl(value, from: this.Owner);
        }

        private void SetCurrentXImpl(double value, object from)
        {
            if (from == this)
            {
                value = MathUtility.Clamp(
                    Math.Floor(value), this.MinimumX, this.MaximumX);
            }

            if (this.SetPropertyValue(ref _currentX, value, nameof(CurrentX)))
            {
                if (this.Owner != from)
                {
                    this.Owner?.SetCurrentXFromChild(value, from: this);
                }
            }
        }

        public void SetOriginXFromOwner(double value)
        {
            this.SetOriginXImpl(value, from: this.Owner);
        }

        private void SetOriginXImpl(double value, object from)
        {
            if (this.SetPropertyValue(ref _originX, value, nameof(OriginX)))
            {
                if (this.Owner != from)
                {
                    this.Owner?.SetOriginXFromChild(value, from: this);
                }
            }
        }

        public void SetScaleXFromOwner(double value)
        {
            this.SetScaleXImpl(value, from: this.Owner);
        }

        private void SetScaleXImpl(double value, object from)
        {
            if (from == this)
            {
                value = Math.Min(value, MaxScaleX);
            }

            if (this.SetPropertyValue(ref _scaleX, value, nameof(ScaleX)))
            {
                if (this.Owner != from)
                {
                    this.Owner?.SetScaleXFromChild(value, from: this);
                }
            }
        }

        public void SetScaleYFromOwner(double value)
        {
            this.SetScaleYImpl(value, from: this.Owner);
        }

        private void SetScaleYImpl(double value, object from)
        {
            if (this.SetPropertyValue(ref _scaleY, value, nameof(ScaleY)))
            {
                if (this.Owner != from)
                {
                    this.Owner?.SetScaleYFromChild(value, from: this);
                }
            }
        }

        public void NotifySamplesChanged()
        {
            this.SamplesChanged?.Invoke(this, EventArgs.Empty);
        }

        IEnumerable<LineChartSample> ILineChartItemData.EnumerateSamples(double beginX, double endX, double scale)
        {
            return this.EnumerateLineChartSamples(beginX, endX, scale, this.ChannelNumber);
        }

        double ILineChartItemData.Evaluate(double x)
        {
            return this.EvaluateLineChartValue(x, this.ChannelNumber);
        }

        private IEnumerable<LineChartSample> EnumerateLineChartSamplesDummy(double start, double end, double scale, int sampleIndex)
        {
            yield break;
        }

        private double EvaluateLineChartValueDummy(double x, int channelIndex)
        {
            return double.NaN;
        }
    }
}
