﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.Spy;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Windows.Data;
using System.Windows.Input;
using System.Windows.Markup;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    [ContentProperty("Items")]
    public class StartSoundHistoryPanelViewModel : ObservableObject
    {
        private readonly object _lockObject = new object();
        private AtkSpyModel _model;
        private readonly ObservableCollection<StartSoundHistoryItemViewModel> _items = new ObservableCollection<StartSoundHistoryItemViewModel>();
        private readonly Dictionary<uint, StartSoundHistoryItemViewModel> _activeItems = new Dictionary<uint, StartSoundHistoryItemViewModel>();
        private SpyGlobalTime _lastTimestamp;
        private ICollectionView _viewItems;
        private StartSoundHistoryItemViewModel _selectedItem;
        private readonly StringFilter _nameFilter = new StringFilter();
        private string _nameFilterString = string.Empty;
        private SpyTimeUnit _timeUnit;
        private bool _showColumnPlayerName = true;
        private bool _showColumnVolume = true;
        private bool _showColumnPriority = true;
        private bool _showColumnSoundType = true;
        private bool _showColumnArchiveName = true;
        private bool _showAddonSoundArchiveInfo = false;

        public StartSoundHistoryPanelViewModel()
        {
        }

        protected override void DisposeManagedInstance()
        {
            _items.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public ObservableCollection<StartSoundHistoryItemViewModel> Items
        {
            get
            {
                return _items;
            }
        }

        public ICollectionView ViewItems
        {
            get
            {
                if (_viewItems == null)
                {
                    lock (_lockObject)
                    {
                        if (_viewItems == null)
                        {
                            this.SetupViewItems(this.Items);
                        }
                    }
                }

                return _viewItems;
            }
        }

        public StartSoundHistoryItemViewModel SelectedItem
        {
            get
            {
                return _selectedItem;
            }
            set
            {
                this.SetPropertyValue(ref _selectedItem, value);
            }
        }

        public string NameFilter
        {
            get
            {
                return _nameFilterString;
            }
            set
            {
                if (this.SetPropertyValue(ref _nameFilterString, value))
                {
                    _nameFilter.Filter = _nameFilterString;
                    this.UpdateItemsVisibility();
                }
            }
        }

        public SpyTimeUnit TimeUnit
        {
            get
            {
                return _timeUnit;
            }

            set
            {
                this.SetPropertyValue(ref _timeUnit, value);
            }
        }

        public bool ShowColumnPlayerName
        {
            get
            {
                return _showColumnPlayerName;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnPlayerName, value);
            }
        }

        public bool ShowColumnVolume
        {
            get
            {
                return _showColumnVolume;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnVolume, value);
            }
        }

        public bool ShowColumnPriority
        {
            get
            {
                return _showColumnPriority;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnPriority, value);
            }
        }

        public bool ShowColumnSoundType
        {
            get
            {
                return _showColumnSoundType;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnSoundType, value);
            }
        }

        public bool ShowColumnArchiveName
        {
            get
            {
                return _showColumnArchiveName && this.ShowAddonSoundArchiveInfo;
            }
            set
            {
                if (this.ShowAddonSoundArchiveInfo)
                {
                    this.SetPropertyValue(ref _showColumnArchiveName, value);
                }
            }
        }

        /// <summary>
        /// 追加サウンドアーカイブは非公開機能なので、
        /// 実際に追加サウンドアーカイブが使われているときにだけ情報を表示します。
        /// </summary>
        public bool ShowAddonSoundArchiveInfo
        {
            get
            {
                return _showAddonSoundArchiveInfo;
            }
            set
            {
                if (this.SetPropertyValue(ref _showAddonSoundArchiveInfo, value))
                {
                    this.NotifyPropertyChanged(nameof(ShowColumnArchiveName));
                }
            }
        }

        public ICommand ListSortedCommand
        {
            get; set;
        }

        private void UpdateItemsVisibility()
        {
            foreach (var item in this.Items)
            {
                item.IsVisible = this.ItemFilter(item);
            }
        }

        private bool ItemFilter(StartSoundHistoryItemViewModel item)
        {
            bool match = true;

            // グループ間はAND, グループ内はOR

            if (match == true)
            {
                // 名前フィルタ
                match = false;

                if (!match && _nameFilter.Execute(item.SoundName))
                {
                    match = true;
                }

                if (!match && _nameFilter.Execute(item.PlayerName))
                {
                    match = true;
                }
            }

#if false // フィルタできるプロパティが追加されたときのためのサンプルコード
            if (match == true)
            {
                // 種別グループ
                match = false;

                if (!match && _showOneshot && item.PlayKind == "Oneshot")
                {
                    match = true;
                }

                if (!match && _showLoop && item.PlayKind == "Loop")
                {
                    match = true;
                }
            }
#endif

            return match;
        }

        /// <summary>
        /// モデルをプロパティと関連付けます。
        /// </summary>
        internal void AttachModel(AtkSpyModel model)
        {
            if (_model == model)
            {
                return;
            }

            if (_model != null)
            {
                _model.SoundStateChanged -= HandleSoundStateChanged;
            }

            _model = model;

            this.Clear();

            if (_model != null)
            {
                _model.SoundStateChanged += HandleSoundStateChanged;
                _model.ReadAllSoundStates();
            }
        }

        private void Clear()
        {
            _items.ForEach(it => it.Dispose());
            _items.Clear();
            _activeItems.Clear();
            _lastTimestamp = SpyGlobalTime.Zero;
            this.ShowAddonSoundArchiveInfo = false;
        }

        private void HandleSoundStateChanged(object sender, AtkSpyModel.SoundStateChangedEventArgs args)
        {
            if (args.Timestamp <= _lastTimestamp)
            {
                this.Clear();
            }

            _lastTimestamp = args.Timestamp;

            var time = new SpyTime(args.Timestamp, args.BelongingFrame);

            // states に新しく現れたアイテムは再生を開始したことを表します。
            // states から消えたアイテムは停止したことを表します。

            var stoppedSounds = _activeItems.Where(item => args.SoundStates.FirstOrDefault(state => state.InstanceId == item.Key) == null).ToArray();
            foreach (var item in stoppedSounds)
            {
                _activeItems.Remove(item.Key);
            }

            var newSounds = args.SoundStates.Where(state => !_activeItems.ContainsKey(state.InstanceId));
            foreach (var state in newSounds)
            {
                var item = CreateItem(state);

                item.Time = time;
                item.IsVisible = this.ItemFilter(item);

                _activeItems.Add(state.InstanceId, item);
                this.Items.Add(item);
            }
        }

        private StartSoundHistoryItemViewModel CreateItem(AtkSpyModel.SoundState state)
        {
            var item = new StartSoundHistoryItemViewModel()
            {
                SoundId = state.SoundId,
            };

            var soundData = state.SoundData;
            item.SoundName = Map(soundData?.Label, s => string.IsNullOrEmpty(s) ? state.SoundId.ToString() : s);
            if (soundData != null)
            {
                item.Volume = soundData.Volume;
                item.Priority = soundData.PlayerPriority;
                item.SoundType = soundData.SoundType.ToString();
                item.PlayerName = Map(soundData.PlayerData?.Label, s => string.IsNullOrEmpty(s) ? soundData.PlayerId.ToString() : s);
                item.ArchiveName = soundData.SoundArchiveData.Name;

                if (soundData.SoundArchiveData.IsAddon)
                {
                    this.ShowAddonSoundArchiveInfo = true;
                }
            }

            return item;
        }

        private U Map<T, U>(T value, Func<T, U> map)
        {
            return map(value);
        }

        /// <summary>
        /// Itemsプロパティにグループおよびデフォルトのソートを設定します。
        /// </summary>
        /// <param name="source"></param>
        private void SetupViewItems(object source)
        {
            var viewItems = CollectionViewSource.GetDefaultView(source);

            if (viewItems != null)
            {
                viewItems.SortDescriptions.Add(new SortDescription("Time.Timestamp", ListSortDirection.Descending));
                viewItems.GroupDescriptions.Add(new PropertyGroupDescription("Time.Timestamp"));
            }

            _viewItems = viewItems;
        }
    }
}
