﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using System;
using System.Windows.Threading;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    internal class LoudnessMeterPeakHold : DisposableObject
    {
        private double _minimum = double.NaN;
        private double _current = double.NaN;
        private double _hold = double.NaN;
        private DateTime _holdStartTime;
        private DispatcherTimer _timer = new DispatcherTimer();

        /// <summary>
        /// ホールド値が減衰を開始するまでの遅延時間です。
        /// </summary>
        public TimeSpan HoldPeriod { get; set; } = TimeSpan.FromMilliseconds(500);

        /// <summary>
        /// ホールド値が減衰する時間間隔です。
        /// </summary>
        public TimeSpan DecayInterval { get; set; } = TimeSpan.FromMilliseconds(10);

        /// <summary>
        /// ホールド値の一回当たりの減衰量です。
        /// </summary>
        public double DecayStep { get; set; } = 0.5;

        /// <summary>
        /// ホールド値の取りうる最小値です。
        /// 有効な値が設定されるまで、ピークホールドは機能しません。
        /// </summary>
        public double Minimum
        {
            get { return _minimum; }
            set
            {
                if (_minimum != value)
                {
                    _minimum = value;

                    if (double.IsNaN(value) || double.IsInfinity(value))
                    {
                        _timer.IsEnabled = false;
                    }
                    else
                    {
                        this.SetupPeakHold();
                    }
                }
            }
        }

        /// <summary>
        /// 現在値です。
        /// </summary>
        public double Current
        {
            get { return _current; }
            set
            {
                if (_current != value)
                {
                    _current = value;

                    if (double.IsNaN(value) || double.IsPositiveInfinity(value))
                    {
                        // Current が無効値なのでホールド値も無効にします。
                        _timer.IsEnabled = false;
                        this.SetHoldValue(value);
                    }
                    else if (double.IsNegativeInfinity(value))
                    {
                        // ピークホールド中なら Minimum に達するまで減衰を続けます。
                        // ピークホールドしていないなら、ただちにホールド値を設定します。
                        if (!_timer.IsEnabled)
                        {
                            this.SetHoldValue(value);
                        }
                    }
                    else
                    {
                        this.SetupPeakHold();
                    }
                }
            }
        }

        /// <summary>
        /// ホールド値です。
        /// </summary>
        public double Hold
        {
            get { return Math.Max(this.CalcHoldValue(), Math.Max(this.Current, this.Minimum)); }
        }

        /// <summary>
        /// 任意のデータを保持します。
        /// </summary>
        public object Tag { get; set; }

        /// <summary>
        /// ホールド値の減衰中、定期的に発火します。
        /// </summary>
        public event EventHandler HoldChanged;

        public LoudnessMeterPeakHold()
        {
            _timer.Tick += OnTimer;
        }

        protected override void DisposeManagedInstance()
        {
            _timer.IsEnabled = false;
        }

        /// <summary>
        /// 必要な情報がそろったらピークホールドの動作を開始します。
        /// </summary>
        private void SetupPeakHold()
        {
            if (double.IsNaN(this.Minimum) || double.IsInfinity(this.Minimum))
            {
                return;
            }

            if (double.IsNaN(this.Current) || double.IsPositiveInfinity(this.Current))
            {
                return;
            }

            var hold = this.CalcHoldValue();
            var limit = Math.Max(this.Current, this.Minimum);
            if (double.IsNaN(hold) || double.IsInfinity(hold) || limit > hold)
            {
                this.SetHoldValue(limit);
                _timer.IsEnabled = false;
            }
            else if (hold > this.Current)
            {
                if (!_timer.IsEnabled)
                {
                    _holdStartTime = DateTime.Now;
                    _timer.Interval = this.HoldPeriod;
                    _timer.IsEnabled = true;
                }
            }
        }

        private void SetHoldValue(double value)
        {
            if (_hold != value)
            {
                _hold = value;
                this.NotifyHoldChanged();
            }
        }

        private double CalcHoldValue()
        {
            if (!_timer.IsEnabled)
            {
                return _hold;
            }

            var timeSpan = DateTime.Now - _holdStartTime;

            if (timeSpan <= this.HoldPeriod)
            {
                return _hold;
            }

            timeSpan -= this.HoldPeriod;

            var decaySycle = Math.Ceiling(timeSpan.TotalSeconds / this.DecayInterval.TotalSeconds);

            var value = _hold - decaySycle * this.DecayStep;

            return value;
        }

        private void OnTimer(object obj, EventArgs args)
        {
            var limit = Math.Max(this.Current, this.Minimum);
            if (limit >= this.CalcHoldValue())
            {
                _timer.IsEnabled = false;
                this.SetHoldValue(limit);
            }
            else
            {
                this.NotifyHoldChanged();

                // 次の減衰時刻までのインターバルを求めます。
                var decayTime = DateTime.Now - _holdStartTime - this.HoldPeriod;
                var interval = TimeSpan.FromSeconds(
                    this.DecayInterval.TotalSeconds - Math.IEEERemainder(
                        decayTime.TotalSeconds, this.DecayInterval.TotalSeconds));

                _timer.Interval = interval;
            }
        }

        private void NotifyHoldChanged()
        {
            this.HoldChanged?.Invoke(this, EventArgs.Empty);
        }
    }
}
