﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.Communications;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading;

namespace NintendoWare.CafeSpyPlugin.Foundation.Communications
{
    /// <summary>
    /// Cafe 向けの HostIO 機能を提供します。
    /// </summary>
    public sealed class HostIOCafe : IComEndPoint
    {
        public static readonly string SyncPortName = "NWSoundSpy_Sync";
        public static readonly string DataPortName = "NWSoundSpy_Data";

        private static readonly Lazy<HostIOCafe> FieldInstance = new Lazy<HostIOCafe>(() => new HostIOCafe());

        private static readonly object FieldConnectLock = new object();

        private readonly Dictionary<string, HostIOCafeChannelInfo> _channelInfos =
            new Dictionary<string, HostIOCafeChannelInfo>();

        private int _connectCount = 0;
        private HostIOCafePortMappingChannel _portMappingChannel;

        private SynchronizationContext _syncContext;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private HostIOCafe()
        {
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// インスタンスを取得します。
        /// </summary>
        public static IComEndPoint Instance
        {
            get { return HostIOCafe.FieldInstance.Value; }
        }

        /// <summary>
        /// 接続の有無を取得します。
        /// </summary>
        public bool IsConnected
        {
            get
            {
                return _connectCount > 0 &&
                    _portMappingChannel != null &&
                    _portMappingChannel.IsOpened;
            }
        }

        /// <summary>
        /// エンディアンを取得します。
        /// </summary>
        public bool IsLittleEndian
        {
            get { return false; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 接続を開始します。
        /// </summary>
        /// <param name="syncContext">同期コンテキストを指定します。</param>
        public void Connect(SynchronizationContext syncContext)
        {
            Ensure.Argument.NotNull(syncContext);

            lock (FieldConnectLock)
            {
                if (_portMappingChannel != null)
                {
                    return;
                }

                _syncContext = syncContext;

                try
                {
                    _portMappingChannel = new HostIOCafePortMappingChannel();

                    _portMappingChannel.ChannelOpend += (sender, e) =>
                    {
                        var context = _syncContext;
                        if (context != null)
                        {
                            context.Post(state => this.OnChannelOpened(((HostIOCafeChannelEventArgs)state).ChannelInfo), e);
                        }
                    };

                    _portMappingChannel.ChannelClosed += (sender, e) =>
                    {
                        var context = _syncContext;
                        if (context != null)
                        {
                            context.Post(state => this.OnChannelClosed(((HostIOCafeChannelEventArgs)state).ChannelInfo), e);
                        }
                    };

                    _portMappingChannel.Open();
                }
                catch
                {
                    try
                    {
                        this.Disconnect();
                    }
                    catch
                    {
                    }

                    throw;
                }

                _connectCount++;
            }
        }

        /// <summary>
        /// 切断します。
        /// </summary>
        public void Disconnect()
        {
            lock (FieldConnectLock)
            {
                if (_portMappingChannel == null)
                {
                    return;
                }

                if (_connectCount > 0)
                {
                    _connectCount--;

                    if (_connectCount > 0)
                    {
                        return;
                    }
                }

                Assertion.Operation.NotNull(_portMappingChannel);

                _portMappingChannel.Close();
                _portMappingChannel = null;

                _channelInfos.Clear();
            }
        }

        public IComChannel OpenChannel(string portName)
        {
            lock (FieldConnectLock)
            {
                HostIOCafeChannelInfo channelInfo;

                if (!_channelInfos.TryGetValue(portName, out channelInfo))
                {
                    return null;
                }

                return HostIOCafeChannel.Open(channelInfo.Port, portName);
            }
        }

        IComChannel IComEndPoint.OpenChannel(object port)
        {
            Ensure.Argument.True(port is string);
            return this.OpenChannel(port as string);
        }

        private void OnChannelOpened(HostIOCafeChannelInfo channelInfo)
        {
            Debug.WriteLine(string.Format("[HostIOCafe] ChannelOpened : Name={0}, Port={1}.", channelInfo.Name, channelInfo.Port));

            if (!this.IsConnected)
            {
                return;
            }

            if (_channelInfos.ContainsKey(channelInfo.Name))
            {
                return;
            }

            _channelInfos.Add(channelInfo.Name, channelInfo);
        }

        private void OnChannelClosed(HostIOCafeChannelInfo channelInfo)
        {
            Debug.WriteLine(string.Format("[HostIOCafe] ChannelClosed : Name={0}, Port={1}.", channelInfo.Name, channelInfo.Port));

            _channelInfos.Remove(channelInfo.Name);
        }
    }
}
