﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Foundation.Communications;
using System;
using System.Threading;

namespace NintendoWare.CafeSpyPlugin.Communications
{
    /// <summary>
    /// Win(NW4F) 向けの HostIO 機能を提供します。
    /// </summary>
    public sealed class HostIOWinsock : IComEndPoint
    {
        public static readonly int SyncPort = 65100;
        public static readonly int DataPort = 65101;

        private static readonly Lazy<HostIOWinsock> FieldInstance = new Lazy<HostIOWinsock>(() => new HostIOWinsock());

        private readonly object _connectLock = new object();

        private uint _connectCount = 0;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        private HostIOWinsock()
        {
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// インスタンスを取得します。
        /// </summary>
        public static IComEndPoint Instance
        {
            get { return HostIOWinsock.FieldInstance.Value; }
        }

        /// <summary>
        /// 接続の有無を取得します。
        /// </summary>
        public bool IsConnected
        {
            get { return _connectCount > 0; }
        }

        /// <summary>
        /// エンディアンを取得します。
        /// </summary>
        public bool IsLittleEndian
        {
            get { return true; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 接続を開始します。
        /// </summary>
        /// <param name="syncContext">同期コンテキストを指定します。</param>
        public void Connect(SynchronizationContext syncContext)
        {
            lock (_connectLock)
            {
                // HostIOWinChannel.Open() に集約されているので、実装不要。
                _connectCount++;
            }
        }

        /// <summary>
        /// 切断します。
        /// </summary>
        public void Disconnect()
        {
            lock (_connectLock)
            {
                if (_connectCount == 0)
                {
                    return;
                }

                // HostIOWinChannel.Close() に集約されているので、実装不要。
                _connectCount--;
            }
        }

        public IComChannel OpenChannel(int port)
        {
            return HostIOWinsockChannel.Open(port, string.Empty);
        }

        IComChannel IComEndPoint.OpenChannel(object port)
        {
            Ensure.Argument.True(port is int);
            return this.OpenChannel((int)port);
        }
    }
}
