﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Globalization;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Shapes;
using System.Windows.Interop;
using System.Runtime.InteropServices;

namespace Nintendo.AudioToolkit.Windows.Controls
{
    [TemplatePart(Name = pointerPart, Type = typeof(Line))]
    public class RulerControl : Control
    {
        private const string pointerPart = "pointerPart";

        private Line pointerLine = null;

        private Point beginPoint;
        private bool tracking = false;

        public double Position
        {
            get { return (double)GetValue(PositionProperty); }
            set { SetValue(PositionProperty, value); }
        }

        public static readonly DependencyProperty PositionProperty =
            DependencyProperty.Register("Position", typeof(double), typeof(RulerControl),
            new FrameworkPropertyMetadata(0.0, new PropertyChangedCallback(OnPositionChanged)));

        private static void OnPositionChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var control = d as RulerControl;
            control.InvalidateVisual();
        }

        public double Scale
        {
            get { return (double)GetValue(ScaleProperty); }
            set { SetValue(ScaleProperty, value); }
        }

        public static readonly DependencyProperty ScaleProperty =
            DependencyProperty.Register("Scale", typeof(double), typeof(RulerControl),
            new FrameworkPropertyMetadata(1.0, new PropertyChangedCallback(OnScaleChanged)));

        private static void OnScaleChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var control = d as RulerControl;
            control.UpdateTick();
            control.InvalidateVisual();
        }

        public Point PointerPosition
        {
            get { return (Point)GetValue(PointerPositionProperty); }
            set { SetValue(PointerPositionProperty, value); }
        }

        public static readonly DependencyProperty PointerPositionProperty =
            DependencyProperty.Register("PointerPosition", typeof(Point), typeof(RulerControl),
            new FrameworkPropertyMetadata(new Point(0.0, 0.0), new PropertyChangedCallback(OnPointerPositionChanged)));

        private static void OnPointerPositionChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            RulerControl control = d as RulerControl;
            control.PointerX = control.PointerPosition.X - control.Position;
        }

        public double PointerX
        {
            get { return (double)GetValue(PointerXProperty); }
            set
            {
                if (double.IsNaN(value) == false)
                {
                    SetValue(PointerXProperty, value);
                    this.pointerLine.Visibility = Visibility.Visible;

                }
                else
                {
                    this.SetValue(PointerXProperty, 0.0);
                    this.pointerLine.Visibility = Visibility.Collapsed;
                }
            }
        }

        public static readonly DependencyProperty PointerXProperty =
            DependencyProperty.Register("PointerX", typeof(double), typeof(RulerControl),
            new FrameworkPropertyMetadata(0.0));

        public double MinorTick
        {
            get { return (double)GetValue(MinorTickProperty); }
            set { SetValue(MinorTickProperty, value); }
        }

        public static readonly DependencyProperty MinorTickProperty =
            DependencyProperty.Register("MinorTick", typeof(double), typeof(RulerControl),
            new FrameworkPropertyMetadata(100.0));

        public double MinimumSpanMajorTick
        {
            get { return (double)GetValue(MinimumSpanMajorTickProperty); }
            set { SetValue(MinimumSpanMajorTickProperty, value); }
        }

        public static readonly DependencyProperty MinimumSpanMajorTickProperty =
            DependencyProperty.Register("MinimumSpanMajorTick", typeof(double), typeof(RulerControl),
            new FrameworkPropertyMetadata(1.0, OnMinimumSpanMajorTickChanged));

        private static void OnMinimumSpanMajorTickChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var control = d as RulerControl;
            control.UpdateTick();
            control.InvalidateVisual();
        }

        public double MajorTick
        {
            get { return (double)GetValue(MajorTickProperty); }
            set { SetValue(MajorTickProperty, value); }
        }

        public static readonly DependencyProperty MajorTickProperty =
            DependencyProperty.Register("MajorTick", typeof(double), typeof(RulerControl),
            new FrameworkPropertyMetadata(1000.0));

        public ILabelProvider MinorLabelProvider
        {
            get { return (ILabelProvider)GetValue(MinorLabelProviderProperty); }
            set { SetValue(MinorLabelProviderProperty, value); }
        }

        public static readonly DependencyProperty MinorLabelProviderProperty =
            DependencyProperty.Register("MinorLabelProvider", typeof(ILabelProvider), typeof(RulerControl),
            new FrameworkPropertyMetadata());

        public ILabelProvider MajorLabelProvider
        {
            get { return (ILabelProvider)GetValue(MajorLabelProviderProperty); }
            set { SetValue(MajorLabelProviderProperty, value); }
        }

        public static readonly DependencyProperty MajorLabelProviderProperty =
            DependencyProperty.Register("MajorLabelProvider", typeof(ILabelProvider), typeof(RulerControl),
            new FrameworkPropertyMetadata());

        static RulerControl()
        {
            DefaultStyleKeyProperty.OverrideMetadata(typeof(RulerControl), new FrameworkPropertyMetadata(typeof(RulerControl)));
        }

        public RulerControl()
        {
            this.MinorLabelProvider = new MinorTickLabelProvider();
            this.MajorLabelProvider = new MajorTickLabelProvider();
        }

        public void UpdateTick()
        {
            var value = TickAndDivideCalculator.GetTick(this.MinimumSpanMajorTick, this.Scale);
            var majorTick = value.Item1;
            var minorTick = value.Item2;

            if (majorTick != this.MajorTick)
            {
                this.MajorTick = majorTick;
            }

            if (minorTick != this.MinorTick)
            {
                this.MinorTick = minorTick;
            }
        }

        public override void OnApplyTemplate()
        {
            base.OnApplyTemplate();

            this.pointerLine = GetTemplateChild(pointerPart) as Line;
        }

        protected override void OnMouseLeftButtonDown(MouseButtonEventArgs e)
        {
            base.OnMouseLeftButtonDown(e);

            this.beginPoint = e.GetPosition(this);
            this.tracking = true;
            Mouse.Capture(this);
        }

        protected override void OnMouseLeftButtonUp(MouseButtonEventArgs e)
        {
            base.OnMouseLeftButtonUp(e);

            this.ReleaseMouseCapture();
            this.tracking = false;
        }

        protected override void OnMouseMove(MouseEventArgs e)
        {
            base.OnMouseMove(e);

            var position = e.GetPosition(this);

            if (this.tracking == true)
            {
                this.Position += this.beginPoint.X - position.X;
                this.beginPoint = position;
            }

            this.PointerPosition = new Point(position.X + this.Position, position.Y);
            this.PointerX = position.X;
        }

        protected override void OnMouseLeave(MouseEventArgs e)
        {
            base.OnMouseLeave(e);

            this.PointerPosition = new Point(double.NaN, double.NaN);
        }

        protected override void OnRender(DrawingContext drawingContext)
        {
            base.OnRender(drawingContext);

            drawingContext.DrawRectangle(this.Background, null, new Rect(0, 0, ActualWidth, ActualHeight));

            this.DrawTick(drawingContext, this.MinorTick, 0.2, this.MinorLabelProvider);
            this.DrawTick(drawingContext, this.MajorTick, 0.4, this.MajorLabelProvider);
        }

        private double Rounding(double target, double align)
        {
            return target - (target % align);
        }

        private void DrawTick(DrawingContext drawingContext, double tick, double lengthRatio, ILabelProvider labelProvider)
        {
            var labelTick = this.Rounding(this.Position / this.Scale, tick);
            var span = tick * this.Scale;
            var x = this.Rounding(this.Position, span) - this.Position;
            var xcount = (int)(this.ActualWidth / span) + 2;
            var y = this.ActualHeight * (1.0 - lengthRatio);

            var brush = new SolidColorBrush(Color.FromRgb(0x80, 0x80, 0x80));
            brush.Freeze();
            var pen = new Pen(brush, 1);
            pen.Freeze();

            drawingContext.PushClip(new RectangleGeometry(new Rect(new Point(0.0, 0.0), new Point(this.ActualWidth, this.ActualHeight))));

            for (int count = 0; count < xcount; count++)
            {
                if (x >= 0)
                {
                    var guideline = new GuidelineSet();
                    guideline.GuidelinesX.Add(x - 0.5);
                    guideline.GuidelinesX.Add(x + 0.5);

                    drawingContext.PushGuidelineSet(guideline);
                    drawingContext.DrawGeometry(null, pen, new LineGeometry(new Point(x, y), new Point(x, this.ActualHeight)));
                    drawingContext.Pop();

                    var label = labelProvider.GetLabel(labelTick);
                    if (label != string.Empty)
                    {
                        var typeface = new Typeface(this.FontFamily, this.FontStyle, this.FontWeight, this.FontStretch);
                        var text = new FormattedText(label, CultureInfo.CurrentCulture, FlowDirection.LeftToRight, typeface, this.FontSize, this.Foreground, null, TextFormattingMode.Display);
                        text.MaxTextWidth = span;

                        var tx = x - text.Width / 2;
                        if (tx >= 0)
                        {
                            drawingContext.DrawText(text, new Point(tx, 0));
                        }
                    }
                }

                x += span;
                labelTick += tick;
            }

            drawingContext.Pop();
        }
    }

    public interface ILabelProvider
    {
        string GetLabel(double value);
    }

    public class MinorTickLabelProvider : ILabelProvider
    {
        string ILabelProvider.GetLabel(double value)
        {
            return string.Empty;
        }
    }

    public class MajorTickLabelProvider : ILabelProvider
    {
        string ILabelProvider.GetLabel(double value)
        {
            var m = (int)value / 1000 / 60;
            var s = ((int)value / 1000 % 60) + (value % 1000) / 1000.0;
            return string.Format("{0}:{1:00.0####}", m, s);
        }
    }

    public static class TickAndDivideCalculator
    {
        public static Tuple<double, double> GetTick(double minimumSpanMajorTick, double scale)
        {
            var value = TickAndDivideCalculator.GetTickAndDivideCount(minimumSpanMajorTick / scale);
            var majorTick = value.Item1;
            var minorTick = majorTick / value.Item2;
            return new Tuple<double, double>(majorTick, minorTick);
        }

        public static Tuple<double, int> GetTickAndDivideCount(double value)
        {
            var baseValue = Math.Pow(10.0, Math.Floor(Math.Log10(value)));
            var relativeValue = value / baseValue;

            if (relativeValue < 2.0)
            {
                return new Tuple<double, int>(baseValue * 1.0, 2);
            }
            else if (relativeValue < 5.0)
            {
                return new Tuple<double, int>(baseValue * 2.0, 2);
            }
            else
            {
                return new Tuple<double, int>(baseValue * 5.0, 5);
            }
        }
    }
}
