﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Windows;
using System.Windows.Input;

namespace Nintendo.AudioToolkit.Views
{
    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.AudioToolkit.Extensions;
    using Nintendo.AudioToolkit.Resources;
    using Nintendo.Foundation.ComponentModel;

    /// <summary>
    ///
    /// </summary>
    public class WaveSoundPropertyViewModel : INotifyPropertyChanged, INotifyDataErrorInfo, IDisposable
    {
        private readonly Dictionary<string, List<string>> errorsDictionary = new Dictionary<string, List<string>>();

        private readonly WeakReference<WaveSoundClipResource> model = new WeakReference<WaveSoundClipResource>(null);
        public event PropertyChangedEventHandler PropertyChanged;
        private PropertyChangedObserver<WaveSoundClipResource> propertyChangedObserverWaveSoundClipResource = null;
        private PropertyChangedObserver<WaveSoundPropertyViewModel> propertyChangedObserverWaveSoundPropertyViewModel = null;

        private int position;
        private int duration;
        private int startOffset;
        private string filePath;
        private double volume = 96.0F;
        private double pitch = 1.0F;
        private double pan = 64.0F;

        public ICommand ChangeFilePathCommand { get; set; }

        public WaveSoundPropertyViewModel(WaveSoundClipResource model)
        {
            this.model.SetTarget(model);

            this.Position = model.Position;
            this.Duration = model.Duration;
            this.StartOffset = model.StartOffset;
            this.FilePath = model.Path;
            this.Volume = model.Volume;
            this.Pitch = model.Pitch;
            this.Pan = model.Pan;

            this.propertyChangedObserverWaveSoundClipResource = new PropertyChangedObserver<WaveSoundClipResource>(model, true);
            this.propertyChangedObserverWaveSoundClipResource.AddHandler((p => p.Position), (t, e) => this.Position = t.Position);
            this.propertyChangedObserverWaveSoundClipResource.AddHandler((p => p.Duration), (t, e) => this.Duration = t.Duration);
            this.propertyChangedObserverWaveSoundClipResource.AddHandler((p => p.StartOffset), (t, e) => this.StartOffset = t.StartOffset);
            this.propertyChangedObserverWaveSoundClipResource.AddHandler((p => p.Path), (t, e) => this.FilePath = t.Path);
            this.propertyChangedObserverWaveSoundClipResource.AddHandler((p => p.Volume), (t, e) => this.Volume = t.Volume);
            this.propertyChangedObserverWaveSoundClipResource.AddHandler((p => p.Pitch), (t, e) => this.Pitch = t.Pitch);
            this.propertyChangedObserverWaveSoundClipResource.AddHandler((p => p.Pan), (t, e) => this.Pan = t.Pan);

            this.propertyChangedObserverWaveSoundPropertyViewModel = new PropertyChangedObserver<WaveSoundPropertyViewModel>(this, true);
            this.propertyChangedObserverWaveSoundPropertyViewModel.AddHandler((p => p.Position), (t, e) => this.Model.Position = (int)t.Position);
            this.propertyChangedObserverWaveSoundPropertyViewModel.AddHandler((p => p.Duration), (t, e) => this.Model.Duration = (int)t.Duration);
            this.propertyChangedObserverWaveSoundPropertyViewModel.AddHandler((p => p.StartOffset), (t, e) => this.Model.SetStartOffsetWithPositionAndDuration((int)t.StartOffset));
            this.propertyChangedObserverWaveSoundPropertyViewModel.AddHandler((p => p.FilePath), (t, e) => this.Model.Path = (string)t.FilePath);
            this.propertyChangedObserverWaveSoundPropertyViewModel.AddHandler((p => p.Volume), (t, e) => this.Model.Volume = (float)t.Volume);
            this.propertyChangedObserverWaveSoundPropertyViewModel.AddHandler((p => p.Pitch), (t, e) => this.Model.Pitch = (float)t.Pitch);
            this.propertyChangedObserverWaveSoundPropertyViewModel.AddHandler((p => p.Pan), (t, e) => this.Model.Pan = (float)t.Pan);
        }

        public void Dispose()
        {
            if (this.propertyChangedObserverWaveSoundClipResource != null)
            {
                this.propertyChangedObserverWaveSoundClipResource.RemoveAllHandlers();
                this.propertyChangedObserverWaveSoundClipResource = null;
            }
            if (this.propertyChangedObserverWaveSoundPropertyViewModel != null)
            {
                this.propertyChangedObserverWaveSoundPropertyViewModel.RemoveAllHandlers();
                this.propertyChangedObserverWaveSoundPropertyViewModel = null;
            }
        }

        public int Position
        {
            get { return this.position; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.position, value) == true)
                {
                    this.ValidateClipOverlapped(value, this.Model.Duration);
                    this.ValidatePropertyPlusValue(this.position);
                }
            }
        }

        public int Duration
        {
            get { return this.duration; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.duration, value) == true)
                {
                    this.ValidateClipOverlapped(this.Model.Position, value);
                    this.ValidateProperty(((v) => 1 <= v), this.duration, MessageResource.ErrorMessage_DurationValue);
                }
            }
        }

        public int StartOffset
        {
            get { return this.startOffset; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.startOffset, value) == true)
                {
                    this.ValidatePropertyPlusValue(this.startOffset);
                }
            }
        }

        public string FilePath
        {
            get { return this.filePath; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.filePath, value) == true)
                {
                    this.ValidateProperty(((f) => File.Exists(f as string)), this.filePath, MessageResource.ErrorMessage_NotExistsFile);
                }
            }
        }

        public double Volume
        {
            get { return this.volume; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.volume, value) == true)
                {
                    this.ValidatePropertyRange(this.volume, 0, 255);
                }
            }
        }

        public double Pitch
        {
            get { return this.pitch; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.pitch, value) == true)
                {
                    this.ValidatePropertyRange(this.pitch, 0.00001F, 8.0F);
                }
            }
        }

        public double Pan
        {
            get { return this.pan; }
            set
            {
                if (this.SetProperty(this.PropertyChanged, ref this.pan, value) == true)
                {
                    this.ValidatePropertyRange(this.pan, 0, 127);
                }
            }
        }

        private WaveSoundClipResource Model
        {
            get
            {
                WaveSoundClipResource model = null;
                this.model.TryGetTarget(out model);
                return model;
            }
        }

        private void ValidateClipOverlapped(double newPosition, double newDuration, [CallerMemberName] string propertyName = null)
        {
            var message = MessageResource.ErrorMessage_OverlappedClip;
            var trackDM = this.Model.Owner;
            var clipDM = this.Model;
            var bounds = new Rect(newPosition, clipDM.Row, newDuration - 1.0, 0.0);
            var isHit = trackDM.Clips
                .Where(c => c != clipDM)
                .Select(c => new Rect(c.Position, c.Row, c.Duration - 1.0, 0.0))
                .Any(r => r.IntersectsWith(bounds));

            if (isHit == true)
            {
                this.AddError(propertyName, message);
            }
            else
            {
                this.RemoveError(propertyName, message);
            }
        }

        private void ValidatePropertyPlusValue<T>(T value, [CallerMemberName] string propertyName = null) where T : IComparable
        {
            this.ValidateProperty(((v) => 0 <= v.CompareTo(0)), value, MessageResource.ErrorMessage_PlusValue, propertyName);
        }

        private void ValidatePropertyRange<T>(T value, T minimum, T maximum, [CallerMemberName] string propertyName = null) where T : IComparable
        {
            this.ValidateProperty(((v) => (0 <= v.CompareTo(minimum) && v.CompareTo(maximum) <= 0)), value, string.Format(MessageResource.ErrorMessage_OutOfRange, minimum, maximum), propertyName);
        }

        private void ValidateProperty<T>(Func<T, bool> predicate, T value, string errorMessage, [CallerMemberName] string propertyName = null)
        {
            if (predicate(value) == true)
            {
                this.RemoveError(propertyName, errorMessage);
            }
            else
            {
                this.AddError(propertyName, errorMessage);
            }
        }

        private void AddError(string propertyName, string errorMessage)
        {
            if (this.errorsDictionary.ContainsKey(propertyName) == false)
            {
                this.errorsDictionary[propertyName] = new List<string>();
            }
            this.errorsDictionary[propertyName].Add(errorMessage);

            if (this.ErrorsChanged != null)
            {
                this.ErrorsChanged(this, new DataErrorsChangedEventArgs(propertyName));
            }
        }

        private void RemoveError(string propertyName, string message)
        {
            if (this.errorsDictionary.ContainsKey(propertyName) == true)
            {
                var list = this.errorsDictionary[propertyName];
                if (list.Contains(message) == true)
                {
                    list.Remove(message);
                    if (list.Count <= 0)
                    {
                        this.errorsDictionary.Remove(propertyName);
                    }
                }
            }

            if (this.ErrorsChanged != null)
            {
                this.ErrorsChanged(this, new DataErrorsChangedEventArgs(propertyName));
            }
        }


        /// <summary>
        /// インターフェース INotifyDataErrorInfo の実装
        /// </summary>

        public event EventHandler<DataErrorsChangedEventArgs> ErrorsChanged;

        public IEnumerable GetErrors(string propertyName)
        {
            if (this.errorsDictionary.ContainsKey(propertyName) == false)
            {
                return null;
            }

            return this.errorsDictionary[propertyName];
        }

        public bool HasErrors
        {
            get { return 0 < this.errorsDictionary.Count; }
        }
    }
}
