﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../precompiled.h"

#include <panels/cafe/SoundSystemCafePanel.h>

namespace nw {
namespace snd {

static const AXDRCVSOutputMode s_DRCVSOutputModes[] =
{
    AX_DRC_VS_OUTPUT_SPEAKER,
    AX_DRC_VS_OUTPUT_HEADPHONE,
};

static const AXDRCVSSpeakerPosition s_DRCVSSpeakerPositions[] =
{
    AX_DRC_VS_SPEAKER_POS_WIDE,
    AX_DRC_VS_SPEAKER_POS_SQUARE,
};

static const AXDRCVSSurroundLevelGain s_DRCVSSurroundLevelGains[] =
{
    AX_DRC_VS_SLEV_1_0,
    AX_DRC_VS_SLEV_0_707,
    AX_DRC_VS_SLEV_0_5,
};

static AXDRCVSOutputMode GetDRCVSOutputMode(SoundSystemCafePanel::DRCVSParams* pParams)
{
    NW_ASSERT_NOT_NULL(pParams);
    return pParams->outputMode;
}

static void SetDRCVSOutputMode(SoundSystemCafePanel::DRCVSParams* pParams, AXDRCVSOutputMode value)
{
    NW_ASSERT_NOT_NULL(pParams);
    pParams->outputMode = value;
}

static void PrintDRCVSOutputMode(char* pText, u32 bufferLength, AXDRCVSOutputMode value)
{
    const char* pValueText = "";

    switch(value)
    {
    case AX_DRC_VS_OUTPUT_SPEAKER:
        pValueText = "SPEAKER";
        break;

    case AX_DRC_VS_OUTPUT_HEADPHONE:
        pValueText = "HEADPHONE";
        break;
    }

    nw::ut::snprintf(pText, bufferLength, bufferLength - 1, "%s", pValueText);
}

static AXDRCVSSpeakerPosition GetDRCVSSpeakerPosition(SoundSystemCafePanel::DRCVSParams* pParams)
{
    NW_ASSERT_NOT_NULL(pParams);
    return pParams->speakerPosition;
}

static void SetDRCVSSpeakerPosition(SoundSystemCafePanel::DRCVSParams* pParams, AXDRCVSSpeakerPosition value)
{
    NW_ASSERT_NOT_NULL(pParams);
    pParams->speakerPosition = value;
#if defined(NW_PLATFORM_CAFE)
    AXSetDRCVSSpeakerPosition(pParams->outputMode, pParams->speakerPosition);
#endif
}

static void PrintDRCVSSpeakerPosition(char* pText, u32 bufferLength, AXDRCVSSpeakerPosition value)
{
    const char* pValueText = "";

    switch(value)
    {
    case AX_DRC_VS_SPEAKER_POS_WIDE:
        pValueText = "WIDE";
        break;

    case AX_DRC_VS_SPEAKER_POS_SQUARE:
        pValueText = "SQUARE";
        break;
    }

    nw::ut::snprintf(pText, bufferLength, bufferLength - 1, "%s", pValueText);
}

static AXDRCVSSurroundLevelGain GetDRCVSSurroundLevelGain(SoundSystemCafePanel::DRCVSParams* pParams)
{
    NW_ASSERT_NOT_NULL(pParams);
    return pParams->surroundLevelGain;
}

static void SetDRCVSSurroundLevelGain(SoundSystemCafePanel::DRCVSParams* pParams, AXDRCVSSurroundLevelGain value)
{
    NW_ASSERT_NOT_NULL(pParams);
    pParams->surroundLevelGain = value;
#if defined(NW_PLATFORM_CAFE)
    AXSetDRCVSSurroundLevelGain(pParams->surroundLevelGain);
#endif
}

static void PrintDRCVSSurroundLevelGain(char* pText, u32 bufferLength, AXDRCVSSurroundLevelGain value)
{
    const char* pValueText = "";

    switch(value)
    {
    case AX_DRC_VS_SLEV_1_0:
        pValueText = "1.0";
        break;

    case AX_DRC_VS_SLEV_0_707:
        pValueText = "0.707";
        break;

    case AX_DRC_VS_SLEV_0_5:
        pValueText = "0.5";
        break;
    }

    nw::ut::snprintf(pText, bufferLength, bufferLength - 1, "%s", pValueText);
}

static f32 GetSurroundDepth(SoundSystemCafePanel::DRCVSParams* pParams)
{
    NW_ASSERT(pParams != NULL);
    return pParams->surroundDepth;
}

static void SetSurroundDepth(SoundSystemCafePanel::DRCVSParams* pParams, f32 value)
{
    NW_ASSERT(pParams != NULL);
    pParams->surroundDepth = value;
#if defined(NW_PLATFORM_CAFE)
    AXSetDRCVSSurroundDepth(pParams->outputMode, pParams->surroundDepth);
#endif
}

static f32 GetDownmixBalance(SoundSystemCafePanel::DRCVSParams* pParams)
{
    NW_ASSERT(pParams != NULL);
    return pParams->downmixBalance;
}

static void SetDownmixBalance(SoundSystemCafePanel::DRCVSParams* pParams, f32 value)
{
    NW_ASSERT(pParams != NULL);
    pParams->downmixBalance = value;
#if defined(NW_PLATFORM_CAFE)
    AXSetDRCVSDownmixBalance(pParams->outputMode, pParams->downmixBalance);
#endif
}

static f32 GetOutputGain(SoundSystemCafePanel::DRCVSParams* pParams)
{
    NW_ASSERT(pParams != NULL);
    return pParams->outputGain;
}

static void SetOutputGain(SoundSystemCafePanel::DRCVSParams* pParams, f32 value)
{
    NW_ASSERT(pParams != NULL);
    pParams->outputGain = value;
#if defined(NW_PLATFORM_CAFE)
    AXSetDRCVSOutputGain(pParams->outputMode, pParams->outputGain);
#endif
}

static void PrintF32Value(char* pText, u32 bufferLength, f32 value)
{
    nw::ut::snprintf(pText, bufferLength, bufferLength - 1, "%4.2f", value);
}

SoundSystemCafePanel::SoundSystemCafePanel() :
m_DRCVSOutputModeControl(AX_DRC_VS_OUTPUT_SPEAKER),
m_DRCVSSpeakerPositionControl(AX_DRC_VS_SPEAKER_POS_WIDE),
m_DRCVSSurroundLevelGainControl(AX_DRC_VS_SLEV_1_0)
{
    m_DRCVSOutputModeControl.SetLabelWidth(80.0f);
    m_DRCVSOutputModeControl.Initialize(
        s_DRCVSOutputModes,
        sizeof(s_DRCVSOutputModes) / sizeof(s_DRCVSOutputModes[0]),
        GetDRCVSOutputMode,
        SetDRCVSOutputMode,
        PrintDRCVSOutputMode
        );
    m_DRCVSOutputModeControl.SetLabelText("DRCVSOutputMode:");
    m_DRCVSOutputModeControl.SetModel(&m_DRCVSParams);

    m_DRCVSSpeakerPositionControl.SetLabelWidth(80.0f);
    m_DRCVSSpeakerPositionControl.Initialize(
        s_DRCVSSpeakerPositions,
        sizeof(s_DRCVSSpeakerPositions) / sizeof(s_DRCVSSpeakerPositions[0]),
        GetDRCVSSpeakerPosition,
        SetDRCVSSpeakerPosition,
        PrintDRCVSSpeakerPosition
        );
    m_DRCVSSpeakerPositionControl.SetLabelText("DRCVSSpeakerPosition:");
    m_DRCVSSpeakerPositionControl.SetModel(&m_DRCVSParams);

    m_DRCVSSurroundDepthControl.Initialize(GetSurroundDepth, SetSurroundDepth, PrintF32Value);
    m_DRCVSSurroundDepthControl.SetLabelText("DRCVSSurroundDepth:");
    m_DRCVSSurroundDepthControl.SetMinimumValue(0.0f);
    m_DRCVSSurroundDepthControl.SetMaximumValue(1.0f);
    m_DRCVSSurroundDepthControl.SetDefaultValue(1.0f);
    m_DRCVSSurroundDepthControl.SetValueChange(0.01f);
    m_DRCVSSurroundDepthControl.SetValueLargeChange(0.1f);
    m_DRCVSSurroundDepthControl.SetModel(&m_DRCVSParams);

    m_DRCVSDownmixBalanceControl.Initialize(GetDownmixBalance, SetDownmixBalance, PrintF32Value);
    m_DRCVSDownmixBalanceControl.SetLabelText("DRCVSDownmixBalance:");
    m_DRCVSDownmixBalanceControl.SetMinimumValue(0.0f);
    m_DRCVSDownmixBalanceControl.SetMaximumValue(1.0f);
    m_DRCVSDownmixBalanceControl.SetDefaultValue(1.0f);
    m_DRCVSDownmixBalanceControl.SetValueChange(0.01f);
    m_DRCVSDownmixBalanceControl.SetValueLargeChange(0.1f);
    m_DRCVSDownmixBalanceControl.SetModel(&m_DRCVSParams);

    m_DRCVSSurroundLevelGainControl.SetLabelWidth(80.f);
    m_DRCVSSurroundLevelGainControl.Initialize(
        s_DRCVSSurroundLevelGains,
        sizeof(s_DRCVSSurroundLevelGains) / sizeof(s_DRCVSSurroundLevelGains[0]),
        GetDRCVSSurroundLevelGain,
        SetDRCVSSurroundLevelGain,
        PrintDRCVSSurroundLevelGain
        );
    m_DRCVSSurroundLevelGainControl.SetLabelText("DRCVSSurroundLevelGain:");
    m_DRCVSSurroundLevelGainControl.SetModel(&m_DRCVSParams);

    m_DRCVSOutputGainControl.Initialize(GetOutputGain, SetOutputGain, PrintF32Value);
    m_DRCVSOutputGainControl.SetLabelText("DRCVSOutputGain:");
    m_DRCVSOutputGainControl.SetMinimumValue(0.0f);
    m_DRCVSOutputGainControl.SetMaximumValue(1.0f);
    m_DRCVSOutputGainControl.SetDefaultValue(1.0f);
    m_DRCVSOutputGainControl.SetValueChange(0.01f);
    m_DRCVSOutputGainControl.SetValueLargeChange(0.1f);
    m_DRCVSOutputGainControl.SetModel(&m_DRCVSParams);

    AddItem(m_DRCVSOutputModeControl);
    AddItem(m_DRCVSSpeakerPositionControl);
    AddItem(m_DRCVSSurroundDepthControl);
    AddItem(m_DRCVSDownmixBalanceControl);
    AddItem(m_DRCVSSurroundLevelGainControl);
    AddItem(m_DRCVSOutputGainControl);
}

SoundSystemCafePanel::~SoundSystemCafePanel()
{
}

} // snd
} // nw
